package ohd.hseb.hefs.utils.effect;

import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;

import ohd.hseb.hefs.utils.xml.ListXMLReader;

/**
 * Performs a specified action for the given arguments. For examples, refer to {@link ListXMLReader}, which uses effects
 * as something to call during the finalize portion of the XML reading.
 * 
 * @author alexander.garbarino
 * @param <A> The {@link Object} upon which the effect takes place. This specifies what is passed into the
 *            {@link #perform(Object)} method.
 */
public abstract class Effect<A>
{
    /**
     * Perform this effect for the specified input.
     * 
     * @param input
     */
    public abstract void perform(A input);

    /**
     * Creates a new effect which first performs {@code this} and then {@code afterEffect}.
     * 
     * @param afterEffect the effect to perform after this one
     */
    @SuppressWarnings("unchecked")
    public Effect<A> then(final Effect<? super A> afterEffect)
    {
        return new CompositeEffect<A>(this, afterEffect);
    }

    /**
     * Creates a new effect which first performs {@code this} and then runs {@code afterRunnable}.
     * 
     * @param afterRunnable the runnable to run after this effect is performed
     */
    @SuppressWarnings("unchecked")
    public Effect<A> then(final Runnable afterRunnable)
    {
        return new CompositeEffect<A>(this, new RunnableEffect(afterRunnable));
    }

    /**
     * Creates a runnable that calls {@link #perform(Object) perform(input)}.
     */
    public Runnable asRunnable(final A input)
    {
        return new Runnable()
        {
            @Override
            public void run()
            {
                perform(input);
            }
        };
    }

    /**
     * Creates an action that calls {@link #perform(Object) perform(input)}.
     */
    @SuppressWarnings("serial")
    public Action asAction(final A input)
    {
        return new AbstractAction()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                perform(input);
            }
        };
    }
}
