package ohd.hseb.hefs.utils.filechooser;

/*
 * @(#)ExampleFileFilter.java 1.8 98/08/26 Copyright 1998 by Sun Microsystems, Inc., 901 San Antonio Road, Palo Alto,
 * California, 94303, U.S.A. All rights reserved. This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information"). You shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement you entered into with Sun.
 */

import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.swing.filechooser.FileFilter;

/**
 * A convenience implementation of FileFilter that filters out all files except for those type extensions that it knows
 * about. Extensions are of the type ".foo", which is typically found on Windows and Unix boxes, but not on Macinthosh.
 * Case is ignored. Example - create a new filter that filerts out all files but gif and jpg image files: JFileChooser
 * chooser = new JFileChooser(); ExampleFileFilter filter = new ExampleFileFilter( new String{"gif", "jpg"},
 * "JPEG & GIF Images") chooser.addChoosableFileFilter(filter); chooser.showOpenDialog(this);
 * 
 * @version 1.8 08/26/98
 * @author Jeff Dinkins
 */
public class FileExtensionFileFilter extends FileFilter
{

    private Hashtable filters = null;
    private String description = null;
    private String fullDescription = null;
    private boolean useExtensionsInDescription = true;

    /**
     * Creates a file filter. If no filters are added, then all files are accepted.
     * 
     * @see #addExtension
     */
    public FileExtensionFileFilter()
    {
        this.filters = new Hashtable();
    }

    /**
     * Creates a file filter that accepts files with the given extension. Example: new ExampleFileFilter("jpg");
     * 
     * @see #addExtension
     */
    public FileExtensionFileFilter(final String extension)
    {
        this(extension, null);
    }

    /**
     * Creates a file filter that accepts the given file type. Example: new ExampleFileFilter("jpg",
     * "JPEG Image Images"); Note that the "." before the extension is not needed. If provided, it will be ignored.
     * 
     * @see #addExtension
     */
    public FileExtensionFileFilter(final String extension, final String description)
    {
        this();
        if(extension != null)
            addExtension(extension);
        if(description != null)
            setDescription(description);
    }

    /**
     * Creates a file filter from the given string array. Example: new ExampleFileFilter(String {"gif", "jpg"}); Note
     * that the "." before the extension is not needed adn will be ignored.
     * 
     * @see #addExtension
     */
    public FileExtensionFileFilter(final String[] filters)
    {
        this(filters, null);
    }

    /**
     * Creates a file filter from the given string array and description. Example: new ExampleFileFilter(String {"gif",
     * "jpg"}, "Gif and JPG Images"); Note that the "." before the extension is not needed and will be ignored.
     * 
     * @see #addExtension
     */
    public FileExtensionFileFilter(final String[] filters, final String description)
    {
        this();
        for(int i = 0; i < filters.length; i++)
        {
            // add filters one by one
            addExtension(filters[i]);
        }
        if(description != null)
            setDescription(description);
    }

    /**
     * Return true if this file should be shown in the directory pane, false if it shouldn't. Files that begin with "."
     * are ignored.
     * 
     * @see #getExtension
     * @see FileFilter#accepts
     */
    @Override
    public boolean accept(final File f)
    {
        if(f != null)
        {
            if(f.isDirectory())
            {
                return true;
            }
            final String extension = getExtension(f);
            if(extension != null && filters.get(getExtension(f)) != null)
            {
                return true;
            }
            ;
        }
        return false;
    }

    /**
     * Return the extension portion of the file's name .
     * 
     * @see #getExtension
     * @see FileFilter#accept
     */
    public String getExtension(final File f)
    {
        if(f != null)
        {
            final String filename = f.getName();
            final int i = filename.lastIndexOf('.');
            if(i > 0 && i < filename.length() - 1)
            {
                return filename.substring(i + 1).toLowerCase();
            }
            ;
        }
        return null;
    }

    /**
     * Adds a filetype "dot" extension to filter against. For example: the following code will create a filter that
     * filters out all files except those that end in ".jpg" and ".tif": ExampleFileFilter filter = new
     * ExampleFileFilter(); filter.addExtension("jpg"); filter.addExtension("tif"); Note that the "." before the
     * extension is not needed and will be ignored.
     */
    @SuppressWarnings("unchecked")
    public void addExtension(final String extension)
    {
        if(filters == null)
        {
            filters = new Hashtable(5);
        }
        filters.put(extension.toLowerCase(), this);
        fullDescription = null;
    }

    /**
     * Returns the human readable description of this filter. For example: "JPEG and GIF Image Files (*.jpg, *.gif)"
     * 
     * @see setDescription
     * @see setExtensionListInDescription
     * @see isExtensionListInDescription
     * @see FileFilter#getDescription
     */
    @Override
    public String getDescription()
    {
        if(fullDescription == null)
        {
            if(description == null || isExtensionListInDescription())
            {
                fullDescription = description == null ? "(" : description + " (";
                // build the description from the extension list
                final Enumeration extensions = filters.keys();
                if(extensions != null)
                {
                    fullDescription += "." + (String)extensions.nextElement();
                    while(extensions.hasMoreElements())
                    {
                        fullDescription += ", " + (String)extensions.nextElement();
                    }
                }
                fullDescription += ")";
            }
            else
            {
                fullDescription = description;
            }
        }
        return fullDescription;
    }

    /**
     * Sets the human readable description of this filter. For example: filter.setDescription("Gif and JPG Images");
     * 
     * @see setDescription
     * @see setExtensionListInDescription
     * @see isExtensionListInDescription
     */
    public void setDescription(final String description)
    {
        this.description = description;
        fullDescription = null;
    }

    /**
     * Determines whether the extension list (.jpg, .gif, etc) should show up in the human readable description. Only
     * relevent if a description was provided in the constructor or using setDescription();
     * 
     * @see getDescription
     * @see setDescription
     * @see isExtensionListInDescription
     */
    public void setExtensionListInDescription(final boolean b)
    {
        useExtensionsInDescription = b;
        fullDescription = null;
    }

    /**
     * Returns whether the extension list (.jpg, .gif, etc) should show up in the human readable description. Only
     * relevent if a description was provided in the constructor or using setDescription();
     * 
     * @see getDescription
     * @see setDescription
     * @see setExtensionListInDescription
     */
    public boolean isExtensionListInDescription()
    {
        return useExtensionsInDescription;
    }

    /**
     * Return the filters hash table so that the extensions can be examined.
     * 
     * @return Hashtable.
     */
    public Hashtable getFilters()
    {
        return filters;
    }
}
