package ohd.hseb.hefs.utils.filechooser;

import java.awt.Component;
import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

/**
 * Create a JFileChooser for opening or saving files. It also checks for file existence for a save dialog.
 * 
 * @author chuir, hank
 */
public class GenericFileChooser extends JFileChooser
{
    private static final long serialVersionUID = 1L;

    /**
     * Construct file chooser within current directory
     * 
     * @param approveButtonText approve button text
     * @param approveButtonMnemonic button mnemonic
     */
    public GenericFileChooser(final String approveButtonText, final char approveButtonMnemonic)
    {
        super(".");

        setDialogType(JFileChooser.CUSTOM_DIALOG);
        setDialogTitle("Choose a file");
        setApproveButtonText(approveButtonText);
        setApproveButtonMnemonic(approveButtonMnemonic);
        setMultiSelectionEnabled(false);
        setApproveButtonToolTipText("Save to selected file, alt+S");

    }

    /**
     * Con_struct file chooser within current directory
     * 
     * @param approveButtonText approve button text
     * @param approveButtonMnemonic button mnemonic
     * @param baseDir The starting directtory.
     */
    public GenericFileChooser(final String approveButtonText, final char approveButtonMnemonic, final String baseDir)
    {
        super(baseDir);

        setDialogType(JFileChooser.CUSTOM_DIALOG);
        setDialogTitle("Choose a file");
        setMultiSelectionEnabled(false);
        setApproveButtonText(approveButtonText);
        setApproveButtonMnemonic(approveButtonMnemonic);
        setApproveButtonToolTipText("Save to selected file, alt+S");

    }

    /**
     * Calls {@link #showSaveDialog(Component)} for the given parent. It will then confirm the save if the file already
     * exists.
     * 
     * @return File to be save
     */
    public File showSaveDialogWithParent(final Component parent)
    {
        File selectedFile = null;

        switch(showSaveDialog(parent))
        {
            case JFileChooser.APPROVE_OPTION:
                selectedFile = getSelectedFile(true);

                if(selectedFile.exists())
                {
                    final int result = JOptionPane.showConfirmDialog(this,
                                                                     "File " + selectedFile.getAbsolutePath()
                                                                         + " exists. Do you want to overwrite it?",
                                                                     "Confirm Selection",
                                                                     JOptionPane.YES_NO_OPTION,
                                                                     JOptionPane.QUESTION_MESSAGE,
                                                                     null);
                    if(result == JOptionPane.NO_OPTION)
                    {
                        selectedFile = null;
                    }
                }
                break;
            case JFileChooser.CANCEL_OPTION:
                break;
        }
        return selectedFile;
    }

    /**
     * Calls {@link #showOpenDialog(Component)} for the settings within and the given parent.
     * 
     * @return The file chosen, or null if cancel was clicked.
     */
    public File showOpenDialogWithParent(final Component parent)
    {
        File selectedFile = null;

        switch(showOpenDialog(parent))
        {
            case JFileChooser.APPROVE_OPTION:
                selectedFile = getSelectedFile();
                break;
            case JFileChooser.CANCEL_OPTION:
                break;
        }
        return selectedFile;
    }

    /**
     * Calls {@link #showDialog(Component, String)} for the given parent and value of {@link #getApproveButtonText()}.
     * 
     * @return The file chosen, or null if cancel was clicked.
     */
    public File showDialogWithParent(final Component parent, final boolean fileOverwriteCheck)
    {
        File selectedFile = null;

        switch(showDialog(parent, getApproveButtonText()))
        {
            case JFileChooser.APPROVE_OPTION:
                selectedFile = getSelectedFile();
                if((fileOverwriteCheck) && (selectedFile.exists()))
                {
                    final int result = JOptionPane.showConfirmDialog(this,
                                                                     "File " + selectedFile.getAbsolutePath()
                                                                         + " exists. Do you want to overwrite it?",
                                                                     "Confirm Selection",
                                                                     JOptionPane.YES_NO_OPTION,
                                                                     JOptionPane.QUESTION_MESSAGE,
                                                                     null);
                    if(result == JOptionPane.NO_OPTION)
                    {
                        selectedFile = null;
                    }
                }
                break;
            case JFileChooser.CANCEL_OPTION:
                break;
        }
        return selectedFile;
    }

    /**
     * Opens up an error message dialog with the given message and title.
     */
    public void errorMessageDialog(final Object message, final String title)
    {
        JOptionPane.showMessageDialog(this, message, title, JOptionPane.ERROR_MESSAGE);
    }

    /**
     * Add a file filter with the specified components.
     * 
     * @param extension The extension string.
     * @param fileType String describing the type of file.
     */
    public void addFileFilter(final String extension, final String fileType)
    {
        addChoosableFileFilter(new FileExtensionFileFilter(extension, fileType));
    }

    /**
     * Pass in a value of true to have the selected file filter extension added if desired. Returns the selected file,
     * as getSelectedFile() does, but with the desired extension specified by the selected file filter added if any one
     * of the extensions for the file filters are not already present.
     * 
     * @param addExtension
     */
    public File getSelectedFile(final boolean addExtension)
    {
        File file = getSelectedFile();

        //If any selected file filter accepts the file as is, just return it.
        int i;
        for(i = 0; i < getChoosableFileFilters().length; i++)
        {
            if(getChoosableFileFilters()[i].accept(file))
            {
                return file;
            }
        }

        //The file does not match the filter at this point.  So, if we are to add
        //the extension, then add it.
        if(addExtension)
        {
            final FileExtensionFileFilter filter = (FileExtensionFileFilter)getFileFilter();
            String newFileName = file.getAbsolutePath();
            newFileName += "." + (String)(filter.getFilters().keys().nextElement());
            file = new File(newFileName);
        }
        return file;
    }

}
