/*
 * Created on Jan 31, 2006 To change the template for this generated file go to Window&gt;Preferences&gt;Java&gt;Code
 * Generation&gt;Code and Comments
 */
package ohd.hseb.hefs.utils.filechooser;

import java.awt.Component;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

import ohd.hseb.util.misc.MiscTools;

/**
 * Class to allow for multiple GUIs to access file choosers with consistent base-paths when the choosing window opens
 * up. If you need to worry about one file chooser remembering where the user was when the last file was selected, then
 * you will want to use the static GLOBAL_CHOOSERS list and call the static initialize methods and getFileChooser
 * method. See ohd.hseb.rfc.ivp.gui.IVPFileChooserController class for an example.<br>
 * <br>
 * If you don't care about then just call the constructor with a base directory and call either chooseFile or saveFile
 * method depending on the task being performed.<br>
 * <br>
 * File filters can be added by calling addFileFilter.<br>
 * <br>
 * Four default identifying strings are provided for the GLOBAL_CHOOSERS: <br>
 * <br>
 * DATA_FILE_CHOOSER: Used for choosing "data" files. For the IVP, these are files that contain data from a chart.<br>
 * TEMPLATE_FILE_CHOOSER: Used for choosing "template" files. For the IVP, these are files that contain chart property
 * modifications.<br>
 * IMAGE_FILE_CHOOSER: Used for choosing "image" files, usually .png or .jpg files.<br>
 * BATCH_FILE_CHOOSER: Used for choosing "batch" files, such as IVP Batch Program batch files.
 * 
 * @author hank
 */
public class HGlobalFileChooser
{
    final static String CLASSNAME = "VerificationGroupPanel";

    private static HashMap<String, HGlobalFileChooser> GLOBAL_CHOOSERS;

    public final static String DATA_FILE_CHOOSER = "DataFileChooser";
    public final static String TEMPLATE_FILE_CHOOSER = "TemplateFileChooser";
    public final static String IMAGE_FILE_CHOOSER = "ImageFileChooser";
    public final static String BATCH_FILE_CHOOSER = "BatchFileChooser";

    public final static String DEFAULT_APPROVE_TEXT = "Select";

    /**
     * The initial base directory for the file chooser.
     */
    private String _baseDir;

    /**
     * The last loaded/saved file, used to determine where the previous loaded/saved file was located.
     */
    private File _previousFile;

    /**
     * Filters to use in the choosers.
     */
    private final List<FileFilter> _filters;

    /**
     * Set the title of the window.
     */
    private String _dialogTitle;

    /**
     * Set the text to display for hte 'approve' button (such as "Open").
     */
    private String _approveButtonText = null;

    /**
     * If true, then when the approve button is clicked, it will check for existence of file and confirm overwrite
     * request.
     */
    private boolean _overwriteCheck = false;

    /**
     * Constrcutor requires a base directory. This is only the initial directory.
     * 
     * @param directory
     */
    public HGlobalFileChooser(final String directory)
    {
        _filters = new ArrayList<FileFilter>();
        _baseDir = directory;
    }

    /**
     * Calls the other version of chooseFile passed in JFileChooser.FILES_ONLY.
     * 
     * @param parent The component parent.
     * @return File selected.
     */
    public File chooseFile(final Component parent)
    {
        return chooseFile(JFileChooser.FILES_ONLY, parent);
    }

    /**
     * Assumes only files can be chosen.
     * 
     * @param initialFile The file to have selected, to start with.
     * @param parent The component parent.
     * @return File (or directory) selected by user via GUI, or null if cancelled.
     */
    public File chooseFile(final File initialFile, final Component parent)
    {
        return chooseFile(JFileChooser.FILES_ONLY, initialFile, parent);
    }

    /**
     * Assumes no initial file selected.
     * 
     * @param fileSelectionMode The fileSelectionMode property for the underlying JFileChooser. Either
     *            JFileChooser.FILES_ONLY, DIRECTORIES_ONLY, FILES_AND_DIRECTORIES.
     * @param parent The component parent.
     * @return File (or directory) selected by user via GUI, or null if cancelled.
     */
    public File chooseFile(final int fileSelectionMode, final Component parent)
    {
        return chooseFile(fileSelectionMode, null, parent);
    }

    /**
     * Choose a file. Return the file choosen, or null if none is chosen.
     * 
     * @param fileSelectionMode The fileSelectionMode property for the underlying JFileChooser. Either
     *            JFileChooser.FILES_ONLY, DIRECTORIES_ONLY, FILES_AND_DIRECTORIES.
     * @param initialFile The file to have selected, to start with.
     * @param parent The component parent.
     * @return File (or directory) selected by user via GUI, or null if cancelled.
     */
    public File chooseFile(final int fileSelectionMode, final File initialFile, final Component parent)
    {
        File theFile;
        GenericFileChooser chooser;
        if(_previousFile != null)
        {
            chooser = new GenericFileChooser("Approve", 's', _previousFile.getParent());
        }
        else
        {
            chooser = new GenericFileChooser("Approve", 's', _baseDir);
        }

        if(initialFile != null)
        {
            chooser.setSelectedFile(initialFile);
        }

        //Set the dialog title if specified.
        if(_dialogTitle != null)
        {
            chooser.setDialogTitle(_dialogTitle);
        }

        //Add the filters.
        int i;
        for(i = 0; i < _filters.size(); i++)
        {
            chooser.addChoosableFileFilter((_filters.get(i)));
        }

        if(_approveButtonText != null)
        {
            chooser.setApproveButtonText(_approveButtonText);
        }
        else
        {
            chooser.setApproveButtonText(DEFAULT_APPROVE_TEXT);
        }

        chooser.setFileSelectionMode(fileSelectionMode);
        theFile = chooser.showDialogWithParent(parent, _overwriteCheck);
        if(theFile == null)
        {
            return null;
        }
        _previousFile = theFile;
        return theFile;
    }

    /**
     * Calls the other version of saveFile passed in JFileChooser.FILES_ONLY.
     * 
     * @param parent The component parent.
     * @return File selected.
     */
    public File saveFile(final Component parent)
    {
        return saveFile(JFileChooser.FILES_ONLY, parent);
    }

    /**
     * Assumes only files can be saved.
     * 
     * @param initialFile The file to have selected, to start with.
     * @param parent The component parent.
     * @return File (or directory) selected by user via GUI, or null if cancelled.
     */
    public File saveFile(final File initialFile, final Component parent)
    {
        return saveFile(JFileChooser.FILES_ONLY, initialFile, parent);
    }

    /**
     * Assumes no initial file selected.
     * 
     * @param fileSelectionMode The fileSelectionMode property for the underlying JFileChooser. Either
     *            JFileChooser.FILES_ONLY, DIRECTORIES_ONLY, FILES_AND_DIRECTORIES.
     * @param parent The component parent.
     * @return File (or directory) selected by user via GUI, or null if cancelled.
     */
    public File saveFile(final int fileSelectionMode, final Component parent)
    {
        return saveFile(fileSelectionMode, null, parent);
    }

    /**
     * Save a file. Return the file to save, or null if none is chosen.
     * 
     * @param fileSelectionMode The fileSelectionMode property for the underlying JFileChooser. Either
     *            JFileChooser.FILES_ONLY, DIRECTORIES_ONLY, FILES_AND_DIRECTORIES.
     * @param initialFile The file to have selected, to start with.
     * @param parent The component parent.
     * @return File (or directory) selected by user via GUI, or null if cancelled.
     */
    public File saveFile(final int fileSelectionMode, final File initialFile, final Component parent)
    {
        File theFile;
        GenericFileChooser chooser;
        if(_previousFile != null)
        {
            chooser = new GenericFileChooser("Approve", 's', _previousFile.getParent());
        }
        else
        {
            chooser = new GenericFileChooser("Approve", 's', _baseDir);
        }

        if(initialFile != null)
        {
            chooser.setSelectedFile(initialFile);
        }

        //Set the dialog title if specified.
        if(_dialogTitle != null)
        {
            chooser.setDialogTitle(_dialogTitle);
        }

        //Add the filters.
        int i;
        for(i = 0; i < _filters.size(); i++)
        {
            chooser.addChoosableFileFilter((_filters.get(i)));
        }
        if(!_filters.isEmpty())
        {
            chooser.setFileFilter(_filters.get(0));
        }

        chooser.setFileSelectionMode(fileSelectionMode);
        theFile = chooser.showSaveDialogWithParent(parent);
        if(theFile == null)
        {
            return null;
        }
        _previousFile = theFile;
        return theFile;
    }

    public String getBaseDir()
    {
        return _baseDir;
    }

    public void setBaseDir(final String dir)
    {
        _baseDir = dir;
    }

    public void setDialogTitle(final String title)
    {
        this._dialogTitle = title;
    }

    public void setApproveButtonText(final String text)
    {
        this._approveButtonText = text;
    }

    public void setOverwriteCheck(final boolean b)
    {
        _overwriteCheck = b;
    }

    /**
     * Calls {@link #setDialogTitle(String)}, {@link #setApproveButtonText(String)}, and
     * {@link #setOverwriteCheck(boolean)} in succession using given parameters.
     * 
     * @param dialogTitle The title of the dialog to display.
     * @param approveButtonText The text displayed on the approve button.
     * @param overwriteCheck Whether or not to check for overwrite when approved.
     */
    public void setOptions(final String dialogTitle, final String approveButtonText, final boolean overwriteCheck)
    {
        setDialogTitle(dialogTitle);
        setApproveButtonText(approveButtonText);
        setOverwriteCheck(overwriteCheck);
    }

    /**
     * Adds a file filter with the passed in extension and file type.
     * 
     * @param extension The extension for the files.
     * @param fileType The type of file the extension corresponds to.
     */
    @SuppressWarnings("unchecked")
    public void addFileFilter(final String extension, final String fileType)
    {
        _filters.add(new FileExtensionFileFilter(extension, fileType));
    }

    /**
     * @param filter {@link FileFilter} to add.
     */
    public void addFileFilter(final FileFilter filter)
    {
        _filters.add(filter);
    }

    /**
     * Initialize the chooser static variable, using the base directory specified by the passed in token. Call
     * addFileFilter after calling this to add filters to the chooser.
     * 
     * @param dialogTitle The title for the chooser window; this can be null for a default title.
     * @param identifier String name for the chooser to initialize.
     * @param saveDirToken String specifying apps-defaults token.
     */
    public static void initializeChooserUsingToken(final String dialogTitle,
                                                   final String identifier,
                                                   final String dirToken)
    {
        String baseDir = MiscTools.getAppsDefaults(dirToken);
        if(baseDir == null)
        {
            baseDir = ".";
        }
        initializeChooser(dialogTitle, identifier, baseDir);
    }

    /**
     * Initialize the chooser static variable, using the base directory specified by the passed in token and
     * subdirectory. Call addFileFilter after calling this to add filters to the chooser.
     * 
     * @param dialogTitle The title for the chooser window; this can be null for a default title.
     * @param identifier String name for the chooser to initialize.
     * @param saveDirToken String specifying apps-defaults token.
     * @param subDir Subdirectory of the directory implied by the saveDirToken.
     */
    public static void initializeChooserUsingToken(final String dialogTitle,
                                                   final String identifier,
                                                   final String dirToken,
                                                   final String subDir)
    {
        String baseDir = MiscTools.getAppsDefaults(dirToken);
        if(baseDir == null)
        {
            baseDir = ".";
        }
        else
        {
            baseDir += "/" + subDir;
        }
        initializeChooser(dialogTitle, identifier, baseDir);

    }

    /**
     * Initializes the chooser to the specified absolute directory. Call addFileFilter after calling this to add filters
     * to the chooser.
     * 
     * @param identifier String identifier for the chooser to initialize
     * @param directory String specifying apps-defaults token.
     */
    public static void initializeChooser(final String dialogTitle, final String identifier, final String directory)
    {
        if(HGlobalFileChooser.GLOBAL_CHOOSERS == null)
        {
            HGlobalFileChooser.GLOBAL_CHOOSERS = new HashMap<String, HGlobalFileChooser>();
        }

        final HGlobalFileChooser chooser = new HGlobalFileChooser(directory);
        chooser.setDialogTitle(dialogTitle);

        //Initialize the specified chooser.
        HGlobalFileChooser.GLOBAL_CHOOSERS.put(identifier, chooser);
    }

    /**
     * Initialize a chooser for global use.
     * 
     * @param identifier Identifier used.
     * @param dialogTitle Title of dialog.
     * @param approveButtonText The text of the approve button.
     * @param overwriteCheck True if file should be checked for existence upon selection and confirmed if yes.
     * @param directory The initial base directory.
     */
    public static void initializeChooser(final String identifier,
                                         final String dialogTitle,
                                         final String approveButtonText,
                                         final boolean overwriteCheck,
                                         final String directory)
    {
        if(HGlobalFileChooser.GLOBAL_CHOOSERS == null)
        {
            HGlobalFileChooser.GLOBAL_CHOOSERS = new HashMap<String, HGlobalFileChooser>();
        }

        final HGlobalFileChooser chooser = new HGlobalFileChooser(directory);
        chooser.setOptions(dialogTitle, approveButtonText, overwriteCheck);

        //Initialize the specified chooser.
        HGlobalFileChooser.GLOBAL_CHOOSERS.put(identifier, chooser);
    }

    /**
     * Gets the specified static file chooser. If it does not exist, one will be initialized with the appropriate index
     * and a directory setting of ".".
     * 
     * @param identifier String name of the file chooser to acquire.
     * @return HGlobalFileChooser specified by the identifier. Call chooseFile or saveFile using the return value as
     *         desired.
     */
    public static HGlobalFileChooser getFileChooser(final String identifier)
    {
        if((HGlobalFileChooser.GLOBAL_CHOOSERS == null) || (HGlobalFileChooser.GLOBAL_CHOOSERS.get(identifier) == null))
        {
            initializeChooser(null, identifier, ".");
        }
        return HGlobalFileChooser.GLOBAL_CHOOSERS.get(identifier);
    }

    /**
     * Calls {@link #getFileChooser(String, String, String, List)} with an initial directory of '.'.
     * 
     * @return An {@link HGlobalFileChooser} with the provided arguments used. If one already exists with the supplied
     *         identifier, the existing one will be returned unchanged. Otherwise, it will be initialized as per the
     *         provided arguments, making use of the specified filters such that the first filter is selected by
     *         default.
     */
    public static HGlobalFileChooser getFileChooser(final String dialogTitle,
                                                    final String identifier,
                                                    final List<FileFilter> filters)
    {
        return getFileChooser(dialogTitle, identifier, new File("."), filters);
    }

    /**
     * @return An {@link HGlobalFileChooser} with the provided arguments used. If one already exists with the supplied
     *         identifier, the existing one will be returned unchanged. Otherwise, it will be initialized as per the
     *         provided arguments, making use of the specified filters such that the first filter is selected by
     *         default.
     */
    public static HGlobalFileChooser getFileChooser(final String dialogTitle,
                                                    final String identifier,
                                                    final File initialDir,
                                                    final List<FileFilter> filters)
    {
        if((HGlobalFileChooser.GLOBAL_CHOOSERS == null) || (HGlobalFileChooser.GLOBAL_CHOOSERS.get(identifier) == null))
        {
            initializeChooser(dialogTitle, identifier, initialDir.getAbsolutePath());
            final HGlobalFileChooser chooser = HGlobalFileChooser.GLOBAL_CHOOSERS.get(identifier);
            for(final FileFilter filter: filters)
            {
                chooser.addFileFilter(filter);
            }
        }
        return HGlobalFileChooser.GLOBAL_CHOOSERS.get(identifier);
    }
}
