package ohd.hseb.hefs.utils.ftp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import ohd.hseb.hefs.utils.tools.StreamTools;

import org.apache.commons.net.ftp.FTP;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPReply;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * This tool wraps commons.net utilities to allow for ftp-ing files to and from a server.
 * 
 * @author hank.herr
 */
public class FTPConductor
{
    private static final Logger LOG = LogManager.getLogger(FTPConductor.class);

    private final FTPClient _ftpClient = new FTPClient();
    private String _serverName = null;

    /**
     * Opens an anonymous connection to the specified server.
     * 
     * @param serverName Name of server.
     * @throws IOException If connection fails.
     */
    public FTPConductor(String serverName) throws IOException
    {
        _serverName = serverName;
        _ftpClient.setDataTimeout(10000);

        establishConnection();
        if(_ftpClient.isConnected())
        {
            LOG.debug("FTPConductor logging in anonymously  to " + serverName + "...");
            if(!_ftpClient.login("anonymous", ""))
            {
                LOG.debug("Login failed: " + _ftpClient.getReplyString());
                throw new IOException("FTP connection failed due to rejected anonymous log-in.");
            }
            LOG.debug("FTPConductor successfully logged in.");
        }
    }

    /**
     * @param serverName Name of server.
     * @param username User name on system to log-in as.
     * @param password The password.
     * @throws IOException If connection fails.
     */
    public FTPConductor(String serverName, String username, String password) throws IOException
    {
        _serverName = serverName;
        _ftpClient.setDataTimeout(30000);
        establishConnection();
        if(_ftpClient.isConnected())
        {
            LOG.debug("FTPConductor logging in as " + username + " to " + serverName + "...");
            if(!_ftpClient.login(username, password))
            {
                LOG.debug("Login failed: " + _ftpClient.getReplyString());
                throw new IOException("FTP connection failed due to rejected anonymous log-in.");
            }
            LOG.debug("FTPConductor successfully logged in.");
        }
    }

    public void attemptReconnect() throws IOException
    {
        establishConnection();
    }

    private void establishConnection() throws IOException
    {
        //Close existing connection if it is not the same server.  If it is the same
        //server, then just return -- nothing to do.
        if(_ftpClient.isConnected())
        {
            return;
        }

        int reply;
        LOG.debug("FTPConductor connecting to " + _serverName + "...");
        _ftpClient.connect(_serverName);
        LOG.debug("FTP connection reply: " + _ftpClient.getReplyString());

        // Validating connection
        reply = _ftpClient.getReplyCode();

        if(!FTPReply.isPositiveCompletion(reply))
        {
            _ftpClient.disconnect();
            LOG.debug("FTP connection validation failed.  Disconnected.");
        }

        LOG.debug("FTPConductor successfully connected to " + _serverName + "...");
    }

    public void setToBinaryTransferMode() throws IOException
    {
        if(_ftpClient.isConnected())
        {
            _ftpClient.setFileType(FTP.BINARY_FILE_TYPE);
        }
        else
        {
            throw new IOException("Cannot setup for binary transfer because FTP session is not connected.");
        }

    }

    public void closeConnection()
    {
        LOG.debug("FTPConductor closing connection...");
        if(_ftpClient.isConnected())
        {
            try
            {
                _ftpClient.logout();
                _ftpClient.disconnect();
            }
            catch(IOException ioe)
            {
                // do nothing
            }
        }
        LOG.debug("FTPConductor connection closed.");
    }

    /**
     * This will not create any directories, so the target parent directory must exist.
     * 
     * @param localAbsolutePathName The full path name of file to put.
     * @param serverAbsolutePathName The full path name of where to place it.
     * @throws IOException
     */
    public void putFileOntoServer(String localAbsolutePathName, String serverAbsolutePathName) throws IOException
    {
        if(!_ftpClient.isConnected())
        {
            throw new IOException("FTP client is not connected.");
        }

        //Change to standard '/' FTP directory separator.
        serverAbsolutePathName = serverAbsolutePathName.replaceAll("\\" + File.separator, "/");

        //Get the parent and file name on server to get.
        int parentIndex = serverAbsolutePathName.lastIndexOf("/");
        String serverParentName = "";
        String serverFileName = serverAbsolutePathName;
        if(parentIndex > 0)
        {
            serverParentName = serverAbsolutePathName.substring(0, parentIndex);
            serverFileName = serverAbsolutePathName.substring(parentIndex + 1);
        }

        InputStream input = null;
        try
        {
            //Change to directory.
            if(!_ftpClient.changeWorkingDirectory(serverParentName))
            {
                LOG.debug("Unable to change to directory " + serverParentName + " on FTP server.");
                throw new IOException("Directory " + serverParentName
                    + " does not appear to exist; cannot write the file " + serverFileName + " within it.");
            }

            //Put the file.
            input = new FileInputStream(localAbsolutePathName);
            LOG.debug("FTPConductor putting file " + localAbsolutePathName + " onto server at "
                + serverAbsolutePathName + " (30 second timeout set)...");
            _ftpClient.storeFile(serverFileName, input);
            LOG.debug("FTPConductor done putting file.");
        }
        catch(IOException e)
        {
            throw new IOException("Failed to put file onto server: " + e.getMessage());
        }
        finally
        {
            StreamTools.closeStream(input);
        }
    }

    /**
     * @param localAbsolutePathName The full path name of where to place it. File.separator should be used to separate
     *            components.
     * @param serverAbsolutePathName The full path name of file to retrieve. File.separator can be used to separate
     *            components; it will be translated to '/' prior to use.
     * @throws IOException
     */
    public void getFileFromServer(String localAbsolutePathName, String serverAbsolutePathName) throws IOException
    {
        if(!_ftpClient.isConnected())
        {
            throw new IOException("FTP client is not connected.");
        }

        //Change to standard '/' FTP directory separator.
        serverAbsolutePathName = serverAbsolutePathName.replaceAll("\\" + File.separator, "/");

        //Get the parent and file name on server to get.
        int parentIndex = serverAbsolutePathName.lastIndexOf("/");
        String serverParentName = "";
        String serverFileName = serverAbsolutePathName;
        if(parentIndex > 0)
        {
            serverParentName = serverAbsolutePathName.substring(0, parentIndex);
            serverFileName = serverAbsolutePathName.substring(parentIndex + 1);
        }

        OutputStream output = null;
        boolean removeFile = false;
        try
        {
            //Change to directory.
            if(!_ftpClient.changeWorkingDirectory(serverParentName))
            {
                LOG.debug("Unable to change to directory " + serverParentName + " on FTP server.");
                throw new IOException("Directory " + serverParentName
                    + " does not appear to exist; cannot download the file " + serverFileName + " within it.");
            }

            //Attempt to retrieve the file.
            LOG.debug("FTPConductor getting file " + serverFileName + " from server and placing it here: "
                + localAbsolutePathName + "...");
            output = new FileOutputStream(localAbsolutePathName);
            _ftpClient.retrieveFile(serverFileName, output);

            //If the output file is 0-size, I'll assume it failed.
            if(new File(localAbsolutePathName).length() == 0)
            {
                LOG.debug("FTPConductor local file " + localAbsolutePathName
                    + " has zero length, meaning the ftp failed.");
                throw new IOException("File to acquire appears to not exist, since the created local file has 0-size.");
            }
            LOG.debug("FTPConductor done getting file.");
        }
        catch(IOException e)
        {
            removeFile = true;
            throw new IOException("Failed to get file from server: " + e.getMessage());
        }
        finally
        {
            StreamTools.closeStream(output);
            File createdFile = new File(localAbsolutePathName);
            if(removeFile)
            {
                createdFile.delete();
            }
        }
    }

    public FTPClient getFTPClient()
    {
        return this._ftpClient;
    }
}
