package ohd.hseb.hefs.utils.ftp;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.base.Strings;
import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.ChannelSftp.LsEntry;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.SftpException;

import ohd.hseb.hefs.utils.TempFile;
import ohd.hseb.hefs.utils.log4j.LoggingTools;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * This tool wraps jsch to allow for sftp-ing files to and from a server.
 * 
 * @author hank.herr
 */
public class SFTPConductor
{
    private static final Logger LOG = LogManager.getLogger(SFTPConductor.class);

    private final JSch _jsch = new JSch();
    private Session _session;
    private ChannelSftp _sftpChannel;
    private final SFTPSettings _sftpSettings;

    /**
     * Opens an anonymous connection to the specified server. Not sure if anonymous works with sftp.
     * 
     * @param serverName Name of server.
     * @throws JSchException
     */
    public SFTPConductor(final String serverName) throws Exception
    {
        _sftpSettings = new SFTPSettings("sftpSettings", serverName);
        _sftpSettings.setUserName("anonymous");
        _sftpSettings.setPassword("filler");
        _sftpSettings.checkSettings();
        establishConnection();
    }

    /**
     * @param serverName Name of server.
     * @param username User name on system to log-in as.
     * @param password The password. To use an generated key, the password must be the passphrase for the key.
     * @throws JSchException
     */
    public SFTPConductor(final String serverName, final String username, final String password) throws Exception
    {
        _sftpSettings = new SFTPSettings("sftpSettings", serverName);
        _sftpSettings.setUserName(username);
        _sftpSettings.setPassword(password);
        _sftpSettings.checkSettings();
        establishConnection();
    }

    /**
     * Open up an SFTP connection using SFTPSettings.
     * 
     * @param settings Settings to use.
     * @throws JSchException
     */
    public SFTPConductor(final SFTPSettings settings) throws Exception
    {
        _sftpSettings = settings;
        _sftpSettings.checkSettings();
        establishConnection();
    }

    public void attemptReconnect() throws JSchException
    {
        establishConnection();
    }

    private void establishConnection() throws JSchException
    {
//XXX TESTING        JSch.setLogger(new MyLogger());

        //If _session is not null and is connected, then return the existing _sftpChannel if 
        //it is connected or establish a new channel and return it.  If a problem occurs, close the session
        //out and let hte code below attempt a reconnect.
        try
        {
            if((_session != null) && (_session.isConnected()))
            {
                if(_sftpChannel.isConnected())
                {
                    return;
                }
                _sftpChannel = (ChannelSftp)_session.openChannel("sftp");
                _sftpChannel.connect();
                return;
            }
        }
        catch(final JSchException e)
        {
            LOG.info("Failed to establish SFTP channel connection for existing SFTP session.  Will close the session and attempt to reconnect.");
            closeConnection();
        }

        //Password message strings used below to simplifyin log messaging
        String passwordStr = "";
        if(_sftpSettings.getPassword().isEmpty())
        {
            passwordStr = " with no password provided";
        }
        String passwordProvided = "provided";
        if(Strings.isNullOrEmpty(_sftpSettings.getPassword()))
        {
            passwordProvided = "not provided";
        }

        //Attempt to set the known hosts for the _jsch.  If a failure occurs, bail.
        try
        {
            //Setup for key authorization if the needed files exist
            if(new File(System.getProperty("user.home") + "/.ssh/known_hosts").exists())
            {
                LOG.info("SFTPConductor found known_hosts file.");
                _jsch.setKnownHosts(System.getProperty("user.home") + "/.ssh/known_hosts");
            }
            if(new File(System.getProperty("user.home") + "/.ssh/id_rsa").exists())
            {
                LOG.info("SFTPConductor found key file id_rsa.");
                _jsch.addIdentity(System.getProperty("user.home") + "/.ssh/id_rsa");
            }
            if(new File(System.getProperty("user.home") + "/.ssh/id_dsa").exists())
            {
                LOG.info("SFTPConductor found key file id_dsa.");
                _jsch.addIdentity(System.getProperty("user.home") + "/.ssh/id_dsa");
            }
        }
        catch(final JSchException e)
        {
            LOG.info("Though an existing known_hosts or id_rsa/id_dsa file was found for this user, a problem occurred attempting to use it (it will be skipped): "
                + e.getMessage());
            if(_sftpSettings.getPassword().isEmpty())
            {
                throw new JSchException("Failed to connect as " + _sftpSettings.getUserName() + " to "
                    + _sftpSettings.getServerName() + " with password " + passwordProvided
                    + ": Since known_hosts/id_rsa/id_dsa could not be processed, sftp with no password cannot work.");
            }
            LOG.debug("Full user login and password was provided for connection, so connection will still be attempted.");
        }

        //Connect...
        try
        {
            //First, the primary server...
            LOG.info("MEFPPE connecting via sftp to primary server " + _sftpSettings.getServerName() + " as user "
                + _sftpSettings.getUserName() + passwordStr + "...");
            connectWithUserAndPassword(_sftpSettings.getServerName(),
                                       _sftpSettings.getUserName(),
                                       _sftpSettings.getPassword());

            //No exception indicates success.
            LOG.info("MEFPPE successfully connected to " + _sftpSettings.getServerName()
                + " with sftp channel established...");
        }
        //Primary connect failed, now try backup servers in order.
        catch(JSchException e)
        {
            LoggingTools.outputStackTraceAsDebug(LOG, e);
            LOG.info("Connection attempt failed: " + e.getMessage());
            for(final String backupServer: _sftpSettings.getBackupServerNames())
            {
                LOG.info("MEFPPE connecting via sftp to backup server " + backupServer + " as user "
                    + _sftpSettings.getUserName() + passwordStr + "...");

                try
                {
                    connectWithUserAndPassword(backupServer, _sftpSettings.getUserName(), _sftpSettings.getPassword());
                    LOG.info("MEFPPE successfully connected to " + backupServer + " with sftp channel established...");
                    return;
                }
                catch(final JSchException e2)
                {
                    //Connection failed, so continue with the for loop.
                    e = e2;
                    LoggingTools.outputStackTraceAsDebug(LOG, e);
                    LOG.info("Connection attempt failed: " + e.getMessage());
                }
            }

            //No success.
            throw new JSchException("Failed to connect as " + _sftpSettings.getUserName() + " with password "
                + passwordProvided + " to primary and all provided backup servers; see earlier messages for reasons.");
        }
    }

    private void connectWithUserAndPassword(final String serverName,
                                            final String userName,
                                            final String password) throws JSchException
    {
        _session = _jsch.getSession(userName, serverName);
        _session.setConfig("StrictHostKeyChecking", "no");
        _session.setConfig("PreferredAuthentications", "publickey,keyboard-interactive,password"); //Prevents having to manually enter Kerberos log-in info.
        _session.setPassword(password);
        _session.connect();
        _sftpChannel = (ChannelSftp)_session.openChannel("sftp");
        _sftpChannel.connect();
    }

// XXX This can be used for testing
//    public static class MyLogger implements com.jcraft.jsch.Logger
//    {
//        static java.util.Hashtable name = new java.util.Hashtable();
//        static
//        {
//            name.put(new Integer(DEBUG), "DEBUG: ");
//            name.put(new Integer(INFO), "INFO: ");
//            name.put(new Integer(WARN), "WARN: ");
//            name.put(new Integer(ERROR), "ERROR: ");
//            name.put(new Integer(FATAL), "FATAL: ");
//        }
//
//        public boolean isEnabled(int level)
//        {
//            return true;
//        }
//
//        public void log(int level, String message)
//        {
//            System.out.print("####>> " + name.get(new Integer(level)));
//            System.out.println("####>> " + message);
//        }
//    }

    public void closeConnection()
    {
        LOG.debug("SFTPConductor closing connection...");
        if(_sftpChannel.isConnected())
        {
            _sftpChannel.disconnect();
            _sftpChannel.exit();
        }
        if(_session.isConnected())
        {
            _session.disconnect();
        }
        LOG.debug("SFTPConductor connection closed.");
    }

    /**
     * This will not create any directories, so the target parent directory must exist.
     * 
     * @param localAbsolutePathName The full path name of file to put.
     * @param serverAbsolutePathName The full path name of where to place it.
     * @throws SftpException
     */
    public void putFileOntoServer(String localAbsolutePathName, String serverAbsolutePathName) throws SftpException
    {
        //Make sure both files use '/' and not '\'.
        localAbsolutePathName = localAbsolutePathName.replace(File.separatorChar, '/');
        serverAbsolutePathName = serverAbsolutePathName.replace(File.separatorChar, '/');

        LOG.debug("Putting file " + localAbsolutePathName + " to file " + serverAbsolutePathName + "...");
        _sftpChannel.put(serverAbsolutePathName, localAbsolutePathName);
        LOG.debug("Done putting file.");
    }

    /**
     * @param localAbsolutePathName The full path name of where to place it. File.separator should be used to separate
     *            components.
     * @param serverAbsolutePathName The full path name of file to retrieve. File.separator can be used to separate
     *            components; it will be translated to '/' prior to use.
     * @throws SftpException
     */
    public void getFileFromServer(final String localAbsolutePathName,
                                  final String serverAbsolutePathName) throws Exception
    {
        getFileFromServer(localAbsolutePathName, serverAbsolutePathName, true);
    }

    private boolean isFileInWorkingDirectory(final String fileName) throws Exception
    {
        return parseFileNamesFromLSOutput(_sftpChannel.ls("*")).contains(fileName);
    }

    /**
     * @param lsOutput {@link Vector} of either {@link LsEntry} instances or {@link String} specifying the file name
     *            directly.
     * @return {@link List} of file names specified in the provided {@link Vector}.
     */
    private List<String> parseFileNamesFromLSOutput(final Vector lsOutput)
    {
        final List<String> names = new ArrayList<>();
        for(final Object o: lsOutput)
        {
            if(o instanceof String)
            {
                names.add((String)o);
            }
            else
            {
                final LsEntry entry = (LsEntry)o;
                names.add(entry.getFilename());
            }
        }
        return names;
    }

    public List<File> listFilesInDirectory(final String baseDirectory, final String pattern) throws Exception
    {
        final String currentPath = _sftpChannel.pwd();

        try
        {
            //Try to cd to the base directory.  Except out if it fails.
            _sftpChannel.cd(baseDirectory);

            //Break down the pattern by forward slash to get the subdirectories.  The standard ls method will not handle
            //* within subdirectories, only at the end, but we need to allow for that possibility.  Hence the algorithm below.
            final SegmentedLine patternSubDirs = new SegmentedLine(pattern, "/", SegmentedLine.MODE_NO_EMPTY_SEGS);

            //Holds the results.
            final List<File> filesFound = new ArrayList<>();

            //No subdirs...
            if(patternSubDirs.getNumberOfSegments() == 1)
            {
                final Vector lsOutput = _sftpChannel.ls(pattern);
                final List<String> fileNames = parseFileNamesFromLSOutput(lsOutput);
                for(final String fileName: fileNames)
                {
                    filesFound.add(FileTools.newFile(baseDirectory, fileName));
                }
            }

            //Subdirs found
            else
            {
                //Build the pattern we are going to use for every subdirectory found.
                final int indexOfFirstSlash = pattern.indexOf('/');
                final String patternWithinSubDir = pattern.substring(indexOfFirstSlash + 1);

                //Get a list of the files within the current base directory that match the first segment of the pattern.
                final String subDirPattern = patternSubDirs.getSegment(0);
                final Vector lsOutput = new Vector();

                //Need to do a special thing here because the ls method will go down into directories if the name does not
                //include a wild card (i.e., its a fixed subdir).  In that case, just use that fixed subdir in the recursion below.
                //Otherwise, we need to process the wildcard, so let ls handle it.
                if(isFileInWorkingDirectory(subDirPattern))
                {
                    lsOutput.add(subDirPattern);
                }
                else
                {
                    lsOutput.addAll(_sftpChannel.ls(subDirPattern));
                }

                final List<String> potentialSubDirsFound = parseFileNamesFromLSOutput(lsOutput);

                //Loop through each one and call this method recursively.
                for(final String potentialSubDirName: potentialSubDirsFound)
                {
                    try
                    {
                        for(final File file: listFilesInDirectory(potentialSubDirName, patternWithinSubDir))
                        {
                            filesFound.add(FileTools.newFile(baseDirectory, file));
                        }
                    }
                    catch(final Throwable t)
                    {
                        //If an exception occurs ithin the recursive call, then either the potentialSubDirName is not 
                        //a subdirectory or the pattern yields nothing under it.  Just skip it without erroring out.
                        LOG.debug(t.getMessage());
                    }
                }
            }

            Collections.sort(filesFound);
            return filesFound;
        }

        //Any exception should cause it to fail out.
        catch(final Throwable t)
        {
            t.printStackTrace();
            throw new Exception("Unable to list files in " + baseDirectory + " matching '" + pattern + "': "
                + t.getMessage());
        }

        //Always return to the original path!
        finally
        {
            try
            {
                _sftpChannel.cd(currentPath);
            }
            catch(final SftpException e)
            {
                e.printStackTrace();
                LOG.error("Failed to return to the path " + currentPath + ": " + e.getMessage());
            }
        }
    }

    /**
     * @param localAbsolutePathName The FINAL location locally of the file to acquire.
     * @param serverAbsolutePathName The location of the file to acquire.
     * @return A {@link TempFile} specifying the temporary file and providing a {@link TempFile#overwrite()} method to
     *         copy the file into its permanent location.
     * @throws Exception If a problem occurs attempting to acquire the file.
     */
    public TempFile getTempFileFromServer(String localAbsolutePathName, String serverAbsolutePathName) throws Exception
    {
        //Make sure both files use '/' and not '\'.
        localAbsolutePathName = localAbsolutePathName.replace(File.separatorChar, '/');
        serverAbsolutePathName = serverAbsolutePathName.replace(File.separatorChar, '/');

        //The resulting temp file.
        final TempFile tempFile = TempFile.make(new File(localAbsolutePathName));

        LOG.debug("Getting file " + serverAbsolutePathName + " as file " + tempFile.getAbsolutePath() + "...");
        _sftpChannel.get(serverAbsolutePathName, tempFile.getAbsolutePath());

        // Check the file size.
        if(tempFile.length() == 0)
        {
            tempFile.delete();
            throw new Exception("Temp file received has 0-length, which is assumed to indicate an error; temp file has been removed.");
        }

        LOG.debug("Done getting file from server as temporary file.");

        return tempFile;
    }

    /**
     * @param localAbsolutePathName The full path name of where to place it. File.separator should be used to separate
     *            components.
     * @param serverAbsolutePathName The full path name of file to retrieve. File.separator can be used to separate
     *            components; it will be translated to '/' prior to use.
     * @param force if true, will overwrite the file even if it is identical.
     * @throws SftpException
     */
    public void getFileFromServer(String localAbsolutePathName,
                                  String serverAbsolutePathName,
                                  final boolean force) throws Exception
    {
        //XXX The functionality of this method may be tested via MEFPPE by manipulating the GEFS test bed files to trigger the various
        //exceptions below.  Uncomment the XXX line below to see the temp dir location (its not tmp on the mac). 

        //Make sure both files use '/' and not '\'.
        localAbsolutePathName = localAbsolutePathName.replace(File.separatorChar, '/');
        serverAbsolutePathName = serverAbsolutePathName.replace(File.separatorChar, '/');

        final TempFile target = TempFile.make(new File(localAbsolutePathName));

        LOG.debug("Getting file " + serverAbsolutePathName + " as file " + target.getAbsolutePath() + "...");
        _sftpChannel.get(serverAbsolutePathName, target.getAbsolutePath());

        try
        {
            // Check the file size.
            if(target.length() == 0)
            {
                throw new Exception("Temp file received has 0-length, which is assumed to indicate an error; temp file will be removed.");
            }

            //Force overwrite
            if(force)
            {
                target.overwrite();
            }
            //Overwrite only if changed
            else
            {
                if(!target.overwriteIfDifferent())
                {
                    throw new Exception("Temp file received is different from the local file, "
                        + "but the attempt to copy over the existing file failed (temp file will be removed):");
                }
            }
        }
        catch(final Throwable t)
        {
            final Exception e = new Exception("Failure to copy received temp file to permanent local location: "
                + t.getMessage());
            e.setStackTrace(t.getStackTrace());
            throw e;
        }
        //ALWAYS DELETE
        finally
        {
            //XXX System.err.println("####>> TEMP FILE -- " + target.getAbsolutePath());
            target.delete();
        }

        LOG.debug("Done getting file.");
    }
}
