package ohd.hseb.hefs.utils.geo;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import ohd.hseb.hefs.utils.tools.NumberTools;

public class CoordinateGrid
{

    public List<Double> _latitudePoints = new ArrayList<Double>();
    public List<Double> _longitudePoints = new ArrayList<Double>();

    /**
     * @param latitudePoints List of latitudes to use.
     * @param longitudePoints List of longitudes to use.
     */
    public CoordinateGrid(final Collection<Double> latitudePoints, final Collection<Double> longitudePoints)
    {
        _latitudePoints.addAll(latitudePoints);
        _longitudePoints.addAll(longitudePoints);
    }

    /**
     * Builds the coordinate grid by generating numbers using
     * {@link NumberTools#generateNumberSequence(double, double, double)} passing in the start, ending bound, and step
     * size for the latitude and longitude. All variable limits are inclusive.
     * 
     * @param latStart Must be between -90 and 90.
     * @param latEndingBound Must be between -90 and 90.
     * @param latStepSize Step size between.
     * @param lonStart Must be between -180 and 180.
     * @param lonEndingBound Must be between -180 and 180.
     * @param lonStepSize Step size between.
     */
    public CoordinateGrid(final double latStart,
                          final double latEndingBound,
                          final double latStepSize,
                          final double lonStart,
                          final double lonEndingBound,
                          final double lonStepSize)
    {
        if((latStart <= -90) || (latStart >= 90))
        {
            throw new IllegalArgumentException("Starting latitude must be between -90 and 90, inclusive.");
        }
        if((latEndingBound <= -90) || (latEndingBound >= 90))
        {
            throw new IllegalArgumentException("Ending bound on latitude must be between -90 and 90, inclusive.");
        }
        if((lonStart <= -180) || (lonStart >= 180))
        {
            throw new IllegalArgumentException("Starting longitude must be between -180 and 180, inclusive.");
        }
        if((lonEndingBound <= -180) || (lonEndingBound >= 180))
        {
            throw new IllegalArgumentException("Ending bound on longitude must be between -180 and 180, inclusive.");
        }

        _latitudePoints = NumberTools.generateNumberSequence(latStart, latStepSize, latEndingBound);
        _longitudePoints = NumberTools.generateNumberSequence(lonStart, lonStepSize, lonEndingBound);

    }

    /**
     * @return Distance between the two CoordinateGridPoints, based on equations in MEFP.
     */
    public double computeCoordinateDistance(final CoordinateGridPoint point1, final CoordinateGridPoint point2)
    {
        return Math.sqrt(Math.pow(point1.getLatitude() - point2.getLatitude(), 2d) + 0.64d
            * Math.pow(point1.getLongitude() - point2.getLongitude(), 2d));
    }

    /**
     * If a point is in the exact middle between two latitudes or longitudes, this algorithm will identify the smaller
     * of the two lats or lons as the nearest neighbor.
     * 
     * @param point Point to look for the nearest neighbor.
     * @param mustBeInsideGrid If true, then return null if the point is outside the grid.
     * @return The nearest grid point, or null if it is outside the grid and mustBeInsideGrid is true.
     */
    public CoordinateGridPoint findNearestPoint(final CoordinateGridPoint point, final boolean mustBeInsideGrid)
    {
        //NOTE: This method is tested via MEFPPE CFSv2CoordinateFileHandlerTest
        final double lat = point.getLatitude();
        final double lon = point.getLongitude();
        int latIndex = 0;
        int lonIndex = 0;

        //Find the first lat and lon that are strictly greater than the provided point
        while((latIndex < _latitudePoints.size()) && (_latitudePoints.get(latIndex) <= lat))
        {
            latIndex++;
        }
        while((lonIndex < _longitudePoints.size()) && (_longitudePoints.get(lonIndex) <= lon))
        {
            lonIndex++;
        }

        //Outside of the grid...
        if((latIndex == 0) || (lonIndex == 0))
        {
            if(mustBeInsideGrid)
            {
                return null;
            }
            return new CoordinateGridPoint(_latitudePoints.get(latIndex), _longitudePoints.get(lonIndex));
        }
        if((latIndex == _latitudePoints.size()) || (lonIndex == _longitudePoints.size()))
        {
            if(mustBeInsideGrid)
            {
                return null;
            }
            return new CoordinateGridPoint(_latitudePoints.get(latIndex), _longitudePoints.get(lonIndex));
        }

        CoordinateGridPoint workingPoint;
        CoordinateGridPoint nearestPoint = null;
        double smallestDist = Double.MAX_VALUE;
        double workingDist;
        for(int i = latIndex - 1; i < latIndex + 1; i++)
        {
            for(int j = lonIndex - 1; j < lonIndex + 1; j++)
            {
                workingPoint = new CoordinateGridPoint(_latitudePoints.get(i), _longitudePoints.get(j));
                workingDist = computeCoordinateDistance(point, workingPoint);
                if(workingDist < smallestDist)
                {
                    smallestDist = workingDist;
                    nearestPoint = workingPoint;
                }
            }
        }
        return nearestPoint;
    }

    @Override
    public String toString()
    {
        final String results = "CoordinateGrid: latitudes = " + _latitudePoints.toString() + "; longitudes = "
            + _longitudePoints.toString();
        return results;
    }
}
