package ohd.hseb.hefs.utils.geo;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import ohd.hseb.hefs.utils.tools.StreamTools;

//import org.apache.log4j.lf5.util.StreamUtils;
import org.apache.commons.io.IOUtils;

/**
 * All of the coordinate system resources are handled via this tool. There are currently three coordinate files, all for
 * the Gaussian coordinate system: T126, T190, and T254. There are three static constants defined to point to those
 * files: {@link #GAUSSIAN_T126_COORDINATES}, {@link #GAUSSIAN_T190_COORDINATES}, and {@link #GAUSSIAN_T254_COORDINATES}
 * . To acquire the {@link CoordinateGrid} for a system of coordinates, call {@link #loadCoordinates()} using the
 * appropriate static variable.
 * 
 * @author hankherr
 */
public class CoordinateSystemFilesHandler
{
//    private static final Logger LOG = Logger.getLogger(CoordinateSystemFilesHandler.class);

    //Use only '/' as file separators here.  Do NOT use '\'!!!
    public static CoordinateSystemFilesHandler GAUSSIAN_T126_COORDINATES = new CoordinateSystemFilesHandler("coordinateSystems/GaussianGrid_T126_LatN_CONUS",
                                                                                                            "coordinateSystems/GaussianGrid_T126_LonW_CONUS");
    public static CoordinateSystemFilesHandler GAUSSIAN_T190_COORDINATES = new CoordinateSystemFilesHandler("coordinateSystems/GaussianGrid_T190_LatN_CONUS",
                                                                                                            "coordinateSystems/GaussianGrid_T190_LonW_CONUS");
    public static CoordinateSystemFilesHandler GAUSSIAN_T254_COORDINATES = new CoordinateSystemFilesHandler("coordinateSystems/GaussianGrid_T254_LatN_CONUS",
                                                                                                            "coordinateSystems/GaussianGrid_T254_LonW_CONUS");

    private final String _latitudeResourceName;
    private final String _longitudeResourceName;

    /**
     * @param latitudeResourceName The resource name for latitudes.
     * @param longitudeResourceName The resource name for longitudes.
     */
    public CoordinateSystemFilesHandler(final String latitudeResourceName, final String longitudeResourceName)
    {
        _latitudeResourceName = latitudeResourceName;
        _longitudeResourceName = longitudeResourceName;
    }

    private String getLatitudeResourceName()
    {
        return _latitudeResourceName;
    }

    private String getLongitudeResourceName()
    {
        return _longitudeResourceName;
    }

    private String getLatitudeFileNameOnly()
    {
        final int index = getLatitudeResourceName().lastIndexOf('/');
        return getLatitudeResourceName().substring(index + 1);
    }

    private String getLongitudeFileNameOnly()
    {
        final int index = getLongitudeResourceName().lastIndexOf('/');
        return getLongitudeResourceName().substring(index + 1);
    }

    /**
     * Assumes that the last character on each line is a 'N', 'W', 'E', 'S' coordinate character. Before that is one
     * number. The number will be assigned a negative value if the character is 'W' or 'S'.
     * 
     * @param file File to read
     * @return List of numbers found
     * @throws Exception
     */
    private Collection<Double> readListOfCoordinates(final InputStream stream) throws Exception
    {
        final BufferedReader reader = new BufferedReader(new InputStreamReader(stream));
        String aline;
        final ArrayList<Double> results = new ArrayList<Double>();
        while((aline = reader.readLine()) != null)
        {
            aline = aline.trim();
            if(!aline.isEmpty())
            {
                final String numStr = aline.substring(0, aline.length() - 1);

                //Check char -- 
                double mult = 1.0D;
                if((aline.charAt(aline.length() - 1) == 'W') || (aline.charAt(aline.length() - 1) == 'S'))
                {
                    mult = -1.0D;
                }

                try
                {
                    results.add(mult * Double.parseDouble(numStr));
                }
                catch(final NumberFormatException e)
                {
                    throw new Exception("String '" + numStr + "' is not a number.");
                }
            }
        }
        Collections.sort(results);
        return results;
    }

    /**
     * @return The coordinates loaded from the system resource files.
     * @throws Exception If the system resource could not be loaded.
     */
    public CoordinateGrid loadCoordinates() throws Exception
    {
        //Latitude stream
        final String latRes = this.getLatitudeResourceName();
        final InputStream latStream = ClassLoader.getSystemResourceAsStream(latRes);
        if(latStream == null)
        {
            throw new Exception("Unable to find CFSv2 gaussian grid latitudes resource " + latRes);
        }

        //Longitude stream
        final String lonRes = this.getLongitudeResourceName();
        final InputStream lonStream = ClassLoader.getSystemResourceAsStream(lonRes);
        if(lonStream == null)
        {
            throw new Exception("Unable to find CFSv2 gaussian grid longitudes resource " + lonRes);
        }

        try
        {
            final Collection<Double> latitudes = readListOfCoordinates(latStream);
            final Collection<Double> longitudes = readListOfCoordinates(lonStream);
            return new CoordinateGrid(latitudes, longitudes);
        }
        finally
        {
            latStream.close();
            lonStream.close();
        }
    }

    public void copyCoordinateFilesToDirectory(final File dir) throws Exception
    {
        InputStream latStream = null;
        InputStream lonStream = null;
        OutputStream latOutStream = null;
        OutputStream lonOutStream = null;
        try
        {
            //Latitude streams
            final String latRes = this.getLatitudeResourceName();
            latStream = ClassLoader.getSystemResourceAsStream(latRes);
            if(latStream == null)
            {
                throw new Exception("Unable to find latitudes resource file " + latRes);
            }
            latOutStream = new FileOutputStream(new File(dir.getAbsolutePath() + File.separator
                + getLatitudeFileNameOnly()));           

            //Longitude streams
            final String lonRes = this.getLongitudeResourceName();
            lonStream = ClassLoader.getSystemResourceAsStream(lonRes);
            if(lonStream == null)
            {
                throw new Exception("Unable to find longitudes resource file " + lonRes);
            }
            lonOutStream = new FileOutputStream(new File(dir.getAbsolutePath() + File.separator
                + getLongitudeFileNameOnly()));
                       
            IOUtils.copy(latStream, latOutStream);
            IOUtils.copy(lonStream, lonOutStream);                                    

        }
        catch(final Exception e)
        {
            throw e;
        }
        finally
        {
            StreamTools.closeStream(latStream);
            StreamTools.closeStream(lonStream);
            StreamTools.closeStream(latOutStream);
            StreamTools.closeStream(lonOutStream);
        }

    }
}
