package ohd.hseb.hefs.utils.gui.about;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.HashMap;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;

import ohd.hseb.hefs.utils.gui.tools.SelfListeningButton;
import ohd.hseb.hefs.utils.gui.tools.SelfListeningMenuItem;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.xml.XMLReadable;

/**
 * <br>
 * The about dialog includes a NOAA logo and an NWS logo on either side of a title followed by lines displaying
 * information about the product version, FEWS version, and CHPS version. A button can be created triggering this dialog
 * via the static method {@link #createAboutDialogButton(Component, String)}, and a menu item via the method
 * {@link #createAboutDialogMenuItem(Component, String)}. A standard infoIcon is used for the button, while the menu
 * item is labeled "About".<br>
 * <br>
 * The constructor is given a parent component for the dialog. It finds the about information file by finding the
 * nearest parent that is annotated with the {@link AboutFile} annotation starting with the provided parent component.
 * The file specified in the value of the annotation must be a system resource (i.e., a jarred file) and must be an XML
 * file that follows the OHDconfigInfo schema. It will be read via the {@link OHDConfigInfo} which implements
 * {@link XMLReadable}.
 * 
 * @author Hank.Herr
 */
public class AboutDialog extends JDialog
{
//    private static final Logger LOG = Logger.getLogger(AboutDialog.class);
    private static HashMap<String, AboutDialog> OPEN_DIALOGS = new HashMap<String, AboutDialog>();
    private static final long serialVersionUID = -3584190748565411890L;

    // _aboutInformation needs to be a storage device containing the about information.
    private OHDConfigInfo _aboutInformation = null;

    /**
     * @param parentComponent The component used to determine which parent relative to which to display the dialog. The
     *            method {@link SwingTools#getGlobalDialogParent(Component)} is called.
     * @param programName Name of hte program
     */
    public AboutDialog(final Component parentComponent, final String programName)
    {
        loadAboutInformation(parentComponent);
        setTitle("About " + programName);
        createDisplay(parentComponent, programName);
    }

    private String getMessage()
    {
        final StringBuffer sb = new StringBuffer();
        sb.append("<html><body>");
        //Display the about file information line by line.
        //The about information must be output here using HTML to make it multiline.  A special message should
        //be displayed if the about information could not be loaded.
        if(_aboutInformation != null)
        {
            sb.append("<div>Product Version: " + _aboutInformation.getVersion() + "<br />");
            sb.append("Date: " + _aboutInformation.getDateString() + "<br />");
            sb.append("<div>Compatible OHD Core Version: " + _aboutInformation.getOhdCoreVersion() + "<br />");
            sb.append("<div>Compatible FEWS Version: " + _aboutInformation.getFewsVersion() + "<br /><br />");
        }
        else
        {
            sb.append("<div>No about/version information can be found.<br/></br>");
        }
        sb.append("Office of Water Prediction<br />");
        sb.append("National Weather Service<br />");
        sb.append("</div></body></html>");

        return sb.toString();
    }

    private void createDisplay(final Component parentComponent, final String programName)
    {
        final JButton okButton = new JButton("OK");

        //Upper panel displays the NOAA logo, a title, and the NWS logo.
        final JPanel upperPanel = new JPanel(new BorderLayout());
        upperPanel.add(new JLabel(SwingTools.loadImageIcon("hefsImages/noaa_logo.png")), BorderLayout.WEST);
        final JPanel labelPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
        labelPanel.add(new JLabel("<html><body><div align=center><h1 align='center' style='color:blue'>" + programName
            + "</h1></div></body></html>"));
        labelPanel.setBackground(new Color(0, 0, 0, 0));
        upperPanel.add(labelPanel, BorderLayout.CENTER);
        upperPanel.add(new JLabel(SwingTools.loadImageIcon("hefsImages/nws_logo.png")), BorderLayout.EAST);
        upperPanel.setBackground(new Color(0, 0, 0, 0));
        upperPanel.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));

        //The middle panels displays the message.
        final JPanel middlePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        middlePanel.add(new JLabel(getMessage()));
        middlePanel.setBackground(new Color(0, 0, 0, 0));
        middlePanel.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));

        //An OK button.
        final JPanel lowerPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
        lowerPanel.add(okButton);
        lowerPanel.setBackground(new Color(0, 0, 0, 0));
        lowerPanel.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));

        final JPanel fullPanel = new JPanel(new BorderLayout());
        fullPanel.add(upperPanel, BorderLayout.NORTH);
        fullPanel.add(middlePanel, BorderLayout.CENTER);
        fullPanel.add(lowerPanel, BorderLayout.SOUTH);
        fullPanel.setBackground(Color.white);

        getContentPane().add(fullPanel);

        okButton.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent evt)
            {
                dispose(); //triggers a window closed event!
            }
        });

        pack();

        setLocationRelativeTo(SwingTools.getGlobalDialogParent(parentComponent));
        setResizable(false);
        setModal(true);
        setDefaultCloseOperation(DISPOSE_ON_CLOSE);
    }

    /**
     * Loads the about information object based on the given component. It finds the first parent of the component (or
     * the component itself) that has the {@link AboutFile} annotation. It then loads the about information based on the
     * annotation's value.
     * 
     * @param component the component to retrieve the about file for
     * @return the specified about file, or null if none was specified
     */
    public void loadAboutInformation(final Component component)
    {
        _aboutInformation = OHDConfigInfo.loadOHDConfigInfoFromParent(component);
    }

    /**
     * @param parent The parent relative to which the dialog is displayed. It, or an ancestor, must be annotated with an
     *            {@link AboutFile} annotation for the about information to be found and loaded.
     * @param programName The name of the program displayed in the title of the dialog and used to track which about
     *            dialogs are open.
     * @return A button that, when clicked, will open an about dialog. The button uses the standard infoIcon images in
     *         nonsrc/hefsIcons.
     */
    @SuppressWarnings("serial")
    public static JButton createAboutDialogButton(final Component parent, final String programName)
    {
        return new SelfListeningButton("infoIcon")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                openAboutDialog(parent, programName);
            }
        };
    }

    /**
     * Returns a JMenuItem that, when clicked, will open an about dialog. The menu item is labeled "About".
     * 
     * @param parent The parent relative to which the dialog is displayed. It, or an ancestor, must be annotated with an
     *            {@link AboutFile} annotation for the about information to be found and loaded.
     * @param programName The name of the program displayed in the title of the dialog and used to track which about
     *            dialogs are open.
     * @return
     */
    @SuppressWarnings("serial")
    public static JMenuItem createAboutDialogMenuItem(final Component parent, final String programName)

    {
        return new SelfListeningMenuItem("About")
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                openAboutDialog(parent, programName);
            }
        };
    }

    /**
     * Opens an AboutDialog. It will check, first, to see if the dialog is already open and being displayed. If so, it
     * will have it request focus. Otherwise, it will create a new one and make it visible.
     * 
     * @param parent
     * @param programName
     */
    private static void openAboutDialog(final Component parent, final String programName)
    {
        //If a dialog is already open, then have it request focus.
        if(OPEN_DIALOGS.containsKey(programName))
        {
            OPEN_DIALOGS.get(programName).requestFocus();
        }
        else
        {
            final AboutDialog dialog = new AboutDialog(parent, programName);

            //This is  needed to remove the dialog when it is closed.
            dialog.addWindowListener(new WindowAdapter()
            {

                @Override
                public void windowClosed(final WindowEvent e)
                {
                    OPEN_DIALOGS.remove(programName);
                }
            });

            dialog.setVisible(true);
            OPEN_DIALOGS.put(programName, dialog);
        }
    }
}
