package ohd.hseb.hefs.utils.gui.about;

import java.awt.Component;

import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import org.w3c.dom.Element;

/**
 * Class for storing OHDConfigInfo file information, which provides application specific version information. This class
 * is used by HEFS to read in the file. There is also a version in CHPS common that reads it in, but requires over twice
 * as much code to do so. The method {@link #printVersionInformation(Logger)} outputs the version info to the logger in
 * a standard way. Static methods are provided for loading instances of {@link OHDConfigInfo} based on resource name,
 * class, object, and component's parent.
 * 
 * @author hank.herr
 */
public class OHDConfigInfo implements XMLReadable
{
    //private static final Logger LOG = Logger.getLogger(OHDConfigInfo.class);
	
    private static final Logger LOG = LogManager.getLogger(OHDConfigInfo.class);
    public static String VERSION_ATTRIBUTE_NAME = "version";

    private final XMLString _version = new XMLString("version");
    private final XMLString _dateString = new XMLString("date");
    private final XMLString _ohdCoreVersion = new XMLString("ohdCoreVersion");
    private final XMLString _fewsVersion = new XMLString("fewsVersion");

    @Override
    public XMLReader getReader()
    {
        return new CompositeXMLReader("OHDconfigInfo", _version, _dateString, _ohdCoreVersion, _fewsVersion);
    }

    public String getVersion()
    {
        return _version.get();
    }

    public String getDateString()
    {
        return _dateString.get();
    }

    public String getOhdCoreVersion()
    {
        return _ohdCoreVersion.get();
    }

    public String getFewsVersion()
    {
        return _fewsVersion.get();
    }

    /**
     * Prints the version information to the provided logger.
     * 
     * @param logger
     */
    public void printVersionInformation(final Logger logger)
    {
        logger.info("Version Control Info:");
        logger.info("Version = " + getVersion());
        logger.info("Date  = " + getDateString());
        logger.info("Compatible OHD Core Version = " + getOhdCoreVersion());
        logger.info("Compatible FEWS Version = " + getFewsVersion());
    }

    /**
     * Loads the {@link OHDConfigInfo} for the provided {@link Class} and appends a "version" XML element to the
     * provided main element.
     */
    public static void appendVersionAttribute(final Element mainElement, final Class versionClass)
    {
        final OHDConfigInfo aboutInfo = OHDConfigInfo.loadOHDConfigInfo(versionClass);
        String version = "unknown";
        if(aboutInfo != null)
        {
            version = aboutInfo.getVersion();
        }
        mainElement.setAttribute(VERSION_ATTRIBUTE_NAME, version);
    }

    /**
     * @param resourceName
     * @return An {@link OHDConfigInfo} containing the version information loaded from the resource.
     */
    public static OHDConfigInfo loadOHDConfigInfo(final String resourceName)
    {//final URL myURL = this.getClass().getClassLoader().getResource(resourceName);
        final OHDConfigInfo aboutInformation = new OHDConfigInfo();
        if(aboutInformation.getClass().getClassLoader().getResource(resourceName) != null)
        {
            //here load the about information from the XML.          
            try
            {
                XMLTools.readXMLFromResource(resourceName, aboutInformation);
                return aboutInformation;
            }
            catch(final Exception e)
            {
                LOG.warn("Unable to load version information from resource '" + resourceName + "': " + e.getMessage());
            }
        }

        return null;
    }

    /**
     * @param aboutClass Class for which to display about information.
     * @return An {@link OHDConfigInfo} containing the version information loaded based on the annotation
     *         {@link AboutFile} for the provided class. Null is returned if their is no {@link AboutFile} annotation
     *         for the given class.
     */
    public static OHDConfigInfo loadOHDConfigInfo(final Class<? extends Object> aboutClass)
    {
        final AboutFile annotation = aboutClass.getAnnotation(AboutFile.class);
        if(annotation == null)
        {
            return null;
        }
        return loadOHDConfigInfo(annotation.value());
    }

    /**
     * @param aboutObject Object for which to display about information.
     * @return Calls {@link #loadOHDConfigInfo(Class)} given the provided object's class. Null is returned if the
     *         provided object's class does not contain an {@link AboutFile} annotation.
     */
    public static OHDConfigInfo loadOHDConfigInfo(final Object aboutObject)
    {
        return loadOHDConfigInfo(aboutObject.getClass());
    }

    /**
     * @param c Component for which to determine about information based on its first found parent with the
     *            {@link AboutFile} annotation.
     * @return Calls {@link SwingTools#getParentWithAnnotation(Component, Class)} for the given component looking for
     *         the {@link AboutFile} annotation. If no parent is found with that annotation, null is returned.
     *         Otherwise, calls {@link #loadOHDConfigInfo(Object)} given the found parent.
     */
    public static OHDConfigInfo loadOHDConfigInfoFromParent(final Component c)
    {
        final Component parent = SwingTools.getParentWithAnnotation(c, AboutFile.class);
        if(parent == null)
        {
            return null;
        }
        return loadOHDConfigInfo(parent);
    }

}
