package ohd.hseb.hefs.utils.gui.components;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.AbstractAction;
import javax.swing.JFrame;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSplitPane;
import javax.swing.SwingUtilities;

import ohd.hseb.hefs.utils.tools.IconTools;
import ohd.hseb.hefs.utils.tools.Procedure;
import ohd.hseb.util.IntegerHolder;

/**
 * A panel designed to be detachable from its parent. The panel must be provided with two {@link Procedure}s: one for
 * detaching the panel and one for attaching the panel. See the methods {@link #detach()} and {@link #reattach()} for a
 * description of what activities are done by this panel. Anything else that must be done when detaching or attaching
 * must be handled in the {@link Procedure} instances.<br>
 * <br>
 * A constructor appropriate for a {@link JSplitPane} is provided.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class DetachablePanel extends JPanel
{
    private final Container _parentWhenAttached;
    private final Procedure _detachProcedure;
    private final Procedure _reattachProcedure;
    private final String _detachedTitle;
    private boolean _isDetached = false;

    /**
     * If not null, this is the frame to displays the detached content.
     */
    private JFrame _frame = null;

    /**
     * Standard constructor.
     * 
     * @param detachedTitle The title of the detached frame.
     * @param parentWhenAttached The parent of this panel when attached.
     * @param detachProcedure The {@link Procedure} to perform when detaching this panel, in addition to steps performed
     *            within the {@link #detach()} method.
     * @param reattachProcedure The {@link Procedure} to perform when re-attaching this panel, in addition to steps
     *            performed within the {@link #reattach()} method.
     */
    public DetachablePanel(final String detachedTitle,
                           final Container parentWhenAttached,
                           final Procedure detachProcedure,
                           final Procedure reattachProcedure)
    {
        _parentWhenAttached = parentWhenAttached;
        _reattachProcedure = reattachProcedure;
        _detachProcedure = detachProcedure;
        _detachedTitle = detachedTitle;
    }

    /**
     * Special constructor designed to use with a {@link JSplitPane} parent.
     * 
     * @param detachedTitle The title of the detached frame.
     * @param splitPaneParent The parent of this panel when attached.
     * @param topOrLeftComponentFlag True if the component is the top or left component in the {@link JSplitPane}
     *            parent; false if bottom or right.
     * @param dividerSizeWhenAttached When attached, this is the size of the divider to use. When the content is
     *            detached, the divider size is set to 0 so that it is not visible. When re-attaching, the divider size
     *            is set to this value.
     */
    public DetachablePanel(final String detachedTitle,
                           final JSplitPane splitPaneParent,
                           final boolean topOrLeftComponentFlag,
                           final int dividerSizeWhenAttached)
    {

        _parentWhenAttached = splitPaneParent;
        _detachedTitle = detachedTitle;
        final IntegerHolder dividerLocationHolder = new IntegerHolder(-1);
        _reattachProcedure = new Procedure()
        {
            @Override
            public void perform()
            {
                if(topOrLeftComponentFlag)
                {
                    splitPaneParent.setTopComponent(DetachablePanel.this);
                }
                else
                {
                    splitPaneParent.setBottomComponent(DetachablePanel.this);
                }
                splitPaneParent.setDividerSize(dividerSizeWhenAttached);
                splitPaneParent.setDividerLocation(dividerLocationHolder.getValue());
                DetachablePanel.this.setVisible(true);
            };
        };
        _detachProcedure = new Procedure()
        {
            @Override
            public void perform()
            {
                dividerLocationHolder.setValue(splitPaneParent.getDividerLocation());
                if(topOrLeftComponentFlag)
                {
                    splitPaneParent.setTopComponent(null);
                }
                else
                {
                    splitPaneParent.setBottomComponent(null);
                }
                splitPaneParent.setDividerSize(0);
            };
        };
    }

    private final void constructFrame()
    {
        _frame = new JFrame(_detachedTitle);
        _frame.setLocationRelativeTo(_parentWhenAttached);
        _frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);

        final JPanel panel = new JPanel(new BorderLayout());
        _frame.setContentPane(panel);

        _frame.addWindowListener(new WindowAdapter()
        {
            @Override
            public void windowClosed(final WindowEvent e)
            {
                reattach();
            }
        });
    }

    /**
     * @return True if this panel is currently detached.
     */
    public boolean isDetached()
    {
        return _isDetached;
    }

    /**
     * This does the detaching of the panel. It calls {@link #_detachProcedure}'s {@link Procedure#perform()} method,
     * constructs the frame if necessary, sets its size based on this panel's current size, places this panel inside the
     * frame, and makes the frame visible. The {@link #_detachProcedure} must handle any activity needed other than
     * those just mentioned, including possibly removing this from the parent.<br>
     * <br>
     * This does nothing if the panel is already detached.
     */
    public final void detach()
    {
        if(isDetached())
        {
            return;
        }
        _isDetached = true;
        _detachProcedure.perform();

        if(_frame == null)
        {
            constructFrame();
        }
        _frame.setSize(getSize().width + 5, getSize().height + 10);
        _frame.getContentPane().add(this, BorderLayout.CENTER);
        _frame.setVisible(true);
    }

    /**
     * This does the reattaching of the panel. It makes the frame invisible, removes its content, and then calls
     * {@link #_reattachProcedure}'s {@link Procedure#perform()} method. The {@link #_detachProcedure} must handle any
     * activity needed other than those just mentioned, including add this from the parent and making it visible.<br>
     */
    public final void reattach()
    {
        if(!isDetached())
        {
            return;
        }
        _isDetached = false;
        _frame.setVisible(false);
        _frame.getContentPane().removeAll();

        _reattachProcedure.perform();
    }

    /**
     * @return A {@link JPopupMenu} that contains a single item for detaching/attaching the table along with other items
     *         dictated by the provided components.
     */
    public JPopupMenu constructPopupMenu(final Component[] additionalComponents)
    {
        final JMenuItem detachItem = new JMenuItem(buildDetachAction());
        final JPopupMenu popupMenu = new JPopupMenu();
        popupMenu.add(detachItem);
        if(additionalComponents != null)
        {
            for(final Component c: additionalComponents)
            {
                popupMenu.add(c);
            }
        }
        return popupMenu;
    }

    /**
     * @return An AbstractAction that can be used to construct a button or menu item to detach/reattach the table.
     */
    public AbstractAction buildDetachAction()
    {
        return new AbstractAction("Detach/Reattach Table", IconTools.getHSEBIcon("detachWindow14x14"))
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                if(isDetached())
                {
                    reattach();
                }
                else
                {
                    detach();
                }
            }
        };
    }

    /**
     * If this is detached, then this will also make the parent {@link #_frame} visible.
     */
    @Override
    public void setVisible(final boolean b)
    {
        super.setVisible(b);
        if(isDetached())
        {
            _frame.setVisible(b);
        }
    }

    public void bringToFront()
    {
        if(isDetached())
        {
            SwingUtilities.invokeLater(new Runnable()
            {
                @Override
                public void run()
                {
                    _frame.toFront();
                    _frame.repaint();
                }
            });
        }
    }
}
