package ohd.hseb.hefs.utils.gui.components;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.WindowConstants;
import javax.swing.border.EtchedBorder;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import ohd.hseb.hefs.utils.EventBusUser;
import ohd.hseb.hefs.utils.collect.CollectionListModel;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.notify.ObjectModifiedNotice;
import ohd.hseb.hefs.utils.notify.collect.NotifyingCollection;
import ohd.hseb.hefs.utils.tools.IconTools;
import ohd.hseb.hefs.utils.tools.MapTools;

import com.google.common.base.Function;
import com.google.common.base.Functions;
import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * A panel which allows the user to select items from a master list.<br/>
 * Any time a change is made, a ItemSelection
 * 
 * @author alexander.garbarino
 * @param <E> the type of item to select from.
 */
@SuppressWarnings("serial")
public class ItemSelectionPanel<E> extends JPanel
{
    private static final ImageIcon _leftIcon = IconTools.getHSEBIcon("navigate_left_big");
    private static final ImageIcon _rightIcon = IconTools.getHSEBIcon("navigate_right_big");

    /**
     * The master collection of all available items.
     */
    private final CollectionListModel<E> _masterCollection;

    /**
     * Map tracking which side an element is currently positioned.
     */
    private final Map<E, Boolean> _usedMap;

    /**
     * A collection which will be modified to contain the selected items.
     */
    private Collection<E> _outputCollection;

    private final CollectionListModel<E> _unselectedCollection;
    private final CollectionListModel<E> _selectedCollection;
    private final JLabel _unusedLabel;
    private final CollectionListModel<E> _unusedView;
    private final JList _unusedList;
    private final JLabel _usedLabel;
    private final CollectionListModel<E> _usedView;
    private final JList _usedList;
    private JButton _moveButton;

    /**
     * Which of the two JLists is currently selected.
     */
    private Boolean _activeList = null;

    private final EventBus _internalBus;

    /**
     * Predicate for a given element being currently selected.
     */
    private final Predicate<E> _isSelected = new Predicate<E>()
    {
        @Override
        public boolean apply(final E element)
        {
            return MapTools.get(_usedMap, element, false);
        }
    };

    /**
     * Predicate for any given element being visible. If this is a component as well, it will be added to this panel to
     * let the user interact with it. If it implements EventBusUser, then this will give it an event post to post
     * {@link ObjectModifiedNotice}s to.
     */
    private final Predicate<E> _isVisible;

    private final Function _displayTransform;

    /**
     * Creates an item selection panel. This takes a collection and allows the user to specify a sub set of the elements
     * to use.
     * 
     * @param items The collection of items that backs this panel.<br/>
     *            (!Currently Unimplemented!) If the collection implements {@link List}, then this panel will allow the
     *            user to reposition items.<br/>
     *            If the collection implements {@link NotifyingCollection}, then this will react to changes in the
     *            collection.
     * @param visibleFilter A predicate specifying which elements of the collection are visible. May be null.<br/>
     *            If this extends {@link Component}, then it will be added to this panel to allow the user to modify it.<br/>
     *            If this implements {@link EventBusUser}, then this will give it an event bus and react to
     *            {@link ObjectModifiedNotice}s by refreshing the display.
     * @param displayTransform An optional function that is applied before the items are displayed in the lists.
     */
    @SuppressWarnings("unchecked")
    public ItemSelectionPanel(final Collection<E> items,
                              final Predicate<E> visibleFilter,
                              final Function<E, ?> displayTransform)
    {
        // Setup collections and views.
        _masterCollection = new CollectionListModel<E>(items);
        if(items instanceof NotifyingCollection)
        {
            ((NotifyingCollection)items).register(this);
        }
        _unselectedCollection = _masterCollection.filter(Predicates.not(_isSelected));
        _selectedCollection = _masterCollection.filter(_isSelected);

        // Setup result collection.
        _outputCollection = null;

        // Setup filters.
        _isVisible = Objects.firstNonNull(visibleFilter, new StringFilter<E>());
        _displayTransform = Objects.firstNonNull(displayTransform, Functions.<E>identity());

        setTitle("Select Items");

        // Setup event bus to notify us of changes if necessary.
        if(items instanceof EventBusUser || _isVisible instanceof EventBusUser)
        {
            _internalBus = new EventBus();
            _internalBus.register(this);
            if(items instanceof EventBusUser)
            {
                ((EventBusUser)items).setEventBus(_internalBus);
            }
            if(_isVisible instanceof EventBusUser)
            {
                ((EventBusUser)_isVisible).setEventBus(_internalBus);
            }
        }
        else
        {
            _internalBus = null;
        }

        // Setup Map.
        _usedMap = Maps.newHashMapWithExpectedSize(_masterCollection.size());
        for(final E element: _masterCollection)
        {
            _usedMap.put(element, false);
        }

        _unusedLabel = new JLabel("Unselected");
        _usedLabel = new JLabel("Selected");

        // Setup JLists.
        _unusedView = _unselectedCollection.split(_isVisible);
        _usedView = _selectedCollection.split(_isVisible);
        _unusedList = new GenericJList(_unusedView.transform(_displayTransform));
        _usedList = new GenericJList(_usedView.transform(_displayTransform));

        initialize();
        initializeListeners();
        initializeFilter();

        if(items instanceof List)
        {
            // TODO turn on move up/down buttons here
        }
    }

    private void initialize()
    {

        this.setLayout(new GridBagLayout());

        final GridBagConstraints cons = new GridBagConstraints();
        cons.gridx = 0;
        cons.gridy = 0;
        cons.weightx = 1;
        this.add(_unusedLabel, cons);

        cons.gridx = 2;
        this.add(_usedLabel, cons);

        cons.gridx = 0;
        cons.gridy = 1;
        cons.weightx = 1;
        cons.weighty = 1;
        cons.fill = GridBagConstraints.BOTH;
        final JScrollPane unusedPane = new JScrollPane(_unusedList);
        this.add(unusedPane, cons);

        _moveButton = new JButton();
        _moveButton.setIcon(_rightIcon);
        _moveButton.setEnabled(false);
        cons.gridx = 1;
        cons.weightx = 0;
        cons.fill = GridBagConstraints.NONE;
        cons.insets = new Insets(0, 15, 0, 15);
        this.add(_moveButton, cons);

        cons.gridx = 2;
        cons.weightx = 1;
        cons.fill = GridBagConstraints.BOTH;
        cons.insets = new Insets(0, 0, 0, 0);
        final JScrollPane usedPane = new JScrollPane(_usedList);
        this.add(usedPane, cons);

        _usedList.setAutoscrolls(true);
        _unusedList.setAutoscrolls(true);

        // Find longest string.
        String s = "";
        for(final E e: _masterCollection)
        {
            @SuppressWarnings("unchecked")
            final String thisString = _displayTransform.apply(e).toString();
            if(thisString.length() > s.length())
            {
                s = thisString;
            }
        }
        _usedList.setPrototypeCellValue(s);
        _unusedList.setPrototypeCellValue(s);
        _usedList.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        _unusedList.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        usedPane.setMinimumSize(new Dimension(_usedList.getFixedCellWidth() + 20,
                                              _usedList.getPreferredSize().height + 20));
        unusedPane.setMinimumSize(usedPane.getMinimumSize());
        this.validate();

        this.setMinimumSize(new Dimension(usedPane.getMinimumSize().width * 3, this.getPreferredSize().height));
    }

    private void initializeListeners()
    {
        _activeList = null;

        _unusedList.addFocusListener(new FocusAdapter()
        {
            @Override
            public void focusGained(final FocusEvent e)
            {
                _activeList = false;
                _moveButton.setIcon(_rightIcon);
                _usedList.setSelectedIndices(new int[0]);
            }
        });
        _usedList.addFocusListener(new FocusAdapter()
        {
            @Override
            public void focusGained(final FocusEvent e)
            {
                _activeList = true;
                _moveButton.setIcon(_leftIcon);
                _unusedList.setSelectedIndices(new int[0]);
            }
        });

        _unusedList.addMouseListener(new MouseAdapter()
        {
            @Override
            public void mouseClicked(final MouseEvent e)
            {
                if(e.getClickCount() % 2 == 0)
                {
                    moveItems();
                }
            }
        });
        _usedList.addMouseListener(new MouseAdapter()
        {
            @Override
            public void mouseClicked(final MouseEvent e)
            {
                _moveButton.setEnabled(_usedList.getSelectedIndices().length > 0);
                if(e.getClickCount() % 2 == 0)
                {
                    moveItems();
                }
            }
        });

        _unusedList.addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                _moveButton.setEnabled(true);
            }
        });
        _usedList.addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                _moveButton.setEnabled(true);
            }
        });

        _moveButton.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                moveItems();
            }
        });
    }

    @SuppressWarnings({"unchecked"})
    private void initializeFilter()
    {
        if(_isVisible instanceof Component)
        {
            final GridBagConstraints cons = new GridBagConstraints();
            cons.gridx = 0;
            cons.gridy = 2;
            cons.gridwidth = GridBagConstraints.REMAINDER;
            cons.gridheight = GridBagConstraints.REMAINDER;
            cons.weightx = 1;
            cons.fill = GridBagConstraints.BOTH;

            final JPanel filterPanel = new JPanel();
            filterPanel.setLayout(new BorderLayout());
            filterPanel.add((Component)_isVisible, BorderLayout.CENTER);

            this.add(filterPanel, cons);
        }
    }

    public void setSelectionTitles(final String available, final String selected)
    {
        _unusedLabel.setText(available);
        _usedLabel.setText(selected);
    }

    /**
     * Sets the currently selected items as all the items in {@code collection}. Furthermore, {@code collection} will be
     * updated as the user selects and unselect items.
     */
    public void setOutputCollection(final Collection<E> collection)
    {
        _outputCollection = collection;
        for(final E e: _usedMap.keySet())
        {
            if(_outputCollection == null)
            {
                _usedMap.put(e, false);
            }
            else
            {
                _usedMap.put(e, collection.contains(e));
            }
        }
        notifyDataChanged();
    }

    @SuppressWarnings("unchecked")
    private void moveItems()
    {
        if(_activeList == null)
        {
            return;
        }

        // Find out selected items.
        int[] indices;
        E[] selected;
        if(_activeList == true)
        {
            indices = _usedList.getSelectedIndices();
            selected = ((E[])new Object[indices.length]);
            for(int i = 0; i < indices.length; i++)
            {
                selected[i] = _usedView.getElementAt(indices[i]);
            }
            _usedList.setSelectedIndices(new int[0]);
        }
        else
        {
            indices = _unusedList.getSelectedIndices();
            selected = (E[])new Object[indices.length];
            for(int i = 0; i < indices.length; i++)
            {
                selected[i] = _unusedView.getElementAt(indices[i]);
            }
            _unusedList.setSelectedIndices(new int[0]);
        }

        // Move items.
        final Collection<E> toMove = Lists.newArrayList();
        for(int i = 0; i < selected.length; i++)
        {
            if(_usedMap.get(selected[i]).equals(_activeList))
            {
                _usedMap.put(selected[i], !_activeList);
                _masterCollection.notifyChanged(selected[i]);
                toMove.add(selected[i]);
            }
        }

        // Update output collection.
        if(_outputCollection != null)
        {
            if(_activeList)
            {
                _outputCollection.removeAll(toMove);
            }
            else
            {
                _outputCollection.addAll(toMove);
            }
        }

        _moveButton.setEnabled(false);
    }

    @Subscribe
    public void notifyDataChanged(final ObjectModifiedNotice e)
    {
        notifyDataChanged();
    }

    /**
     * Notifies this panel that the backing data was changed and it needs to redisplay the selections.
     */
    public void notifyDataChanged()
    {
        _masterCollection.notifyChanged();
    }

    public void setTitle(final String title)
    {
        this.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(1), title, null));
    }

    public Collection<E> getSelected()
    {
        return _selectedCollection;
    }

    @Override
    public void setEnabled(final boolean enabled)
    {
        super.setEnabled(enabled);
        _unusedList.setEnabled(enabled);
        _usedList.setEnabled(enabled);
    }

    public static void main(final String[] args)
    {
        final Set<String> set = Sets.newTreeSet();
        set.add("Hello");
        set.add("Hemoglobin");
        set.add("Helmet");
        set.add("1");
        set.add("2");
        set.add("3");
        set.add("13");
        @SuppressWarnings({"rawtypes", "unchecked"})
        final ItemSelectionPanel panel = new ItemSelectionPanel(set, new StringFilter(), null);

        final JFrame frame = new JFrame(ItemSelectionPanel.class.getCanonicalName());
        frame.setLayout(new BorderLayout());
        frame.add(panel, BorderLayout.CENTER);

        frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        frame.pack();
        frame.setVisible(true);
    }

    private static class StringFilter<E> extends JPanel implements EventBusUser, Predicate<E>
    {
        private EventBus _bus;

        private final JTextField _text;

        public StringFilter()
        {
            _text = new JTextField();
            _text.setText("");
            _text.getDocument().addDocumentListener(new DocumentListener()
            {
                @SuppressWarnings({"unchecked", "rawtypes"})
                @Override
                public void insertUpdate(final DocumentEvent e)
                {
                    _bus.post(new ObjectModifiedNotice(StringFilter.this));
                }

                @SuppressWarnings({"rawtypes", "unchecked"})
                @Override
                public void removeUpdate(final DocumentEvent e)
                {
                    _bus.post(new ObjectModifiedNotice(StringFilter.this));
                }

                @SuppressWarnings({"rawtypes", "unchecked"})
                @Override
                public void changedUpdate(final DocumentEvent e)
                {
                    _bus.post(new ObjectModifiedNotice(StringFilter.this));
                }
            });

            this.setLayout(new BorderLayout());
            this.add(_text);

            this.setBorder(HSwingFactory.createTitledBorder(new EtchedBorder(1), "Filter", null));
        }

        @Override
        public boolean apply(final E element)
        {
            return element.toString().contains(_text.getText());
        }

        @Override
        public void setEventBus(final EventBus bus)
        {
            _bus = bus;
        }

    }
}
