package ohd.hseb.hefs.utils.gui.components;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.JListUtilities;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;

import com.google.common.collect.Lists;

/**
 * This is a {@link JPanel} for listing any {@link Object}s and providing a mechanism to move them around. The list is
 * displayed in a {@link JList} within a {@link JScrollPane} and includes up and down arrows on the right side. When an
 * item is moved, an action is fired for all {@link MoveableJListPanelListener} listeners. Be sure to properly override
 * the {@link Object#toString()} methods since the return of that method is displayed in the GUI {@link JList}.
 * 
 * @author herrhd
 */
public class MoveableJListPanel extends JPanel implements ActionListener
{
    private static final long serialVersionUID = 1L;

    private final JList _list;
    private final JScrollPane _listSP;
    private final JButton _upButton = HSwingFactory.createJButtonWithIcon("hefsIcons/row_up.png",
                                                                          "Move Item(s) Up",
                                                                          this);
    private final JButton _downButton = HSwingFactory.createJButtonWithIcon("hefsIcons/row_down.png",
                                                                            "Move Item(s) Down",
                                                                            this);
    private final JButton _topButton = HSwingFactory.createJButtonWithIcon("hefsIcons/row_top.png",
                                                                           "Move Item(s) to Top",
                                                                           this);
    private final JButton _bottomButton = HSwingFactory.createJButtonWithIcon("hefsIcons/row_bottom.png",
                                                                              "Move Item(s) to Bottom",
                                                                              this);
    private final List<Object> _managedList;

    private final List<MoveableJListPanelListener> _listeners = new ArrayList<MoveableJListPanelListener>();

    /**
     * @param managedList A list of anything. Its {@link Object#toString()} returns will be displayed in the list. This
     *            list is not modified directly, but is copied and its copy is modified. Be sure to call
     *            {@link #updateJListToReflectManagedList()} if the contents of the list are changed externally and must
     *            be reflected herein.
     */
    @SuppressWarnings("unchecked")
    public MoveableJListPanel(final List managedList)
    {
        _upButton.setPreferredSize(new Dimension(20, 20));
        _downButton.setPreferredSize(new Dimension(20, 20));
        _topButton.setPreferredSize(new Dimension(20, 20));
        _bottomButton.setPreferredSize(new Dimension(20, 20));

        _managedList = managedList;

        _list = new JList()
        {
            private static final long serialVersionUID = 1L;

            @Override
            public String getToolTipText(final MouseEvent e)
            {
                final int index = locationToIndex(e.getPoint());
                if(-1 < index)
                {
                    final String item = getModel().getElementAt(index).toString();
                    return item;
                }
                else
                {
                    return null;
                }
            }
        };
        _list.setVisibleRowCount(5);
        _listSP = new JScrollPane(_list);
        _listSP.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        _listSP.setPreferredSize(new Dimension(100, 100));
        updateJListToReflectManagedList();

        this.setLayout(new GridBagLayout());

        GridBagConstraints constraints;
        Insets theinsets = new Insets(0, 0, 0, 0);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          4,
                                                          1,
                                                          1,
                                                          GridBagConstraints.CENTER,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        this.add(_listSP, constraints);
        theinsets = new Insets(1, 1, 1, 1);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          0,
                                                          1,
                                                          1,
                                                          0,
                                                          1,
                                                          GridBagConstraints.SOUTH,
                                                          GridBagConstraints.NONE,
                                                          theinsets,
                                                          0,
                                                          0);
        this.add(_topButton, constraints);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.CENTER,
                                                          GridBagConstraints.NONE,
                                                          theinsets,
                                                          0,
                                                          0);
        this.add(_upButton, constraints);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          2,
                                                          1,
                                                          1,
                                                          0,
                                                          0,
                                                          GridBagConstraints.CENTER,
                                                          GridBagConstraints.NONE,
                                                          theinsets,
                                                          0,
                                                          0);
        this.add(_downButton, constraints);
        constraints = SwingTools.returnGridBagConstraints(1,
                                                          3,
                                                          1,
                                                          1,
                                                          0,
                                                          1,
                                                          GridBagConstraints.NORTH,
                                                          GridBagConstraints.NONE,
                                                          theinsets,
                                                          0,
                                                          0);
        this.add(_bottomButton, constraints);

    }

    public void removeSelectedItems()
    {
        final int[] indices = _list.getSelectedIndices();
        int i;
        for(i = indices.length - 1; i >= 0; i--)
        {
            _managedList.remove(indices[i]);
        }
        this.updateJListToReflectManagedList();
    }

    public void addItem(final String item)
    {
        this._managedList.add(item);
        this.updateJListToReflectManagedList();
        JListUtilities.scrollToRow(_list, _managedList.size() - 1);
    }

    public void addItem(final int index, final String item)
    {
        this._managedList.add(index, item);
        this.updateJListToReflectManagedList();
        JListUtilities.scrollToRow(_list, index);
    }

    public void addListener(final MoveableJListPanelListener listener)
    {
        this._listeners.add(listener);
    }

    public void removeListener(final MoveableJListPanelListener listener)
    {
        this._listeners.remove(listener);
    }

    private void fireOrderChanged()
    {
        for(int i = 0; i < _listeners.size(); i++)
        {
            this._listeners.get(i).listOrderChanged();
        }
    }

    /**
     * Once done modifying the managed list, call this method to update the interface to display the new list.
     */
    public void updateJListToReflectManagedList()
    {
        _list.setListData(_managedList.toArray());
        _listSP.setViewportView(_list);
    }

    public void clear()
    {
        this._managedList.clear();
        this.updateJListToReflectManagedList();
    }

    public int getSelectedIndex()
    {
        return _list.getSelectedIndex();
    }

    /**
     * @return The managed list of strings in the user specified order.
     */
    public List<Object> getManagedList()
    {
        return this._managedList;
    }

    public Object getItem(final int index)
    {
        return this._managedList.get(index);
    }

    public int getNumberOfItems()
    {
        return this._managedList.size();
    }

    @Override
    public void setEnabled(final boolean b)
    {
        super.setEnabled(b);
        this._list.setEnabled(b);
        this._upButton.setEnabled(b);
        this._downButton.setEnabled(b);
    }

    /**
     * Calls {@link #fireOrderChanged()} if the selectedIndices length is positive. Then calls
     * {@link #updateJListToReflectManagedList()}, sets the selected indices in {@link #_list}, and scrolls to the
     * specified row index called {@link JListUtilities#scrollToRow(JList, int)}.
     * 
     * @param selectedIndices
     */
    private void updateListAndReselect(final int[] selectedIndices, final int scrollToIndex)
    {
        if(selectedIndices.length > 0)
        {
            fireOrderChanged();
        }

        updateJListToReflectManagedList();
        _list.setSelectedIndices(selectedIndices);
        JListUtilities.scrollToRow(_list, scrollToIndex);
    }

    /**
     * Move the selected items up one row.
     */
    private void moveItemsUp()
    {
        final int[] selectedIndices = _list.getSelectedIndices();
        Arrays.sort(selectedIndices);

        for(int i = 0; i < selectedIndices.length; i++)
        {
            //If the previous selected index is the same as this one, minus 1, then we cannot move
            //this one.  This should only happen when all the previous rows are selected (otherwise,
            //the previous one would have been moved and the equality would be false).
            if((i > 0) && (selectedIndices[i - 1] == selectedIndices[i] - 1))
            {
                continue;
            }

            //This will move the row and make the selected index change to still point to the same row.
            if(selectedIndices[i] > 0)
            {
                final Object itemToMove = _managedList.get(selectedIndices[i]);
                _managedList.remove(selectedIndices[i]);
                //Move the selected indices with the list item.
                _managedList.add(selectedIndices[i] - 1, itemToMove);
                selectedIndices[i] = selectedIndices[i] - 1;
            }
        }
        updateListAndReselect(selectedIndices, selectedIndices[0]);
    }

    /**
     * Move the selected items to the top.
     */
    private void moveItemsToTop()
    {
        final int[] selectedIndices = _list.getSelectedIndices();
        Arrays.sort(selectedIndices);

        final List<Object> selectedValuesInOrder = Lists.newArrayList();
        for(int i = 0; i < selectedIndices.length; i++)
        {
            selectedValuesInOrder.add(_managedList.get(selectedIndices[i]));
            selectedIndices[i] = selectedValuesInOrder.size() - 1; //Sets the new index appropriately
        }

        _managedList.removeAll(selectedValuesInOrder);
        _managedList.addAll(0, selectedValuesInOrder);
        updateListAndReselect(selectedIndices, selectedIndices[0]);
    }

    /**
     * Moves the selected items down one row.
     */
    private void moveItemsDown()
    {
        final int[] selectedIndices = _list.getSelectedIndices();
        Arrays.sort(selectedIndices);

        for(int i = selectedIndices.length - 1; i >= 0; i--)
        {
            //If the previous selected index is the same as this 1, minus 1, then we cannot move
            //this one.  This should only happen when all the previous rows are selected (otherwise,
            //the previous one would have been moved and the equality would be false).
            if((i < selectedIndices.length - 1) && (selectedIndices[i + 1] == selectedIndices[i] + 1))
            {
                continue;
            }

            //This will move the row and make the selected index change to still point to the same row.
            if(selectedIndices[i] < this.getNumberOfItems() - 1)
            {
                final Object itemToMove = _managedList.get(selectedIndices[i]);
                _managedList.remove(selectedIndices[i]);
                //Move the selected indices with the list item.
                _managedList.add(selectedIndices[i] + 1, itemToMove);
                selectedIndices[i] = selectedIndices[i] + 1;
            }
        }
        updateListAndReselect(selectedIndices, selectedIndices[selectedIndices.length - 1]);
    }

    /**
     * Moves the selected items to the bottom.
     */
    private void moveItemsToBottom()
    {
        final int[] selectedIndices = _list.getSelectedIndices();
        Arrays.sort(selectedIndices);

        final List<Object> selectedValuesInOrder = Lists.newArrayList();
        for(int i = 0; i < selectedIndices.length; i++)
        {
            selectedValuesInOrder.add(_managedList.get(selectedIndices[i]));
            selectedIndices[i] = _managedList.size() - selectedIndices.length + (selectedValuesInOrder.size() - 1); //Sets the new index appropriately
        }

        _managedList.removeAll(selectedValuesInOrder);
        _managedList.addAll(selectedValuesInOrder);
        updateListAndReselect(selectedIndices, selectedIndices[selectedIndices.length - 1]);
    }

    @Override
    public void actionPerformed(final ActionEvent e)
    {
        if(e.getSource() == _upButton)
        {
            moveItemsUp();
        }
        if(e.getSource() == _downButton)
        {
            moveItemsDown();
        }
        if(e.getSource() == _topButton)
        {
            moveItemsToTop();
        }
        if(e.getSource() == _bottomButton)
        {
            moveItemsToBottom();
        }
    }

}
