package ohd.hseb.hefs.utils.gui.help;

import java.awt.Component;
import java.awt.Container;
import java.awt.MouseInfo;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.SwingUtilities;

import ohd.hseb.hefs.utils.effect.Effect;

/**
 * Passes events through until it gets a {@link MouseListener#mouseClicked(MouseEvent) mouseClicked} action, and then
 * performs a given effect based on the button clicked and makes itself invisible. A flag determines if it will
 * redispatch all other events to the component behind the glass.<br>
 * <br>
 * This class will replace the glass pane within a given {@link JRootPane}.
 * 
 * @author alexander.garbarino
 */
@SuppressWarnings("serial")
public class ClickComponentGlassPane extends JPanel implements MouseListener, MouseMotionListener
{
    private final JRootPane _parent;
    private final Effect<Component> _leftClickEffect;
    private final Effect<Component> _otherClickEffect;
    private final HighlightComponentEffect _highlightEffect;
    private final boolean _redispatchEvents;

    /**
     * An initial highlighting effect is created at the end of this constructor. All future highlights occur when the
     * mouse moves.
     * 
     * @param parent The {@link JRootPane} to which this glass pane will be attached.
     * @param leftClickEffect The effect to show when a left-click occurs. A click will never be redispatched. The
     *            effect will be handed the {@link Component} clicked, not necessarily the highlighted component.
     * @param otherClickEffect The effect to show when a right-click occurs. A click will never be redispatched. The
     *            effect will be handed the {@link Component} clicked, not necessarily the highlighted component.
     * @param highlightEffect The effect to show when this mouse hovers over a component. See
     *            {@link HighlightComponentEffect}.
     * @param redispatchEvents If true, all mouse events other than a click will be redispatched to the appropriate
     *            component behind the glass.
     */
    public ClickComponentGlassPane(final JRootPane parent,
                                   final Effect<Component> leftClickEffect,
                                   final Effect<Component> otherClickEffect,
                                   final HighlightComponentEffect highlightEffect,
                                   final boolean redispatchEvents)
    {
        _leftClickEffect = leftClickEffect;
        _otherClickEffect = otherClickEffect;
        _highlightEffect = highlightEffect;
        _redispatchEvents = redispatchEvents;

        _parent = parent;
        parent.setGlassPane(this);
        addMouseListener(this);
        addMouseMotionListener(this);

        setOpaque(false);
        setVisible(true);

        //Highlight the current comp
        final Point pt = new Point(MouseInfo.getPointerInfo().getLocation());
        SwingUtilities.convertPointFromScreen(pt, _parent.getContentPane());
        highlightComponent(getTargetComponent(pt));
    }

    private Component getTargetComponent(final Point mousePointRelativeToThis)
    {
        final Container contentPane = _parent.getContentPane();
        if(contentPane == null)
        {
            throw new IllegalStateException("The parent root pane does not have its content pane set; call setContentPane!");
        }
        final Point contentPoint = SwingUtilities.convertPoint(this, mousePointRelativeToThis, contentPane);
        return SwingUtilities.getDeepestComponentAt(contentPane, contentPoint.x, contentPoint.y);
    }

    @Override
    public void mouseClicked(final MouseEvent e)
    {
        removeMouseListener(this);
        removeMouseMotionListener(this);
        setVisible(false);

        if(SwingUtilities.isLeftMouseButton(e))
        {
            _leftClickEffect.perform(getTargetComponent(e.getPoint()));
        }
        else
        {
            _otherClickEffect.perform(getTargetComponent(e.getPoint()));
        }

        repaint(); //Clear everything.
    }

    private void redispatch(final MouseEvent e)
    {
        if(_redispatchEvents)
        {
            final Component targetComponent = getTargetComponent(e.getPoint());
            if(targetComponent == null)
            {
                return;
            }

            final Point targetPoint = SwingUtilities.convertPoint(this, e.getPoint(), targetComponent);
            targetComponent.dispatchEvent(new MouseEvent(targetComponent,
                                                         e.getID(),
                                                         e.getWhen(),
                                                         e.getModifiersEx(),
                                                         targetPoint.x,
                                                         targetPoint.y,
                                                         e.getClickCount(),
                                                         e.isPopupTrigger()));
        }
    }

    private void highlightComponent(final Component c)
    {
        if(_highlightEffect != null)
        {
            _highlightEffect.setRootPane(_parent);
            _highlightEffect.perform(c);
        }
    }

    @Override
    public void mouseDragged(final MouseEvent e)
    {
        redispatch(e);
    }

    @Override
    public void mouseMoved(final MouseEvent e)
    {
        highlightComponent(getTargetComponent(e.getPoint()));
        redispatch(e);
    }

    @Override
    public void mousePressed(final MouseEvent e)
    {
        redispatch(e);
    }

    @Override
    public void mouseReleased(final MouseEvent e)
    {
        redispatch(e);
    }

    @Override
    public void mouseEntered(final MouseEvent e)
    {
        _highlightEffect.clearMemory();
        highlightComponent(getTargetComponent(e.getPoint()));
        redispatch(e);
    }

    @Override
    public void mouseExited(final MouseEvent e)
    {
        repaint();
        redispatch(e);
    }

}
