package ohd.hseb.hefs.utils.gui.help;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.io.File;

import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JRootPane;

import ohd.hseb.hefs.utils.effect.Effect;
import ohd.hseb.hefs.utils.gui.tools.CursorTools;
import ohd.hseb.hefs.utils.gui.tools.IconAction;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.IconTools;

/**
 * The {@link ClickHelpManager} requires a target component. When the {@link #_startHelpAction} is triggered, it will
 * search for the first {@link JRootPane} parent of target component (including the component, itself). It then uses
 * that as the basis for a {@link ClickComponentGlassPane}.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class ClickHelpManager extends HelpManager
{

    /**
     * Cursor displays the same images as the help button, which is kinda cool looking.
     */
    private static final Cursor HELP_CURSOR = CursorTools.createCursorFromImage(IconTools.getHSEBIcon("helpIcon")
                                                                                         .getImage(), "helpIcon");

    private final Component _targetComponent;

    /**
     * Action associated with the help. When triggered, it sets up a {@link ClickComponentGlassPane}. The left click
     * effect calls {@link #getDisplayHelpEffect()} in the super class and follows it with an effect that turns off the
     * help cursor. Other clicks turn off the cursor without doing anything.
     */
    public final Action _startHelpAction = new IconAction("helpIcon")
    {
        @Override
        public synchronized void actionPerformed(final ActionEvent e)
        {
            final JRootPane rootPane = SwingTools.getParentOfClassIncludingSelf(_targetComponent, JRootPane.class);

            //Add to the help effect so that the cursor reverts after helping.
            final Effect<Component> leftClickEffect = getDisplayHelpEffect().then(new Runnable()
            {
                @Override
                public void run()
                {
                    rootPane.getGlassPane().setCursor(Cursor.getDefaultCursor());
                }
            });

            //For right click, just make the cursor reset
            final Effect<Component> otherClickEffect = new Effect<Component>()
            {
                @Override
                public void perform(final Component input)
                {
                    rootPane.getGlassPane().setCursor(Cursor.getDefaultCursor());
                }
            };

            //Clear the highlighted component memory, setup the glass pane, and and start the help process.
            ((HighlightHelpComponentEffect)getHighlightComponentEffect()).clearMemory();
            new ClickComponentGlassPane(rootPane,
                                        leftClickEffect,
                                        otherClickEffect,
                                        getHighlightComponentEffect(),
                                        false);
            rootPane.getGlassPane().setCursor(HELP_CURSOR);
        }
    };

    public ClickHelpManager(final File directory, final Component component)
    {
        super(directory);
        _targetComponent = component;
    }

    /**
     * Short cut to creating a {@link JButton} for the {@link ClickHelpManager} using the {@link #_startHelpAction} as
     * the basis for the button.
     * 
     * @param helpDirectory Directory containing help files.
     * @param helpComponent The {@link ClickHelpManager} finds the first parent of this component that is a
     *            {@link JRootPane} and uses it to track help actions.
     * @return A {@link JButton} that can be added and uses the standard help icon. If the help directory is not found,
     *         this returns {@link #createDisabledHelpAction(String)}.
     */
    public static JButton createHelpButton(final File helpDirectory, final Component helpComponent)
    {
        if(helpDirectory.exists())
        {
            return new JButton(new ClickHelpManager(helpDirectory, helpComponent)._startHelpAction);
        }
        return createDisabledHelpAction("Unabled to find directory containing help file: "
            + helpDirectory.getAbsolutePath());
    }

    /**
     * @return A disabled {@link JButton} with the specified tool tip. This should be used when the help button is not
     *         available due to an unfound help directory.
     */
    public static JButton createDisabledHelpAction(final String toolTip)
    {
        final JButton b = new JButton(IconTools.getHSEBIcon("helpIcon"));
        b.setToolTipText(toolTip);
        b.setEnabled(false);
        return b;
    }
}
