package ohd.hseb.hefs.utils.gui.help;

import java.awt.Component;
import java.awt.Desktop;
import java.io.File;
import java.net.URL;
import java.util.List;

import ohd.hseb.hefs.utils.effect.Effect;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.PDFTools;
import edu.stanford.ejalbert.BrowserLauncher;

/**
 * Superclass of {@link ClickHelpManager} (button-action base help manager) and {@link CurrentFocusHelpManager} (F1-key
 * based help manager).<br>
 * <br>
 * To use this class, you need to construct a HelpManager and provide it with a base directory for help files, to be
 * used if a help file is not in a jar file (pdfs can be in a jar file) and is not a complete URL. Then call the
 * {@link Effect} returned by {@link #getDisplayHelpEffect()} when help is to be displayed. Given a {@link Component}
 * for which help is to be displayed, the effect calls {@link #displayHelpFile(String)} for the help file/url name
 * associated with the component. The help file/url name is specified by the {@link HelpFile} annotation associated with
 * the component itself or its nearest ancestor that defines the annotation. See the annotation {@link HelpFile} for
 * information on how to specify it. This constructs the appropriate URL to open the help file and either opens it in a
 * browser or jpedal wrapping {@link PDFViewerWrapper} for pdf files.<br>
 * <br>
 * 
 * @author hank.herr
 */
public abstract class HelpManager
{
    public final static String PDF_HELP_PREFIX = "PDFHELP:";

    /**
     * If null, this is not used.
     */
    private final File _directory;

    /**
     * Effect will display a help file specified by a component returned by getHelpFile.
     */
    private final Effect<Component> _displayHelpEffect = new Effect<Component>()
    {
        @Override
        public void perform(final Component input)
        {
            displayHelpFile(getHelpFileName(input));
        }
    };

    /**
     * Effect allows for highlighting the help component relative to a root pane.
     * 
     * @author hank.herr
     */
    protected class HighlightHelpComponentEffect extends HighlightComponentEffect
    {
        @Override
        protected Component getHighlightComponent(final Component component)
        {
            return getHelpComponent(component);
        }
    }

    private final HighlightComponentEffect _highlightComponentEffect = new HighlightHelpComponentEffect();

    protected Effect<Component> getDisplayHelpEffect()
    {
        return _displayHelpEffect;
    }

    protected HighlightComponentEffect getHighlightComponentEffect()
    {
        return _highlightComponentEffect;
    }

    /**
     * @param directory The base directory under which all help files are placed. Can be null if all file names are
     *            specified by absolute, full paths or URLs are used.
     */
    protected HelpManager(final File directory)
    {
        _directory = directory;
    }

    /**
     * Gets the help file specified by the given component's {@link HelpFile} tag. If the component does not have one,
     * the nearest ancestor (regarding components, not classes) with {@link HelpFile} is used instead.
     * 
     * @param component the component to retrieve the help file for
     * @return the specified help file name, or null if none was specified
     */
    public String getHelpFileName(final Component component)
    {
        final Component helpComponent = getHelpComponent(component);
        if(helpComponent == null)
        {
            return null;
        }
        return helpComponent.getClass().getAnnotation(HelpFile.class).value();
    }

    /**
     * Returns the first parent above component which is marked with a {@link HelpFile} annotation.
     * 
     * @param component
     * @return
     */
    private Component getHelpComponent(final Component component)
    {
        return SwingTools.getParentWithAnnotation(component, HelpFile.class);
    }

//XXX Lobo:
//    private static JFrame _helpFrame = null;

    public void displayHelpFile(String fileName)
    {
        if(!Desktop.isDesktopSupported())
        {
            return;
        }

        //XXX Using Lobo:
//        try
//        {
//            // This step is necessary for extensions to work:
////            PlatformInit.getInstance().init(false, false);
//
//            if(_helpFrame == null)
//            {
//                PlatformInit.getInstance().initLogging(false);
//                PlatformInit.getInstance().initProtocols();
//                PlatformInit.getInstance().initExtensions();
//
//                final BrowserPanel bp = new BrowserPanel();
//
//                _helpFrame = new JFrame("Help Dialog");
//                _helpFrame.setContentPane(bp);
//                _helpFrame.setSize(800, 600);
//                _helpFrame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
//            }
//
//            ((BrowserPanel)_helpFrame.getContentPane()).navigate(new URL("file:" + file.getAbsolutePath()).toString());
//            _helpFrame.setVisible(true);
//        }
//        catch(final Exception e)
//        {
//            e.printStackTrace();
//        }

        //The file name for PDF help must start with the prefix.  Optionally, it can end with a page number, #<##>
        if(fileName.startsWith(PDF_HELP_PREFIX))
        {
            //Extract the file name
            fileName = fileName.substring(PDF_HELP_PREFIX.length(), fileName.length());

            int pageNumber = 1;
            String bookmark = null;

            //Get item at the end after the #.
            final int indexOfPound = fileName.lastIndexOf("#");
            if(indexOfPound >= 0)
            {
                final String poundStr = fileName.substring(indexOfPound + 1, fileName.length());
                fileName = fileName.substring(0, indexOfPound);

                try
                {
                    pageNumber = Integer.parseInt(poundStr);
                }
                catch(final Exception e)
                {
                    bookmark = poundStr;
                }
            }

            //Append the directory to the file name if it does not start with http, jar:, or file:, each of which imply a URL.
            if(!fileName.startsWith("http") && !fileName.startsWith("jar:") && !fileName.startsWith("file:"))
            {
                fileName = FileTools.newFile(_directory, fileName).getAbsolutePath();
            }

            //If a bookmark is given...
            if(bookmark != null)
            {
                PDFTools.viewPDF(fileName, bookmark);
            }
            //Otherwise open with the page number...
            else
            {
                PDFTools.viewPDF(fileName, pageNumber);
            }
        }

        //In all other cases, HTML is assumed and the file must be on the file system under _directory.
        else
        {
            try
            {
                final BrowserLauncher launcher = new BrowserLauncher();
                final List browserList = launcher.getBrowserList();

                //If the fileName starts with http: or file:, then it is assumed to specify a complete URL.  Just open it...
                if((fileName.startsWith("http:")) || (fileName.startsWith("file:")))
                {
                    launcher.openURLinBrowser(browserList, fileName);
                }
                //Otherwise, it specifies a file on the file system under _directory if not null...
                else
                {
                    File helpFile = new File(fileName);
                    if(_directory != null)
                    {
                        helpFile = FileTools.newFile(_directory, fileName);
                    }
                    launcher.openURLinBrowser(browserList, new URL("file:" + helpFile.getAbsolutePath()).toString());
                }

//OLD for opening browser, doesn't work all the time:            Desktop.getDesktop().browse(file.toURI());
            }
            catch(final Exception e)
            {
                e.printStackTrace();
            }
        }
    }

//    public static void main(final String[] args)
//    {
//////        final PDFHelpDialog demo = new PDFHelpDialog("Help Viewer: ",
//////                                               "/Users/hankherr/workspace/HEFS -- hefsplugins/MEFP_Release_0.3.1_Users_Manual_2.pdf",
//////                                               20);
////
//        final JFrame frame = new JFrame();
//        final Viewer viewer = new Viewer(frame, "jar:/help/pdfHelpProperties.txt");
//        viewer.setupViewer();
//        viewer.openDefaultFileAtPage("jar:/mefppe/help/MEFP_Release_0.3.1_Users_Manual.pdf", 20);
//    }
}
