package ohd.hseb.hefs.utils.gui.help;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Stroke;

import javax.swing.JRootPane;
import javax.swing.SwingUtilities;

import ohd.hseb.hefs.utils.effect.Effect;

/**
 * Superclass of highlight effects for use in {@link ClickComponentGlassPane}. The method of that must be overridden is
 * the one that identifies the component to highlight, {@link #getHighlightComponent(Component)}. The highlight
 * mechanism is fixed to a faded red outline of the highlighted component.
 * 
 * @author hankherr
 */
public abstract class HighlightComponentEffect extends Effect<Component>
{

    private Component _previousComponent = null;
    private JRootPane _rootPane;

    public void setRootPane(final JRootPane rootPane)
    {
        _rootPane = rootPane;
    }

    public void clearMemory()
    {
        _previousComponent = null;
    }

    @Override
    public void perform(final Component input)
    {
        final Component helpComponent = getHighlightComponent(input);
        if(helpComponent != _previousComponent)
        {
            _rootPane.getContentPane().repaint();
            SwingUtilities.invokeLater(new Runnable()
            {
                @Override
                public void run()
                {
                    highlight(helpComponent);
                }
            });
            _previousComponent = helpComponent;
        }
    }

    private void highlight(final Component helpComponent)
    {
        final Rectangle r = SwingUtilities.convertRectangle(helpComponent.getParent(),
                                                            helpComponent.getBounds(),
                                                            _rootPane.getContentPane());

        final Color c = _rootPane.getGlassPane().getGraphics().getColor();
        final Graphics2D g = (Graphics2D)_rootPane.getGlassPane().getGraphics();
        g.setColor(new Color(255, 0, 0, 64));
        final Stroke oldStroke = g.getStroke();
        g.setStroke(new BasicStroke(10));
        g.drawRect(r.x, r.y, r.width, r.height);
        g.setColor(c);
        g.setStroke(oldStroke);
    }

    /**
     * The method to override.
     * 
     * @param component The component for which to acquire the component to highlight.
     * @return The {@link Component} to highlight, determined by the subclass however it desires.
     */
    protected abstract Component getHighlightComponent(final Component component);
}
