package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.Rectangle;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.ToolTipManager;
import javax.swing.table.TableModel;

import ohd.hseb.charter.panel.notices.ChartEngineTableCellSelectedNotice;
import ohd.hseb.charter.panel.notices.ChartTableVisibilityChangedNotice;
import ohd.hseb.charter.panel.notices.GeneralTableCellSelectedNotice;
import ohd.hseb.charter.panel.notices.TableCellControlClickedNotice;
import ohd.hseb.hefs.utils.notify.Notice;

import org.jdesktop.swingx.rollover.TableRolloverProducer;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * Subclass of {@link GenericTable} that allows for row, column and cell selection. It forces painting updates to rows
 * and columns of selected cells. It also posts a {@link GeneralTableCellSelectedNotice} event to its
 * {@link #_centralBus} whenever a cell is selected. A {@link TableCellControlClickedNotice} is posted whenever a cell
 * is double clicked, but nothing else is done (the cell will not be selected).<br>
 * <br>
 * This table takes over the mouse listening functionality of {@link JTable}. It will make it so that cell selection
 * only occurs upon left mouse button single clicks (not press, release, or drag). It also adds a
 * {@link DragScrollHandler} to allow for dragging to move the table within a scrolling viewer. Bottom line, if you need
 * to change how row/column/cell selection works, changes will need to be made in here.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class EventPostingCellSelectableTable extends GenericTable implements
ChartTableVisibilityChangedNotice.Subscriber
{

    /**
     * Central {@link EventBus} used for post {@link Notice}s and other events.
     */
    private EventBus _centralBus;

    public EventPostingCellSelectableTable(final TableModel model)
    {
        super(model);
        setRowSelectionAllowed(true);
        setColumnSelectionAllowed(true);
        setCellSelectionEnabled(true);

        //Remove existing.
        for(final MouseListener listener: getMouseListeners())
        {
            removeMouseListener(listener);
        }
        for(final MouseMotionListener listener: getMouseMotionListeners())
        {
            removeMouseMotionListener(listener);
        }

        //Re-register this component with ToolTipManager.  By removing the mouse motion listener, this became unregistered.
        ToolTipManager.sharedInstance().registerComponent(this);

        addMouseListener(new MouseAdapter()
        {

            @Override
            public void mouseClicked(final MouseEvent e)
            {
                //Only allow left mouse single clicks for selecting cells.  Right mouse button is reserved for other purposes; e.g.,
                //popup menus.  A double click will result in a notice being posted.
                if(SwingUtilities.isLeftMouseButton(e))
                {
                    //One click and  not a popup trigger...
                    if(!e.isPopupTrigger())
                    {
                        if(e.isControlDown())
                        {
                            final int viewRow = rowAtPoint(e.getPoint());
                            final int viewCol = columnAtPoint(e.getPoint());
                            final int modelRow = convertRowIndexToModel(viewRow);
                            final int modelCol = convertColumnIndexToModel(viewCol);

                            //A left button double click will result in a notice being posted.
                            if(SwingUtilities.isLeftMouseButton(e))
                            {
                                //Double clicked -- post a TableCellDoubleClickedNotice.
                                getCentralBus().post(new TableCellControlClickedNotice(this,
                                                                                       modelRow,
                                                                                       modelCol,
                                                                                       getModel().getValueAt(modelRow,
                                                                                                             modelCol)));
                            }
                        }
                        else
                        //includes shift-click or just click, as well.
                        {
                            final int viewRow = rowAtPoint(e.getPoint());
                            final int viewCol = columnAtPoint(e.getPoint());

                            //If cell is already selected, then clear selections.
                            if(isRowSelected(viewRow) && isColumnSelected(viewCol))
                            {
                                clearSelection();
                                reactToPossibleSelectedCell();
                                repaint();
                            }
                            else if((viewRow >= 0) && (viewCol >= 0))
                            {
                                setRowSelectionInterval(viewRow, viewRow);
                                setColumnSelectionInterval(viewCol, viewCol);
                                reactToPossibleSelectedCell();
                                repaint();
                            }
                        }
                    }
                }
            }
        });

        //Add the table dragging mouse listener.
        addMouseListener(new DragScrollHandler(this));

        //Add the rollover producer to allow for rollover highlighting, which is on for GenericTable.
        addMouseListener(new TableRolloverProducer());
        addMouseMotionListener(new TableRolloverProducer());
    }

    /**
     * Registers itself with the provided {@link EventBus}.
     */
    public void setCentralBus(final EventBus bus)
    {
        _centralBus = bus;
        _centralBus.register(this);
    }

    /**
     * Initializes the {@link #_centralBus} if null.
     */
    public EventBus getCentralBus()
    {
        if(_centralBus == null)
        {
            _centralBus = new EventBus();
            _centralBus.register(this);
        }
        return _centralBus;
    }

    /**
     * Posts a {@link GeneralTableCellSelectedNotice} event to the {@link #_centralBus}.
     */
    private void reactToPossibleSelectedCell()
    {
        //Make the row dirty.
        int selectedRow = getSelectedRow();
        if(selectedRow >= 0)
        {
            final Rectangle firstCell = getCellRect(selectedRow, 0, false);
            final Rectangle lastCell = getCellRect(selectedRow, getColumnCount(), false);
            final Rectangle rowDirtyRegion = firstCell.union(lastCell);
            repaint(rowDirtyRegion);

            //Convert for later use.
            selectedRow = convertRowIndexToModel(selectedRow);
        }

        //Make the column dirty.
        int selectedCol = getSelectedColumn();
        if(selectedCol >= 0)
        {
            final Rectangle firstCell = getCellRect(0, selectedCol, false);
            final Rectangle lastCell = getCellRect(getRowCount(), selectedCol, false);
            final Rectangle colDirtyRegion = firstCell.union(lastCell);
            repaint(colDirtyRegion);

            //Convert for later use.
            selectedCol = convertColumnIndexToModel(selectedCol);
        }

        //Post the notice for table cell selection.
        if(_centralBus != null)
        {
            if((selectedRow >= 0) && (selectedCol >= 0))
            {
                //Post the general notice.
                _centralBus.post(new GeneralTableCellSelectedNotice(this,
                                                                    selectedRow,
                                                                    selectedCol,
                                                                    getModel().getValueAt(selectedRow, selectedCol)));
            }
            else if(selectedRow >= 0)
            {
                //Post the general notice.
                _centralBus.post(new GeneralTableCellSelectedNotice(this, selectedRow, -1, null));
            }
            else if(selectedCol >= 0)
            {
                //Post the general notice.
                _centralBus.post(new GeneralTableCellSelectedNotice(this, -1, selectedCol, null));
            }
            else
            {
                _centralBus.post(new GeneralTableCellSelectedNotice(this));
            }
        }
    }

    @Override
    public void changeSelection(final int rowIndex, final int columnIndex, final boolean toggle, final boolean extend)
    {
        super.changeSelection(rowIndex, columnIndex, toggle, extend);
        reactToPossibleSelectedCell();
    }

    /**
     * Overridden to force a reaction to a possible selected cell whenever the model is changed (likely posting empty
     * events).
     */
    @Override
    public void setModel(final TableModel model)
    {
        super.setModel(model);
        reactToPossibleSelectedCell();
    }

    @Override
    @Subscribe
    public void reactToChartTableVisibilityChangedNotice(final ChartTableVisibilityChangedNotice evt)
    {
        if(!evt.isVisible())
        {
            _centralBus.post(new GeneralTableCellSelectedNotice(this));
            _centralBus.post(new ChartEngineTableCellSelectedNotice(this));
        }
        else
        {
            reactToPossibleSelectedCell();
        }
    }

}
