package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ColorHighlighter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.renderer.DefaultTableRenderer;
import org.jdesktop.swingx.table.ColumnControlButton;

import com.google.common.collect.Lists;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

import ohd.hseb.hefs.utils.gui.jtable.buttons.RefreshButton;
import ohd.hseb.hefs.utils.gui.jtable.buttons.SelectAllButton;
import ohd.hseb.hefs.utils.gui.jtable.buttons.SelectUnreadyRowsButton;
import ohd.hseb.hefs.utils.gui.jtable.buttons.UnselectAllButton;
import ohd.hseb.hefs.utils.gui.jtable.models.AddDeleteRowTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.ColumnHeaderRendererTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.DefaultEditorsAndRenderersOverrideTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.GetRowTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.MaxWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.MinWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.PreferredWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RefreshTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.SelectionMaintainingTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.TableAwareTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.WrapRendererTableModel;
import ohd.hseb.hefs.utils.gui.jtable.renderers.ColorByEditableTableCellRenderer;
import ohd.hseb.hefs.utils.gui.jtable.renderers.EmptyLabelOnNullOrNaNRenderer;
import ohd.hseb.hefs.utils.gui.jtable.renderers.JLabelTableCellRenderer;
import ohd.hseb.hefs.utils.gui.jtable.renderers.StringCellEditor;
import ohd.hseb.hefs.utils.gui.tools.JToolBarBuilder;
import ohd.hseb.hefs.utils.gui.tools.MultiItemButton;
import ohd.hseb.hefs.utils.status.StatusIndicator;
import ohd.hseb.hefs.utils.tools.IconTools;
import ohd.hseb.util.misc.HCalendar;

/**
 * Table which automatically detects and deals with several of the special TableModels in this package.<br>
 * <br>
 * RENDERERS: <br>
 * Object: {@link JTable#getDefaultRenderer(Class))}. <br>
 * Number: {@link JTable#getDefaultRenderer(Class))}. <br>
 * Boolean: {@link JTable#getDefaultRenderer(Class))}. <br>
 * Integer: see below. <br>
 * Double: see below. <br>
 * JLabel: {@link JLabelTableCellRenderer}. <br>
 * <br>
 * EDITORS: <br>
 * String: {@link StringCellEditor#StringCellEditor(int)} passing in a 1. <br>
 * Integer: {@link NumberCellEditor}. Renderer is created from {@link NumberCellEditor#constructCorrespondingRenderer()}
 * . <br>
 * Double: {@link NumberCellEditor} using format "##0.000####". Renderer is created from
 * {@link NumberCellEditor#constructCorrespondingRenderer()}. <br>
 * <br>
 * Subclass specific editors and renderers can be set by having the model implement
 * {@link DefaultEditorsAndRenderersOverrideTableModel} and implementing the method therein appropriately. That method
 * is called as part of {@link #resetRenderersAndEditors()} at the end after the standard defaults are set. It must be
 * done via this method because {@link #resetRenderersAndEditors()} may be called whenever
 * {@link #tableChanged(TableModelEvent)} is called, and it will undo any changes to the defaults if those defaults are
 * not captured within {@link DefaultEditorsAndRenderersOverrideTableModel}.<br>
 * <br>
 * Behavior based on the type of model provided is as follows:<br>
 * <br>
 * If assigned a (Preferred, Max, or Min)WidthsTableModel, it will assign its columns the given widths.<br>
 * If assigned a {@link AddDeleteRowTableModel}, it will initialize a mouse listener for it.<br>
 * If assigned a {@link TableAwareTableModel}, it will register itself with the model.<br>
 * If assigned a {@link SelectionMaintainingTableModel}, it will reselect selected rows whenever
 * {@link #tableChanged(TableModelEvent)} is called (the method {@link #tableChanged(TableModelEvent)} is overridden)
 * Rows are recorded when selections are changed in the selection model, but not if those changes are caused by
 * {@link #tableChanged(TableModelEvent)}.<br>
 * If assigned a {@link WrapRendererTableModel}, it will use its method to wrap default {@link JTable} cell renderers;
 * otherwise, the renderer {@link ColorByEditableTableCellRenderer} is used. <br>
 * <br>
 * An {@link EventBus} is provided with an instance of this via {@link #getRowSelectionBus()}. Whenever a list selection
 * is made, an event will be posted using an instance of {@link List} ({@link ArrayList} to be specific) whose size is
 * the number of rows selected. If assigned a {@link GetRowTableModel}, the elements of the list will be instances of
 * RowT, which must be the object type used within {@link GetRowTableModel}. Otherwise, elements will be null.
 * Additionally, a {@link GeneralRowSelectionNotice} will also be posted that provides the same list, but also an
 * array of the selected (model) row indices and an integer indicating how many rows the table has altogether (for
 * identifying if the last row is selected). Eventually, we may get rid of the posting of just the list.<br>
 * <br>
 * A default {@link JToolBarBuilder} is provided under the table which includes a {@link SelectAllButton},
 * {@link UnselectAllButton}, and a {@link SelectUnreadyRowsButton} (this button has other requirements of the table
 * model involving {@link StatusIndicator}). If the assigned a {@link RefreshTableModel}, a {@link RefreshButton} will
 * be added right justified.<br>
 * <br>
 * The table will, by default, be sortable; its {@link #setAutoCreateRowSorter(boolean)} method will be called with
 * true. The table will inherit its popup menu from its parent unless set manually via
 * {@link #setComponentPopupMenu(javax.swing.JPopupMenu)}.
 * 
 * @author alexander.garbarino
 * @author hank.herr
 * @parameter RowT The object that corresponds to rows of the table, which must match that used for
 *            {@link GetRowTableModel} and {@link SelectionMaintainingTableModel}.
 */

@SuppressWarnings("serial")
public class GenericTable<RowT> extends ToolTipJTable
{
    private boolean _addDeleteInitialized = false;

    /**
     * Bus has events posted to it whenever row selections change. Is null until {@link #initializeRowBus()} is called,
     * possibly via {@link #getRowSelectionBus()}; this allows for faster performance if a table does not require
     * listening for selected rows. If the table implements {@link GetRowTableModel}, then the items posted to the
     * {@link Collection} event will be the row objects, otherwise it will be a list of null entries as long as the
     * number of selected rows.
     */
    private EventBus _rowSelectionBus = null;

    /**
     * Responsible for posting events to {@link #_rowSelectionBus}. The listener is add to the table selection model
     * whenever it changes; hence the reason why it is an attribute.
     */
    private ListSelectionListener _rowBusPoster;

    /**
     * Records selected row objects if the model implements {@link SelectionMaintainingTableModel}.
     */
    private Object[] _selectedRowObjects;

    /**
     * If true, then the table is currently undergoing changes. One thing this means is that selections are not recorded
     * in {@link #_selectedRowObjects}.
     */
    private boolean _tableChanging = false;

    /**
     * Flag indicates if the renderers have been constructed once already.
     */
    private boolean _renderersConstructedFirstTime = false;

    /**
     * Stores listeners that are tailed for GenericTable.
     */
    private final List<GenericTableListener> _genericTableListeners = Lists.newArrayList();

    /**
     * Calls the empty constructor in {@link JTable} followed by {@link #initialize()}.
     */
    public GenericTable()
    {
        super();
        initialize();
    }

    /**
     * Calls the appropriate constructor for {@link JTable} passing in the model and then calls {@link #initialize()}.
     */
    public GenericTable(final TableModel model)
    {
        super(model);
        initialize();
    }

    /**
     * Overridden to make sure that if the old model is a {@link TableAwareTableModel}, this table is removed from it,
     * and if the new model is {@link TableAwareTableModel}, this table is added to it. Also calls
     * {@link #initializeAddDelete()} if the model implements {@link AddDeleteRowTableModel}.
     */
    @Override
    public void setModel(final TableModel model)
    {
        final TableModel oldModel = getModel();
        if(oldModel instanceof TableAwareTableModel)
        {
            ((TableAwareTableModel)oldModel).removeTable(this);
        }
        //This must be done before calling super setModel(...) because that action may cause the table 
        //to be needed by a renderer when initializing.
        if(model instanceof TableAwareTableModel)
        {
            ((TableAwareTableModel)model).addTable(this);
        }
        super.setModel(model);
        if(model instanceof AddDeleteRowTableModel)
        {
            initializeAddDelete();
        }
    }

    /**
     * May override as needed. If the return of {@link #getModel()} is an instance of {@link WrapRendererTableModel}, it
     * will call that method to wrap the default {@link JTable} renderer which is data type specific. Otherwise, and by
     * default, it uses a {@link ColorByEditableTableCellRenderer}. Whichever renderer is used, it then wraps that
     * within an {@link EmptyLabelOnNullOrNaNRenderer}, which for {@link JLabel} columns, if the label is null, makes
     * the cell empty.
     * 
     * @param renderer Default renderer used to draw cells.
     * @return A new renderer as described above.
     */
    protected TableCellRenderer wrapRenderer(TableCellRenderer renderer)
    {
        //Wrap replaces default color by editable renderer.
        if((getModel() instanceof WrapRendererTableModel)
            && !((WrapRendererTableModel)getModel()).applyAfterAllOtherRenderers())
        {
            renderer = ((WrapRendererTableModel)getModel()).wrapRenderer(renderer);
        }
        //Default action
        else
        {
            renderer = new ColorByEditableTableCellRenderer(renderer, new Color(237, 237, 237));
        }

        //Always do this!
        renderer = new EmptyLabelOnNullOrNaNRenderer(renderer);

        //Wrap is performed after all other defaults are applied.
        if((getModel() instanceof WrapRendererTableModel)
            && ((WrapRendererTableModel)getModel()).applyAfterAllOtherRenderers())
        {
            renderer = ((WrapRendererTableModel)getModel()).wrapRenderer(renderer);
        }
        return renderer;
    }

    /**
     * Default renderers and editors are set here.
     */
    private void initialize()
    {
        this.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        this.setToolTipText(null);

        resetRenderersAndEditors();
        setupHeaderRenderers();

        setInheritsPopupMenu(true);

        //Listener records selected rows as selected via the recordSelectedRowObjects method.  Rows are not
        //recorded if the model does not implement GetRowTableModel (or SelectionMaintainingTableModel).
        this.getSelectionModel().addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                if(!e.getValueIsAdjusting())
                {
                    recordSelectedRowObjects();
                }
            }
        });

        addHighlighter(new ColorHighlighter(HighlightPredicate.ROLLOVER_ROW, null, Color.BLUE));

        final ColumnControlButton controlButton = new ColumnControlButton(this,
                                                                          IconTools.getHSEBIcon("yellowGear14x14"));
        controlButton.setBackground(new Color(128, 128, 255));
        controlButton.setToolTipText("Open table control tools");
        setColumnControl(controlButton);
        setColumnControlVisible(false);
        setTerminateEditOnFocusLost(true);

        this.setAutoCreateRowSorter(true);
    }

    /**
     * Sets up the renderers and editors for this table. This must be called whenever the structure of the table changes
     * because of the wrap renderer, which may vary by applied {@link TableModel}. See
     * {@link #wrapRenderer(TableCellRenderer)}. Note that this should only be called AFTER the super {@link JXTable}
     * constructor is called. If called during construction (i.e. see {@link #setModel(TableModel)} override and check
     * the if clause), a null pointer exception will occur when getting the default renderers.
     */
    private void resetRenderersAndEditors()
    {
        //Clear default renderers to clear the memory.  This prevents the renderers from using old renderers
        //with associated old table models.  Without this, renderers may wrap old wrapped renderers, thereby
        //remembering old table model.  
        this.setDefaultRenderer(Object.class, null);
        this.setDefaultRenderer(Number.class, null);
        this.setDefaultRenderer(Boolean.class, null);
        this.setDefaultEditor(String.class, null);
        this.setDefaultEditor(Integer.class, null);
        this.setDefaultRenderer(Integer.class, null);
        this.setDefaultEditor(Double.class, null);
        this.setDefaultRenderer(Double.class, null);
        this.setDefaultRenderer(JLabel.class, null);

        //Create the standard defaults.  These are needed for the below wrapping to work.
        //Note that the standard defaults make use of SwingX DefaultTableRenderer.
        this.createDefaultRenderers();
        this.createDefaultEditors();

        // First wrap JTable Defaults renderers, which are SwingX renderers.
        this.setDefaultRenderer(Object.class, wrapRenderer(getDefaultRenderer(Object.class)));
        this.setDefaultRenderer(Number.class, wrapRenderer(getDefaultRenderer(Number.class)));
        this.setDefaultRenderer(Boolean.class, wrapRenderer(getDefaultRenderer(Boolean.class)));

        // Then set other defaults.
        this.setDefaultEditor(String.class, new StringCellEditor(1));

        //Ints, longs, shorts
        NumberCellEditor editor = new NumberCellEditor();
        this.setDefaultEditor(Integer.class, editor);
        this.setDefaultRenderer(Integer.class, wrapRenderer(editor.constructCorrespondingRenderer()));
        this.setDefaultEditor(Long.class, editor);
        this.setDefaultRenderer(Long.class, wrapRenderer(editor.constructCorrespondingRenderer()));
        this.setDefaultEditor(Short.class, editor);
        this.setDefaultRenderer(Short.class, wrapRenderer(editor.constructCorrespondingRenderer()));

        //Doubles, floats
        editor = new NumberCellEditor("##0.000###");
        this.setDefaultEditor(Double.class, editor);
        this.setDefaultRenderer(Double.class, wrapRenderer(editor.constructCorrespondingRenderer()));
        this.setDefaultEditor(Float.class, editor);
        this.setDefaultRenderer(Float.class, wrapRenderer(editor.constructCorrespondingRenderer()));

        //Dates are displayed in a date-time-tz string, by default.
        this.setDefaultRenderer(Date.class, wrapRenderer(new TableCellRenderer()
        {
            @Override
            public Component getTableCellRendererComponent(final JTable table,
                                                           final Object value,
                                                           final boolean isSelected,
                                                           final boolean hasFocus,
                                                           final int row,
                                                           final int column)
            {
                if(value == null)
                {
                    return new JLabel();
                }
                return new DefaultTableRenderer().getTableCellRendererComponent(table,
                                                                                HCalendar.buildDateTimeStr(((Date)value).getTime()),
                                                                                isSelected,
                                                                                hasFocus,
                                                                                row,
                                                                                column);
            }
        }));

        //General labels are displayed using JLabelTableCellRenderer.
        this.setDefaultRenderer(JLabel.class, wrapRenderer(new JLabelTableCellRenderer()));

        //Call model specific overrides if defined.
        if(getModel() instanceof DefaultEditorsAndRenderersOverrideTableModel)
        {
            ((DefaultEditorsAndRenderersOverrideTableModel)getModel()).setupOverrideEditorsAndRenderers();
        }

        _renderersConstructedFirstTime = true;
    }

    /**
     * Sets up the column headers to be rendered by {@link GenericTableHeaderRenderer}. Always call whenever the table
     * structure changes.
     */
    private void setupHeaderRenderers()
    {
        //I want the headers to be able to handle multiline HTML text properly, so I created a JLabel 
        //base header renderer.  Without this renderer, a weird double image is created if the width of
        //the header is wider than the column.
        final Enumeration e = getColumnModel().getColumns();
        while(e.hasMoreElements())
        {
            final TableColumn column = (TableColumn)e.nextElement();
            if(getModel() instanceof ColumnHeaderRendererTableModel)
            {
                column.setHeaderRenderer(((ColumnHeaderRendererTableModel)getModel()).getColumnHeaderRenderer(column.getModelIndex()));
            }
            else
            {
                column.setHeaderRenderer(new GenericTableHeaderRenderer());
            }
        }
    }

    /**
     * Called to initialize the add delete columns and make them work. It creates the needed mouse listener for the
     * labels in the add delete column.
     */
    private void initializeAddDelete()
    {
        if(_addDeleteInitialized)
        {
            return;
        }

        this.addMouseListener(new MouseAdapter()
        {
            @Override
            public void mousePressed(final MouseEvent e)
            {
                final TableModel model = GenericTable.this.getModel();
                if(!(model instanceof AddDeleteRowTableModel))
                {
                    return;
                }

                final AddDeleteRowTableModel adrModel = (AddDeleteRowTableModel)model;

                final int col = GenericTable.this.columnAtPoint(e.getPoint());
                final int row = GenericTable.this.rowAtPoint(e.getPoint());

                if(col == -1 || row == -1 || !(GenericTable.this.getValueAt(row, col) instanceof JLabel))
                {
                    return;
                }

                final JLabel label = (JLabel)GenericTable.this.getValueAt(row, col);
                if(TableTools.isJLabelADeleteLabel(label))
                {
                    //Establish the row to be deleted as selected and then call deleteRow.  
                    //Only one row can be deleted at a time via a delete cell.
                    GenericTable.this.clearSelection();
                    GenericTable.this.getSelectionModel().addSelectionInterval(row, row);
                    adrModel.deleteRow(row, TableTools.getSelectedRowsConvertedToModel(GenericTable.this));
                }
                else if(TableTools.isJLabelAnAddLabel(label))
                {
                    adrModel.addRow();
                }
            }
        });

        _addDeleteInitialized = true;
    }

    /**
     * Called when something shows need to hear for row selections by calling the method {@link #getRowSelectionBus()}.
     */
    private void initializeRowBus()
    {
        if(_rowSelectionBus != null)
        {
            return;
        }

        _rowSelectionBus = new EventBus();
        _rowBusPoster = new ListSelectionListener()
        {
            @Override
            @SuppressWarnings({"unchecked", "rawtypes"})
            public void valueChanged(final ListSelectionEvent e)
            {
                final List list = Lists.newArrayList();
                final int[] modelRows = getSelectedModelRows();
                final TableModel model = getModel();
                if(model instanceof GetRowTableModel)
                {
                    for(int i = 0; i < modelRows.length; i++)
                    {
                        list.add(((GetRowTableModel)model).getRow(modelRows[i]));
                    }
                }
                else
                {
                    for(int i = 0; i < modelRows.length; i++)
                    {
                        list.add(null);
                    }
                }
                _rowSelectionBus.post(list);
                _rowSelectionBus.post(new GeneralRowSelectionNotice(this, modelRows, list, getRowCount()));
            }
        };

        getSelectionModel().addListSelectionListener(_rowBusPoster);
    }

    /**
     * Overridden in order to attach the EventBus to the selection model.
     */
    @Override
    public void setSelectionModel(final ListSelectionModel model)
    {
        if(_rowSelectionBus != null)
        {
            getSelectionModel().removeListSelectionListener(_rowBusPoster);
            model.addListSelectionListener(_rowBusPoster);
        }
        super.setSelectionModel(model);
    }

    /**
     * If the supplied model implements {@link PreferredWidthsTableModel}, {@link MaxWidthsTableModel}, or
     * {@link MinWidthsTableModel}, then set the column size accordingly.
     */
    @Override
    public TableColumnModel createDefaultColumnModel()
    {
        //WARNING: If this is a DefaultTableColumnModelExt, an infinite loop occurs when changing from 1997 to 1999 in the RFC Diag Panel
        return new DefaultTableColumnModel()
        {
            @Override
            public void addColumn(final TableColumn column)
            {
                final TableModel model = getModel();
                if(model instanceof PreferredWidthsTableModel)
                {
                    final Integer width = ((PreferredWidthsTableModel)model).getPreferredWidth(getColumnCount());
                    if(width != null)
                    {
                        column.setPreferredWidth(width);
                    }
                }
                if(model instanceof MaxWidthsTableModel)
                {
                    final Integer width = ((MaxWidthsTableModel)model).getMaxWidth(getColumnCount());
                    if(width != null)
                    {
                        column.setMaxWidth(width);
                    }
                }
                if(model instanceof MinWidthsTableModel)
                {
                    final Integer width = ((MinWidthsTableModel)model).getMinWidth(getColumnCount());
                    if(width != null)
                    {
                        column.setMinWidth(width);
                    }
                }
                super.addColumn(column);
            }
        };
    }

    /**
     * @return True if the table is in the process of changing; this means {@link JXTable#tableChanged(TableModelEvent)}
     *         is about to be called. False if not (i.e., after the change is done).
     */
    public boolean isTableChanging()
    {
        return _tableChanging;
    }

    @Override
    public void tableChanged(final TableModelEvent e)
    {
        _tableChanging = true;
        super.tableChanged(e);
        _tableChanging = false;
        selectRecordedRowObjects();

        //The above may have resulted in a restructuring of the table, so reset the header renderer.
        if(e.getFirstRow() == TableModelEvent.HEADER_ROW) //fireTableRestructured causes this
        {
            //Only call this if it has already been called once before during initialization.
            if(_renderersConstructedFirstTime)
            {
                resetRenderersAndEditors();
            }
            setupHeaderRenderers();
        }

        fireTableUpdatedDueToModelChange(e);
    }

    /**
     * Only works if model implements {@link SelectionMaintainingTableModel}.
     * 
     * @return Array of selected Objects based on current selected indices. Uses {@link #convertRowIndexToModel(int)} to
     *         convert view row indices to model row indices.
     */
    protected void recordSelectedRowObjects()
    {
        if(getModel() instanceof SelectionMaintainingTableModel)
        {
            //If we are not recording, just return.
            if(_tableChanging)
            {
                return;
            }

            //Get the selected rows.
            final int[] selectedRows = getSelectedModelRows();
            if(selectedRows.length == 0)
            {
                _selectedRowObjects = null;
            }
            //If there are selected rows, then record the selection objects.
            else
            {
                _selectedRowObjects = new Object[selectedRows.length];
                for(int i = 0; i < selectedRows.length; i++)
                {
                    _selectedRowObjects[i] = ((SelectionMaintainingTableModel)getModel()).getRow(selectedRows[i]);
                }
            }
        }
    }

    /**
     * Only works if model implements {@link SelectionMaintainingTableModel}. This selects rows based on the given
     * Objects. Uses {@link #convertRowIndexToView(int)} to convert model row indices to view row indices.
     */
    private void selectRecordedRowObjects()
    {
        if(getModel() instanceof SelectionMaintainingTableModel)
        {
            //Only reselect recorded row objects if there are some to select.
            if((_selectedRowObjects != null) && (_selectedRowObjects.length > 0))
            {
                final int[] selectedModelRows = new int[_selectedRowObjects.length];
                for(int i = 0; i < _selectedRowObjects.length; i++)
                {
                    @SuppressWarnings({"unchecked", "rawtypes"})
                    final int modelRowIndex = ((SelectionMaintainingTableModel)getModel()).getRowIndex(_selectedRowObjects[i]);
                    if(modelRowIndex >= 0)
                    {
                        selectedModelRows[i] = convertRowIndexToView(modelRowIndex);
                    }
                    else
                    {
                        selectedModelRows[i] = -1;
                    }
                }
                TableTools.selectRows(this, selectedModelRows);
            }
        }
    }

    /**
     * @param listener The listener to add.
     */
    public void addGenericTableListener(final GenericTableListener listener)
    {
        _genericTableListeners.add(listener);
    }

    /**
     * @param listener The listener to remove.
     */
    public void removeGenericTableListener(final GenericTableListener listener)
    {
        _genericTableListeners.remove(listener);
    }

    /**
     * Clears all listeners.
     */
    public void clearGenericTableListener()
    {
        _genericTableListeners.clear();
    }

    /**
     * Fires an event to all {@link GenericTableListener}s, calling their
     * {@link GenericTableListener#tableUpdatedDueToModelChange()} method.
     */
    public void fireTableUpdatedDueToModelChange(final TableModelEvent e)
    {
        if(_genericTableListeners != null) //Appparently a risk when a table is first created!
        {
            for(final GenericTableListener listener: _genericTableListeners)
            {
                listener.tableUpdatedDueToModelChange(e);
            }
        }
    }

    /**
     * It appears that if this {@link GenericTable} is registered to an {@link EventBus} that posts a {@link Collection}
     * , then the table is enabled only if there is a positive number of items. I'm not sure who registers a
     * GenericTable for this purpose, however.
     * 
     * @param collection Posted with the event, it must represent a collection of the rows of the table?
     */
    @Subscribe
    public void setItems(final Collection collection)
    {
        try
        {
            this.setEnabled(collection.size() >= 1);
        }
        catch(final NullPointerException e)
        {
            this.setEnabled(false);
        }
    }

    /**
     * Gets an {@link EventBus} which is posted to with a list of the selected {@link GetRowTableModel#getRow(int)}, if
     * it is {@link GetRowTableModel} by the model. If it is not implemented, it will still post an event, but it will
     * be a list where the size is the number of selected rows but all entries are null.<br>
     * <br>
     * To register to listen to the events, call the {@link EventBus#register(Object)} method and implement a method
     * that receives a {@link Collection} and is annotated with Subcribe; see {@link MultiItemButton} for an example.
     * 
     * @return
     */
    public EventBus getRowSelectionBus()
    {
        initializeRowBus();
        return _rowSelectionBus;
    }

    /**
     * @return the model index of the selected row
     */
    public int getSelectedModelRow()
    {
        final int row = getSelectedRow();
        if(row == -1)
        {
            return -1;
        }
        else
        {
            return convertRowIndexToModel(row);
        }
    }

    /**
     * @return the model index of the selected column
     */
    public int getSelectedModelColumn()
    {
        final int col = getSelectedColumn();
        if(col == -1)
        {
            return -1;
        }
        else
        {
            return convertColumnIndexToModel(col);
        }
    }

    /**
     * @return the model indices of the selected rows
     */
    public int[] getSelectedModelRows()
    {
        final int[] rows = getSelectedRows();
        for(int i = 0; i < rows.length; i++)
        {
            if(rows[i] != -1)
            {
                rows[i] = convertRowIndexToModel(rows[i]);
            }
        }
        return rows;
    }

    /**
     * @return the model indices of the selected columns
     */
    public int[] getSelectedModelColumns()
    {
        final int[] cols = getSelectedColumns();
        for(int i = 0; i < cols.length; i++)
        {
            if(cols[i] != -1)
            {
                cols[i] = convertColumnIndexToModel(cols[i]);
            }
        }
        return cols;
    }

    /**
     * Creates a default {@link JToolBarBuilder}. It includes a {@link SelectAllButton}, {@link UnselectAllButton}, and
     * a {@link SelectUnreadyRowsButton}. If the assigned a {@link RefreshTableModel}, a {@link RefreshButton} will be
     * added right justified after a filler panel.<br>
     */
    public JToolBarBuilder createDefaultToolBarBuilder()
    {
        final JToolBarBuilder builder = new JToolBarBuilder();
        builder.setFloatable(false);
        builder.add(new SelectAllButton(this));
        builder.add(new UnselectAllButton(this));
        builder.add(new SelectUnreadyRowsButton(this));
        builder.add(JToolBarBuilder.FILLER);
        if(getModel() instanceof RefreshTableModel)
        {
            builder.add(new RefreshButton(this));
        }
        return builder;
    }

    /**
     * Sets up a panel to properly display this table with the given border. The table is placed in a
     * {@link JScrollPane} within a {@link BorderLayout} {@link JPanel}.
     * 
     * @param Border The border to add to the scroll panel created.
     */
    public JPanel createDisplayPanel(final Border border)
    {
        final JScrollPane pane = new JScrollPane(this);
        pane.setBorder(border);
        final JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        panel.add(pane, BorderLayout.CENTER);
        panel.add(createDefaultToolBarBuilder().build(), BorderLayout.SOUTH);
        return panel;
    }

    /**
     * @param row Index of row according to current visibility. This calls {@link #convertRowIndexToModel(int)} for the
     *            row before calling the method {@link GetRowTableModel#getRow(int)}.
     * @return the object representing a row via {@link GetRowTableModel}. Returns null if the model does not implement
     *         {@link GetRowTableModel} (or {@link SelectionMaintainingTableModel}.
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public RowT getWholeRow(final int visibleRow)
    {
        final TableModel model = getModel();
        if(model instanceof GetRowTableModel)
        {
            return (RowT)((GetRowTableModel)model).getRow(convertRowIndexToModel(visibleRow));
        }
        return null;
    }

    /**
     * @param row Index of row.
     * @return the object representing the currently selected row if one is selected via {@link GetRowTableModel}.
     *         Returns null if the model does not implement {@link GetRowTableModel} (or
     *         {@link SelectionMaintainingTableModel}.
     */
    public RowT getSelectedWholeRow()
    {
        final int visRow = getSelectedRow();
        if(visRow == -1)
        {
            return null;
        }
        else
        {
            return getWholeRow(visRow);
        }
    }

    /**
     * @return List of all selected rows via {@link GetRowTableModel}, which may be empty. Returns null if the model
     *         does not implement {@link GetRowTableModel} (or {@link SelectionMaintainingTableModel}.
     */
    public List<RowT> getSelectedWholeRows()
    {
        final List<RowT> list = Lists.newArrayList();
        final TableModel model = getModel();
        if(model instanceof GetRowTableModel)
        {
            final int[] visRows = this.getSelectedRows();
            for(int i = 0; i < visRows.length; i++)
            {
                list.add(getWholeRow(visRows[i]));
            }
            return list;
        }
        else
        {
            return null;
        }
    }

    /**
     * Executes the refresh() command within the {@link TableModel} if it implements {@link RefreshTableModel}.
     */
    public void refresh()
    {
        final TableModel model = getModel();
        if(model instanceof RefreshTableModel)
        {
            ((RefreshTableModel)model).refresh();
        }
    }
}
