package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.ScrollPaneConstants;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;

import ohd.hseb.hefs.utils.gui.jtable.models.MaxWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.MinWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.PreferredWidthsTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RowHeaderTableModel;

/**
 * A {@link JScrollPane} row header {@link JViewport} that takes a {@link GenericTable} and, if its model is a
 * {@link RowHeaderTableModel}, displays a single column from the {@link GenericTable} dictated by the models
 * {@link RowHeaderTableModel#getRowHeaderColumn()} method. This is used within {@link MarkedTableScrollPanel}.<br>
 * <br>
 * This inherits its popup menu from its parent: the scroll pane that displays it. It also provides a
 * {@link #_showRowHeaderMenuItem} that allows the user to dictate if the row header column is displayed. If the model
 * is not an instance of {@link RowHeaderTableModel}, the former button will be deselected and disabled. Otherwise, it
 * is enabled and tracks the visibility of the header.<br>
 * <br>
 * Whenever the table model is changed either via a normal {@link TableModelListener} event or via a
 * {@link GenericTableListener} {@link GenericTableListener#tableUpdatedDueToModelChange(TableModelEvent)} call, the
 * status of the row header will be reinitialized from scratch: the {@link #_rowHeaderTable} is updated to match the new
 * model, the row header visibility is redetermined, and the enabledness of the button is determined. The calls are
 * {@link #updateRowHeaderForNewModel()}, {@link #initializeRowHeaderVisibility()}, and
 * {@link #checkEnablednessOfButton()}. This sequence is also done when this is constructed.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class GenericTableRowHeaderViewport extends JViewport
{

    private final GenericTable _baseTable;
    private final GenericTable _rowHeaderTable;
    private final JScrollPane _displayingPane;
    private JMenuItem _showRowHeaderMenuItem;

    public GenericTableRowHeaderViewport(final GenericTable baseTable, final JScrollPane displayingPane)
    {
        _baseTable = baseTable;
        _displayingPane = displayingPane;

        _rowHeaderTable = new GenericTable();
        _rowHeaderTable.setShowVerticalLines(true);
        _rowHeaderTable.setCellSelectionEnabled(false);
        _rowHeaderTable.setRowSelectionAllowed(false);
        _rowHeaderTable.setColumnSelectionAllowed(false);

        //Listen for whenever the model undergoes a change.
        _baseTable.getModel().addTableModelListener(new TableModelListener()
        {
            @Override
            public void tableChanged(final TableModelEvent e)
            {
                updateRowHeaderForNewModel();
                initializeRowHeaderVisibility();
                checkEnablednessOfButton();
            }
        });
        _baseTable.addGenericTableListener(new GenericTableListener()
        {
            @Override
            public void tableUpdatedDueToModelChange(final TableModelEvent e)
            {
                updateRowHeaderForNewModel();
                initializeRowHeaderVisibility();
                checkEnablednessOfButton();

                //A model change triggers a new row sorter, I think.  So get a copy of it.
                if (_baseTable.getColumnCount() > 0 && _baseTable.getRowSorter().getModelRowCount() > 0)
                {
                	_rowHeaderTable.setRowSorter(_baseTable.getRowSorter());	
                }
            }
        });

        buildButtonToTrackVisibility();

        setInheritsPopupMenu(true);

        updateRowHeaderForNewModel();
        initializeRowHeaderVisibility();
        checkEnablednessOfButton();
    }

    /**
     * Buids a button designed to track the visibility of the row header.
     */
    public void buildButtonToTrackVisibility()
    {
        _showRowHeaderMenuItem = new JCheckBoxMenuItem("Show Row Header Column", false);
        _showRowHeaderMenuItem.addActionListener(new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                if(_baseTable.getModel() instanceof RowHeaderTableModel)
                {
                    if(_showRowHeaderMenuItem.isSelected())
                    {
                        addToScrollPane();
                    }
                    else
                    {
                        _displayingPane.setRowHeader(null);
                    }
                }
            }
        });
    }

    /**
     * Checks to see if the {@link #_showRowHeaderMenuItem} should be enabled or not depending on if the underyling
     * model is a {@link RowHeaderTableModel}.
     */
    private void checkEnablednessOfButton()
    {
        if((_baseTable.getModel() instanceof RowHeaderTableModel) && (getView() instanceof GenericTable))
        {
            _showRowHeaderMenuItem.setEnabled(true);
        }
        else
        {
            _showRowHeaderMenuItem.setSelected(false);
            _showRowHeaderMenuItem.setEnabled(false);
        }
    }

    /**
     * Updates {@link #_rowHeaderTable} for the current model, but only if it implements {@link RowHeaderTableModel}.
     */
    private void updateRowHeaderForNewModel()
    {
        if(_baseTable.getModel() instanceof RowHeaderTableModel)
        {
            final int baseColumn = ((RowHeaderTableModel)_baseTable.getModel()).getRowHeaderColumn();
            if(baseColumn >= 0)
            {
                _rowHeaderTable.setModel(new SingleColumnTableModel(_baseTable.getModel(), baseColumn));
                _rowHeaderTable.setRowSorter(null);
                _rowHeaderTable.setDefaultRenderer(_baseTable.getModel().getColumnClass(baseColumn),
                                                   new DefaultTableCellRenderer()
                                                   {
                                                       @Override
                                                       public Component getTableCellRendererComponent(final JTable table,
                                                                                                      final Object value,
                                                                                                      final boolean isSelected,
                                                                                                      final boolean hasFocus,
                                                                                                      final int row,
                                                                                                      final int column)
                                                       {
                                                           final Component c = _baseTable.getCellRenderer(row,
                                                                                                          baseColumn)
                                                                                         .getTableCellRendererComponent(_rowHeaderTable,
                                                                                                                        value,
                                                                                                                        isSelected,
                                                                                                                        hasFocus,
                                                                                                                        row,
                                                                                                                        baseColumn);
                                                           c.setBackground(Color.DARK_GRAY);
                                                           c.setForeground(Color.WHITE);
                                                           return c;
                                                       }
                                                   });
                setView(_rowHeaderTable);
            }
            else
            {
                _rowHeaderTable.setModel(new DefaultTableModel());
                setView(new JPanel());
            }

        }
        else
        {
            setView(new JPanel());
        }
    }

    /**
     * Adds this as the row header to {@link #_displayingPane}, but only if the {@link #_baseTable} model is a
     * {@link RowHeaderTableModel}, the current return of {@link #getView()} is a {@link GenericTable} (and not an empty
     * {@link JPanel} which it will be if the viewer cannot be displayed), and the number of columns in the displayed
     * {@link #_baseTable} exceeds 4.
     */
    private void initializeRowHeaderVisibility()
    {
        //If the table is a RowHeaderTableModel, and the view is showing the row header (and not a blank JPanel), 
        //check to see if enough columns are shown for the view to be added.
        if((_baseTable.getModel() instanceof RowHeaderTableModel) && (getView() instanceof GenericTable))
        {
            if(_baseTable.getModel().getColumnCount() <= 4)
            {
                removeFromScrollPane();
            }
            else
            {
                addToScrollPane();
            }
        }
        else
        {
            _showRowHeaderMenuItem.setSelected(false);
            removeFromScrollPane();
        }
    }

    /**
     * Adds this row header to {@link #_displayingPane} and sets the {@link #_showRowHeaderMenuItem} to be selected.
     */
    private void addToScrollPane()
    {
        final int baseColumn = ((RowHeaderTableModel)_baseTable.getModel()).getRowHeaderColumn();
        if(baseColumn >= 0)
        {
            _displayingPane.setRowHeader(this);
            _displayingPane.setCorner(ScrollPaneConstants.UPPER_LEFT_CORNER, _rowHeaderTable.getTableHeader());
            _displayingPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
            _displayingPane.getRowHeader().setVisible(true);
            _showRowHeaderMenuItem.setSelected(true);
        }
    }

    /**
     * Makes the {@link #_displayingPane} row header be null and sets the {@link #_showRowHeaderMenuItem} to be
     * deselected.
     */
    private void removeFromScrollPane()
    {
        _displayingPane.setRowHeader(null);
        _showRowHeaderMenuItem.setSelected(false);
    }

    /**
     * @return An array wrapping {@link #_showRowHeaderMenuItem}.
     */
    public Component[] getAdditionalPopUpComponent()
    {
        return new Component[]{_showRowHeaderMenuItem};
    }

    /**
     * A {@link TableModel} that displays one column within a provided base {@link TableModel}. It implements the width
     * {@link GenericTable} interfaces so that they return the same width as specified by the base table model, or null
     * if they aren't specified.
     * 
     * @author hankherr
     */
    private static class SingleColumnTableModel extends AbstractTableModel implements PreferredWidthsTableModel,
    MaxWidthsTableModel, MinWidthsTableModel
    {
        private final TableModel _baseModel;
        private final int _modelColumn;

        public SingleColumnTableModel(final TableModel baseModel, final int modelColumn)
        {
            super();
            _baseModel = baseModel;
            _modelColumn = modelColumn;
        }

        @Override
        public int getRowCount()
        {
            return _baseModel.getRowCount();
        }

        @Override
        public int getColumnCount()
        {
            return 1;
        }

        @Override
        public String getColumnName(final int columnIndex)
        {
            return _baseModel.getColumnName(_modelColumn);
        }

        @Override
        public Class<?> getColumnClass(final int columnIndex)
        {
            return _baseModel.getColumnClass(_modelColumn);
        }

        @Override
        public boolean isCellEditable(final int rowIndex, final int columnIndex)
        {
            return false;
        }

        @Override
        public Object getValueAt(final int rowIndex, final int columnIndex)
        {
            return _baseModel.getValueAt(rowIndex, _modelColumn);
        }

        @Override
        public void setValueAt(final Object aValue, final int rowIndex, final int columnIndex)
        {
        }

        @Override
        public Integer getPreferredWidth(final int column)
        {
            if(_baseModel instanceof PreferredWidthsTableModel)
            {
                return ((PreferredWidthsTableModel)_baseModel).getPreferredWidth(_modelColumn);
            }
            return null;
        }

        @Override
        public Integer getMinWidth(final int column)
        {
            if(_baseModel instanceof MinWidthsTableModel)
            {
                return ((MinWidthsTableModel)_baseModel).getMinWidth(_modelColumn);
            }
            return null;
        }

        @Override
        public Integer getMaxWidth(final int column)
        {
            if(_baseModel instanceof MaxWidthsTableModel)
            {
                return ((MaxWidthsTableModel)_baseModel).getMaxWidth(_modelColumn);
            }
            return null;
        }
    }
}
