package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.BorderLayout;
import java.awt.Color;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.ScrollPaneConstants;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;


/**
 * Given a standard {@link TableModel} and background color for the row header, this will create a {@link JTable} inside
 * a JScrollPane that displays the first column of the table model as a row header column, meaning it is always
 * displayed in the scrollpane as the user scrolls (similar to column header). The two underlying tables can be acquired
 * via get methods, but note that the indices of the columns in the {@link #_mainTable} {@link JTable} will be one less
 * than the corresponding index in the table model, because the first table model column is displayed in the
 * {@link #_rowHeaderTable}. <br>
 * <br>
 * Note that this is NOT compatible with {@link GenericTable}! For {@link GenericTable}, use
 * {@link GenericTableRowHeaderViewport} combined with a {@link MarkedTableScrollPanel} for display.
 * 
 * @author herrhd
 */
public class JTableWithRowHeaderPanel extends JPanel
{
    private static final long serialVersionUID = 1L;

    private JTable _mainTable = null;
    private JTable _rowHeaderTable = null;
    private JScrollPane _tableScrollPane = null;

    public JTableWithRowHeaderPanel(final TableModel tableModel,
                                    final Color rowHeaderCellBackgroundColor,
                                    final int rowHeaderFixedWidth)
    {
        final TableColumnModel mainColumnModel = new DefaultTableColumnModel()
        {
            private static final long serialVersionUID = 1L;
            boolean first = true;

            @Override
            public void addColumn(final TableColumn tc)
            {
                if(first)
                {
                    first = false;
                    return;
                }
                super.addColumn(tc);
            }
        };

        final TableColumnModel rowHeaderModel = new DefaultTableColumnModel()
        {
            private static final long serialVersionUID = 1L;
            boolean first = true;

            @Override
            public void addColumn(final TableColumn tc)
            {
                if(first)
                {
                    if(rowHeaderFixedWidth > 0)
                    {
                        tc.setMaxWidth(rowHeaderFixedWidth);
                    }
                    super.addColumn(tc);
                    first = false;
                }
            }
        };

        _mainTable = new JTable(tableModel, mainColumnModel);
        _rowHeaderTable = new JTable(tableModel, rowHeaderModel);
        _mainTable.createDefaultColumnsFromModel();
        _rowHeaderTable.createDefaultColumnsFromModel();

        _mainTable.setSelectionModel(_rowHeaderTable.getSelectionModel());

        _rowHeaderTable.setBackground(rowHeaderCellBackgroundColor);
        _rowHeaderTable.setColumnSelectionAllowed(false);

        final JViewport jv = new JViewport();
        jv.setView(_rowHeaderTable);
        jv.setPreferredSize(_rowHeaderTable.getMaximumSize());

        _mainTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

        _tableScrollPane = new JScrollPane(_mainTable);
        _tableScrollPane.setRowHeader(jv);
        _tableScrollPane.setCorner(ScrollPaneConstants.UPPER_LEFT_CORNER, _rowHeaderTable.getTableHeader());
        _tableScrollPane.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);

        this.setLayout(new BorderLayout());
        add(_tableScrollPane, BorderLayout.CENTER);
    }

    public JTable getMainTable()
    {
        return this._mainTable;
    }

    public JTable getRowHeaderTable()
    {
        return this._rowHeaderTable;
    }

    public JScrollPane getTableScrollPane()
    {
        return this._tableScrollPane;
    }
}
