package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;

import javax.swing.JComponent;
import javax.swing.JScrollBar;
import javax.swing.JTable;
import javax.swing.SwingUtilities;

import ohd.hseb.charter.panel.TrackExposingBasicScrollBarUI;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * Paired with a {@link JScrollBar}, this panel can draw marks that correspond to positions within whatever the
 * {@link JScrollBar} is scrolling, usually a {@link JTable}. It is meant to be displayed next to or below whatever
 * panel includes the scrollbar, so that the parent of this panel is a parent of the tracked scrollbar. For example, see
 * {@link MarkedTableScrollPanel}. <br>
 * <br>
 * The marks to draw are defined as {@link Mark} instances and are provided via the {@link #setMarks(Collection)}
 * method. Note that the Mark class is static within the class below, so you need to access it as 'MarkPanel.Mark'. Each
 * {@link Mark} has a priority associated with it that dictates the order in which they are drawn (lowest to highest, so
 * that higher priority marks are drawn on top; see the constants {@link #TIER_0_PRIORITY}, {@link #TIER_1_PRIORITY},
 * {@link #TIER_2_PRIORITY}, {@link #TIER_3_PRIORITY}, {@link #TIER_4_PRIORITY}, {@link #TIER_5_PRIORITY} for examples)
 * and the minimum pixel size of a mark (priority 0 or less can be 1 pixel in size, while others with greater priority
 * are {@link #MINIMUM_MARK_PIXEL_SIZE} pixels minimum in size, making them more easily clickable).
 * 
 * @author hankherr
 */
public class MarkPanel extends JComponent
{
    private static final long serialVersionUID = 1L;
    private static final int PANEL_WIDTH = 8;
    private static final int MINIMUM_MARK_PIXEL_SIZE = 2;

    /**
     * Lowest tier or background tier.
     */
    public static final int TIER_0_PRIORITY = 0;
    public static final int TIER_1_PRIORITY = 10;
    public static final int TIER_2_PRIORITY = 100;
    public static final int TIER_3_PRIORITY = 1000;
    public static final int TIER_4_PRIORITY = 10000;

    /**
     * Drawn on top of all other tiers.
     */
    public static final int TIER_5_PRIORITY = 100000;

    private final JScrollBar _trackedBar;

    /**
     * Stores for each {@link Mark}, a {@link Rectangle} that reflects a clickable rectangle. This is a rectangle that
     * is identical to the drawn mark in the long dimension, but as wide/high as this panel in the short dimension. That
     * extra width/height makes it easier to users to click on the mark panel within {@link MarkedTableScrollPanel}.
     */
    private final LinkedHashMap<Mark, Rectangle> _markToRectangleMap = Maps.newLinkedHashMap();

    /**
     * @param trackedBar The {@link JScrollBar} which this panel is tracking.
     */
    public MarkPanel(final JScrollBar trackedBar)
    {
        trackedBar.setUI(new TrackExposingBasicScrollBarUI());
        _trackedBar = trackedBar;

        //Assumes it is laid out appropriately in a BorderLayout so that the 100 value does not matter.  Only the 8-value does.
        if(isVertical())
        {
            setPreferredSize(new Dimension(PANEL_WIDTH, 100));
        }
        else
        {
            setPreferredSize(new Dimension(100, PANEL_WIDTH));
        }

        //Keep this in synch with the tracked bar.
        _trackedBar.addComponentListener(new ComponentListener()
        {
            @Override
            public void componentResized(final ComponentEvent e)
            {
                //Should be repainted automatically
            }

            @Override
            public void componentMoved(final ComponentEvent e)
            {
            }

            @Override
            public void componentShown(final ComponentEvent e)
            {
                setVisible(true);
            }

            @Override
            public void componentHidden(final ComponentEvent e)
            {
                setVisible(false);
            }
        });

        //Mouse motion listener used for changing the cursor.
        this.addMouseMotionListener(new MouseAdapter()
        {
            @Override
            public void mouseMoved(final MouseEvent e)
            {
                if(getMarkAtPoint(e.getPoint()) != null)
                {
                    setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
                }
                else
                {
                    setCursor(Cursor.getDefaultCursor());
                }
            }
        });
    }

    /**
     * @return True if the {@link #_trackedBar} orientation is vertical.
     */
    protected boolean isVertical()
    {
        return (_trackedBar.getOrientation() == JScrollBar.VERTICAL);
    }

    /**
     * @return The bounds of {@link #_trackedBar} converted to the parent of this panel, which should be a parent of the
     *         panel displaying the {@link #_trackedBar}.
     */
    private Rectangle getScrollBarTrackRectangle()
    {
        final Rectangle rect = ((TrackExposingBasicScrollBarUI)_trackedBar.getUI()).getDrawnTrackBounds();
        return SwingUtilities.convertRectangle(_trackedBar, rect, getParent());
    }

    /**
     * Records the marks after sorting them by priority. Calls {@link #repaint()}.
     */
    public void setMarks(final Collection<Mark> marks)
    {
        final List<Mark> sortedMarks = Lists.newArrayList(marks);
        Collections.sort(sortedMarks);
        _markToRectangleMap.clear();
        for(final Mark mark: sortedMarks)
        {
            _markToRectangleMap.put(mark, null);
        }
        repaint();
    }

    protected Set<Mark> getMarks()
    {
        return _markToRectangleMap.keySet();
    }

    /**
     * @param pixelPoint The point to check relative to this component.
     * @return The {@link Mark} that was drawn to contain the provided pixel point.
     */
    public Mark getMarkAtPoint(final Point pixelPoint)
    {
        final List<Mark> marks = Lists.newArrayList(_markToRectangleMap.keySet());
        Mark foundMark = null;
        for(int i = marks.size() - 1; i >= 0; i--)
        {
            //The null check is important if the table is not actually being displayed due to have a teeny tiny size for it in 
            //the split pane.  In that case, the marks will not be drawn, so that null is returned from the rectangle map.
            if((_markToRectangleMap.get(marks.get(i)) != null)
                && (_markToRectangleMap.get(marks.get(i)).contains(pixelPoint)))
            {
                if((foundMark == null) || (marks.get(i).getPriority() > foundMark.getPriority()))
                {
                    foundMark = marks.get(i);
                }
            }
        }
        return foundMark;
    }

    /**
     * @param pixelPoint The point to check in this panel.
     * @param useXValue Whether to use the points x-value or y-value for checking (related to if this panel is vertical
     *            or horizontal.
     * @return A value in the scale of the {@link Mark} instances that corresponds to the pixel point. If no
     *         {@link Mark} includes the specified point, {@link Integer#MIN_VALUE} is returned.
     */
    public int getValueAtPoint(final Point pixelPoint)
    {
        final Mark mark = getMarkAtPoint(pixelPoint);
        if(mark == null)
        {
            return Integer.MIN_VALUE;
        }

        //If the start and end are the same, return it.
        if(mark.getStartValue() == mark.getEndValue())
        {
            return mark.getStartValue();
        }

        //Otherwise, figure out the value to return from the size of the draw rectangle.  Note that the size has a minimum
        //size of 
        final Rectangle markRectangle = _markToRectangleMap.get(mark);
        int startPixel = -1;
        int endPixel = -1;
        if(isVertical())
        {
            endPixel = markRectangle.y + markRectangle.height - 1;
            startPixel = markRectangle.y;
        }
        else
        {
            startPixel = markRectangle.x;
            endPixel = markRectangle.x + markRectangle.width - 1; //End point is NOT inclusive in a rectangle.
        }

        //With a 1-pixel width mark, the ratio below cannot be computed, so just return the start value of the mark.
        if(endPixel == startPixel)
        {
            return mark._startValue;
        }

        //Determine if x or y is used for the provided point, compute a ratio, and return the approximate clicked value.
        int pixelUsed = pixelPoint.x;
        if(isVertical())
        {
            pixelUsed = pixelPoint.y;
        }
        final double ratio = (double)(pixelUsed - startPixel) / (double)(endPixel - startPixel);

        return (int)(mark._startValue + ratio * (mark._endValue - mark._startValue));
    }

    /**
     * @param mark The mark for which to compute the pixel.
     * @param value The value of the mark, typically {@link Mark#_startValue} or {@link Mark#_endValue} + 1.
     * @return The pixel location.
     */
    private int computeStartPixelForMarkValue(final Mark mark, final int value)
    {
        final Rectangle bounds = getScrollBarTrackRectangle();
        if(isVertical())
        {
            //I don't want any units issues.
            return bounds.y + (int)Math.round(value * computeMarkSize(mark));
        }
        else
        {
            //I don't want any units issues.
            return bounds.x + (int)Math.round(value * computeMarkSize(mark));
        }
    }

    /**
     * @return Based on {@link Mark#_maximum} and {@link Mark#_minimum} as well as the bounds returned by
     *         {@link #getScrollBarTrackRectangle()}, this computes the size of mark of size one (where
     *         {@link Mark#_startValue} equals {@link Mark#_endValue}).
     */
    private double computeMarkSize(final Mark mark)
    {
        final Rectangle bounds = getScrollBarTrackRectangle();
        if(isVertical())
        {
            return (bounds.getHeight() / (mark._maximum - mark._minimum + 1));
        }
        else
        {
            return (bounds.getWidth() / (mark._maximum - mark._minimum + 1));
        }
    }

    /**
     * Calls {@link #computeStartPixelForMarkValue(Mark, int)} passing in the mark's {@link Mark#_startValue}.
     */
    private int computeStartPixelForMarkStart(final Mark mark)
    {
        return computeStartPixelForMarkValue(mark, mark._startValue);
    }

    /**
     * Paints the mark on a vertical version of this panel.
     */
    private void drawMarkOnVerticalPanel(final Graphics2D g, final Mark mark)
    {
        //The top pixel is derived from the mark start value.  The bottom pixel is one step after the mark's
        //end value.  This ensures that there is no gap between adjacent marks.
        final int topPixel = computeStartPixelForMarkStart(mark);
        final int bottomPixel = computeStartPixelForMarkValue(mark, mark._endValue + 1);
        final int width = getBounds().width;

        g.setColor(mark._color);
        g.fillRect(2, topPixel, width - 3, Math.max(mark.getMinimumPixelSize(), bottomPixel - topPixel));

        //Store a rectangle that's as wide as this panel to provide a bigger target that's easier to click.
        _markToRectangleMap.put(mark,
                                new Rectangle(0, topPixel, width, Math.max(mark.getMinimumPixelSize(), bottomPixel
                                    - topPixel)));
    }

    /**
     * Paints the mark on a horizontal version of this panel.
     */
    private void drawMarkOnHorizontalPanel(final Graphics2D g, final Mark mark)
    {
        //The left pixel is derived from the mark start value.  The right pixel is one step after the mark's
        //end value.  This ensures that there is no gap between adjacent marks.
        final int leftPixel = computeStartPixelForMarkStart(mark);
        final int rightPixel = computeStartPixelForMarkValue(mark, mark._endValue + 1);
        final int height = getBounds().height;

        g.setColor(mark._color);
        g.fillRect(leftPixel, 2, Math.max(mark.getMinimumPixelSize(), rightPixel - leftPixel), height - 3);

        //Store a rectangle that's as wide as this panel to provide a bigger target that's easier to click.
        _markToRectangleMap.put(mark,
                                new Rectangle(leftPixel,
                                              0,
                                              Math.max(mark.getMinimumPixelSize(), rightPixel - leftPixel),
                                              height));
    }

    /**
     * Calls eitehr {@link #drawMarkOnHorizontalPanel(Graphics2D, Mark)} or
     * {@link #drawMarkOnVerticalPanel(Graphics2D, Mark)} depending on the return of {@link #isVertical()}.
     */
    private void drawMark(final Graphics2D g, final Mark mark)
    {
        if(isVertical())
        {
            drawMarkOnVerticalPanel(g, mark);
        }
        else
        {
            drawMarkOnHorizontalPanel(g, mark);
        }
    }

    @Override
    public void paint(final Graphics g)
    {
//Black frame on a white rectangle. --REMOVED BG FOR NOW!!!
//        if(isVertical())
//        {
//            g.setColor(Color.BLACK);
//            g.drawRect(1,
//                       getScrollBarTrackRectangle().y - 1,
//                       getBounds().width - 2,
//                       getScrollBarTrackRectangle().height + 1);
//            g.setColor(Color.WHITE);
//            g.fillRect(2,
//                       getScrollBarTrackRectangle().y,
//                       getBounds().width - 3,
//                       getScrollBarTrackRectangle().height - 1);
//        }
//        else
//        {
//            g.setColor(Color.BLACK);
//            g.drawRect(getScrollBarTrackRectangle().x - 1,
//                       1,
//                       getScrollBarTrackRectangle().width + 1,
//                       getBounds().height - 2);
//            g.setColor(Color.WHITE);
//            g.fillRect(getScrollBarTrackRectangle().x,
//                       2,
//                       getScrollBarTrackRectangle().width - 2,
//                       getBounds().height - 3);
//        }

        for(final Mark mark: _markToRectangleMap.keySet())
        {
            drawMark((Graphics2D)g, mark);
        }
    }

    /**
     * Object used to store marks. Each mark is specified by a start value, end value, minimum possible overall value,
     * maximum possible overall value, color, and priority level. The minimum and maximum values indicate the scale for
     * the mark panel and are translated to pixels for drawing.
     * 
     * @author hankherr
     */
    public static class Mark implements Comparable<Mark>
    {
        /**
         * First value for the marked range.
         */
        private int _startValue;

        /**
         * End value for the marked range.
         */
        private int _endValue;

        /**
         * Minimum overall possible value, providing scale for the mark.
         */
        private final int _minimum;

        /**
         * Maximum overall possible value, providing scale for the mark.
         */
        private final int _maximum;

        private final Color _color;

        /**
         * Priority, with higher value marks being drawn on top of lower value marks, meaning that it is more important
         * they be seen. Without priority levels, its possible for some marks to squeeze out other marks altogether,
         * particularly when marks are one pixel or less in width.
         */
        private final int _priority;

        /**
         * @param priority For priority, it is recommended that the tier constants be used, for example
         *            {@link MarkPanel#TIER_0_PRIORITY}. The lower tier priorities are drawn behind higher tiers.
         */
        public Mark(final int startValue,
                    final int endValue,
                    final int minimum,
                    final int maximum,
                    final Color c,
                    final int priority)
        {
            _startValue = startValue;
            _endValue = endValue;
            _minimum = minimum;
            _maximum = maximum;
            _color = c;
            _priority = priority;
        }

        public int getStartValue()
        {
            return _startValue;
        }

        public int getEndValue()
        {
            return _endValue;
        }

        public int getMinimum()
        {
            return _minimum;
        }

        public int getMaximum()
        {
            return _maximum;
        }

        public Color getColor()
        {
            return _color;
        }

        public int getPriority()
        {
            return _priority;
        }

        public int getMinimumPixelSize()
        {
            if(_priority <= 0)
            {
                return 1;
            }
            return MINIMUM_MARK_PIXEL_SIZE;
        }

        /**
         * Attempt to merge the mark with this mark. If it cannot be merged, implying that the marks are non-overlapping
         * and not adjacent, then do nothing. Otherwise, expand this mark if necessary to encompass the provided mark.
         * 
         * @return False if the provided mark is incompatible with this mark, or it is neither contained within,
         *         overlapping, nor adjacent to this mark.
         */
        public boolean mergeWithMark(final Mark mark)
        {
            //Incompatible...
            if((!_color.equals(mark.getColor())) || (_minimum != mark.getMinimum()) || (_maximum != mark.getMaximum()))
            {
                return false;
            }
            //Neither overlapping nor adjacent.
            if((mark.getEndValue() < _startValue - 1) || (mark.getStartValue() > _endValue + 1))
            {
                return false;
            }

            _startValue = Math.min(_startValue, mark.getStartValue());
            _endValue = Math.max(_endValue, mark.getEndValue());
            return true;
        }

        @Override
        public String toString()
        {
            return "Mark: " + _startValue + ", " + _endValue + ", " + _minimum + ", " + _maximum + ", " + _color + ".";
        }

        /**
         * Sort by priority level.
         */
        @Override
        public int compareTo(final Mark o)
        {
            return Integer.valueOf(getPriority()).compareTo(o.getPriority());
        }

    }

}
