package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.RowSorterEvent;
import javax.swing.event.RowSorterListener;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.event.TableColumnModelListener;
import javax.swing.event.TableModelEvent;

import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.gui.jtable.models.RowColumnMarkingTableModel;
import ohd.hseb.hefs.utils.gui.jtable.models.RowHeaderTableModel;

import com.google.common.collect.Lists;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;

/**
 * This panel displays a {@link GenericTable} within a {@link JScrollPane}. If the table's model is an instance of
 * {@link RowColumnMarkingTableModel}, then this will setup a vertical and horizontal {@link MarkPanel} to allow for
 * marks generated by the model.<br>
 * <br>
 * The marks are recomputed whenever the table is updated due to a model change, selecting or sorting rows, or selecting
 * or moving columns. If the marks should be updated any other time, post a {@link RecomputeMarksNotice} to the
 * {@link EventBus} provided to the constructor of this panel, or, if that is not provided, then call
 * {@link #recomputeMarks()} manually.<br>
 * <br>
 * If the provided model implements {@link RowHeaderTableModel}, this will add a {@link GenericTableRowHeaderViewport}
 * as the header of {@link #_scrollPane} when constructed via calling {@link #createRowHeaderViewport()}. The header can
 * then be turned on-or-off via a menu item returned by {@link #constructTablePanelAdditionalComponents()}. The row
 * header is only displayed if the table model's method {@link RowHeaderTableModel#getRowHeaderColumn()} returns a
 * non-negative number. The header will display the column specified by that value. See
 * {@link GenericTableRowHeaderViewport} for more information. <br>
 * <br>
 * This panel and its displayed {@link #_scrollPane} inherit the popup menu from its parent. However, the
 * {@link MarkPanel}s displayed within this do not. I don't think its right for a {@link MarkPanel} to display a popup;
 * clicking on a {@link MarkPanel} should have its own meaning.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class MarkedTableScrollPanel extends JPanel implements RecomputeMarksNotice.Subscriber
{

    private final JScrollPane _scrollPane;
    private final GenericTable _table;
    private MarkPanel _verticalMarkPanel;
    private MarkPanel _horizontalMarkPanel;
    private GenericTableRowHeaderViewport _rowHeaderViewport;
    private boolean _ignoreRequestsToRecomputeMarks = false;

    /**
     * @param table The {@link GenericTable} for which marks will be displayed. Its model must implement
     *            {@link RowColumnMarkingTableModel} for any marks to be drawn.
     * @param markEventBus The {@link EventBus} to which to listen for {@link RecomputeMarksNotice}.
     */
    public MarkedTableScrollPanel(final GenericTable table, final EventBus markEventBus)
    {
        this.setLayout(new BorderLayout());
        _table = table;
        _table.repaint();
        if(markEventBus != null)
        {
            markEventBus.register(this);
        }

        _scrollPane = new JScrollPane(table);
        add(_scrollPane, BorderLayout.CENTER);

        setInheritsPopupMenu(true);
        _scrollPane.setInheritsPopupMenu(true);

        _rowHeaderViewport = new GenericTableRowHeaderViewport(_table, _scrollPane);

        if(table.getModel() instanceof RowColumnMarkingTableModel)
        {
            _verticalMarkPanel = new MarkPanel(_scrollPane.getVerticalScrollBar());
            _horizontalMarkPanel = new MarkPanel(_scrollPane.getHorizontalScrollBar());

            this.add(_verticalMarkPanel, BorderLayout.EAST);
            this.add(_horizontalMarkPanel, BorderLayout.SOUTH);

            //Listeners used to handle clicking on the mark panels.
            _verticalMarkPanel.addMouseListener(new MouseAdapter()
            {
                @Override
                public void mouseClicked(final MouseEvent e)
                {
                    final int rowToScrollTo = _verticalMarkPanel.getValueAtPoint(e.getPoint());
                    if(rowToScrollTo < 0)
                    {
                        return;
                    }

                    //Used to adjust the value for centering the column.
                    final int firstVisibleRow = _table.rowAtPoint(new Point(0, _table.getVisibleRect().y));
                    final int lastVisibleRow = _table.rowAtPoint(new Point(0, _table.getVisibleRect().y
                        + _table.getVisibleRect().height - 1));

                    //Scroll to it.
                    if(rowToScrollTo < firstVisibleRow)
                    {
                        _table.scrollRowToVisible(rowToScrollTo - (lastVisibleRow - firstVisibleRow) / 2);
                    }
                    else if(rowToScrollTo > lastVisibleRow)
                    {
                        _table.scrollRowToVisible(rowToScrollTo + (lastVisibleRow - firstVisibleRow) / 2);
                    }
                }
            });
            _horizontalMarkPanel.addMouseListener(new MouseAdapter()
            {
                @Override
                public void mouseClicked(final MouseEvent e)
                {
                    final int colToScrollTo = _horizontalMarkPanel.getValueAtPoint(e.getPoint());
                    if(colToScrollTo < 0)
                    {
                        return;
                    }

                    //Used to adjust the value for centering the column.
                    final int firstVisibleCol = _table.columnAtPoint(new Point(_table.getVisibleRect().x, 0));
                    final int lastVisibleCol = _table.columnAtPoint(new Point(_table.getVisibleRect().x
                        + _table.getVisibleRect().width, 0));

                    //Scroll to it.
                    if(colToScrollTo < firstVisibleCol)
                    {
                        _table.scrollColumnToVisible(colToScrollTo - (lastVisibleCol - firstVisibleCol) / 2);
                    }
                    else if(colToScrollTo > lastVisibleCol)
                    {
                        _table.scrollColumnToVisible(colToScrollTo + (lastVisibleCol - firstVisibleCol) / 2);
                    }
                }
            });
        }

        //Various listeners have to be added to make sure marks are recomputed as needed.
        //Recompute whenever the table updates due to a model change.
        _table.addGenericTableListener(new GenericTableListener()
        {
            @Override
            public void tableUpdatedDueToModelChange(final TableModelEvent e)
            {
                if(_table.getModel() instanceof RowColumnMarkingTableModel)
                {
                    recomputeMarks();
                }
            }
        });
        //Recompute whenever the rows are sorted.  Note that if a row is selected, the selected row
        //is not accurate when this is called.  Hence the reason for the selection listener below.
        _table.getRowSorter().addRowSorterListener(new RowSorterListener()
        {
            @Override
            public void sorterChanged(final RowSorterEvent e)
            {
                recomputeMarks();
            }
        });
        //Recompute whenever the columns are moved or selected.
        _table.getColumnModel().addColumnModelListener(new TableColumnModelListener()
        {
            @Override
            public void columnAdded(final TableColumnModelEvent e)
            {
                //This occurs while a table is being built to reflect a model, so we cannot recompute marks yet.  
                //When the table model changes, we need to rely on the tableChanged(...) above to recompute marks.
            }

            @Override
            public void columnRemoved(final TableColumnModelEvent e)
            {
                //This occurs while a table is being built to reflect a model, so we cannot recompute marks yet.  
                //When the table model changes, we need to rely on the tableChanged(...) above to recompute marks.
            }

            @Override
            public void columnMoved(final TableColumnModelEvent e)
            {
                //Requires remarking the horizontal axis, possibly.
                recomputeMarks();
            }

            @Override
            public void columnMarginChanged(final ChangeEvent e)
            {
                //I think this is just a GUI thing.
            }

            @Override
            public void columnSelectionChanged(final ListSelectionEvent e)
            {
                //To allow marks to vary by selected column.
                if(!e.getValueIsAdjusting())
                {
                    recomputeMarks();
                }
            }
        });
        //Recompute whenever a row is selected.
        _table.getSelectionModel().addListSelectionListener(new ListSelectionListener()
        {
            @Override
            public void valueChanged(final ListSelectionEvent e)
            {
                recomputeMarks();
            }
        });
    }

    /**
     * Calls {@link MarkedTableScrollPanel#MarkedTableScrollPanel(GenericTable, EventBus)} passing in null for the event
     * bus.
     */
    public MarkedTableScrollPanel(final GenericTable table)
    {
        this(table, null);
    }

    /**
     * @return Components to add to the popup menu for the table displayed within.
     */
    public Component[] constructTablePanelAdditionalComponents()
    {
        return _rowHeaderViewport.getAdditionalPopUpComponent();
    }

    /**
     * Calls {@link #buildColumnMarks()} and {@link #buildRowMarks()} to build the marks for the
     * {@link #_horizontalMarkPanel} and {@link #_verticalMarkPanel}.
     */
    public void recomputeMarks()
    {
        if(!ignoreRequestsToRecomputeMarks())
        {
            _verticalMarkPanel.setMarks(buildRowMarks());
            _horizontalMarkPanel.setMarks(buildColumnMarks());
        }
    }

    /**
     * Makes use of the {@link MarkPanel.Mark#mergeWithMark(ohd.hseb.hefs.utils.gui.jtable.MarkPanel.Mark)} method to
     * either merge the new mark with an existing mark or, if no appropriate mark is found, add the mark to the list.
     * 
     * @param marks List to which to add.
     * @param addMark Mark to add.
     */
    private void addMarkToList(final List<MarkPanel.Mark> marks, final MarkPanel.Mark addMark)
    {
        for(final MarkPanel.Mark mark: marks)
        {
            if(mark.mergeWithMark(addMark))
            {
                return;
            }
        }
        marks.add(addMark);
    }

    /**
     * This method assumes that the table has finished building relative to the underlying model.
     * 
     * @return {@link List} of marks constructed based on the table rows. If the table is currently changing, determined
     *         by calling {@link #_table}'s {@link GenericTable#isTableChanging()} method, then an empty list is
     *         returned.
     */
    private List<MarkPanel.Mark> buildRowMarks()
    {
        final List<MarkPanel.Mark> marks = Lists.newArrayList();
        if(_table.isTableChanging())
        {
            return marks;
        }
        final RowColumnMarkingTableModel model = (RowColumnMarkingTableModel)_table.getModel();
        for(int row = 0; row < _table.getRowCount(); row++)
        {
            final Dyad<Color, Integer> markParms = model.getRowMarkColor(_table.convertRowIndexToModel(row));
            if(markParms != null)
            {
                addMarkToList(marks, new MarkPanel.Mark(row,
                                                        row,
                                                        0,
                                                        model.getRowCount() - 1,
                                                        markParms.getFirst(),
                                                        markParms.getSecond()));
            }
        }
        return marks;
    }

    /**
     * This method assumes that the table has finished building relative to the underlying model.
     * 
     * @return {@link List} of marks constructed based on the table columns. If the table is currently changing,
     *         determined by calling {@link #_table}'s {@link GenericTable#isTableChanging()} method, then an empty list
     *         is returned.
     */
    private List<MarkPanel.Mark> buildColumnMarks()
    {
        final List<MarkPanel.Mark> marks = Lists.newArrayList();
        if(_table.isTableChanging())
        {
            return marks;
        }
        final RowColumnMarkingTableModel model = (RowColumnMarkingTableModel)_table.getModel();
        for(int col = 0; col < _table.getColumnCount(); col++)
        {
            final Dyad<Color, Integer> markParms = model.getColumnMarkColor(_table.convertColumnIndexToModel(col));
            if(markParms != null)
            {
                addMarkToList(marks, new MarkPanel.Mark(col,
                                                        col,
                                                        0,
                                                        model.getColumnCount() - 1,
                                                        markParms.getFirst(),
                                                        markParms.getSecond()));
            }
        }
        return marks;
    }

    /**
     * Exposed for tailoring of the scroll pane.
     * 
     * @return The {@link JScrollPane} displayed within this panel.
     */
    public JScrollPane getScrollPane()
    {
        return _scrollPane;
    }

    public boolean ignoreRequestsToRecomputeMarks()
    {
        return _ignoreRequestsToRecomputeMarks;
    }

    public void setIgnoreRequestsToRecomputeMarks(final boolean ignoreRequestsToRecomputeMarks)
    {
        _ignoreRequestsToRecomputeMarks = ignoreRequestsToRecomputeMarks;
    }

    @Override
    @Subscribe
    public void reactToRecomputMarks(final RecomputeMarksNotice evt)
    {
        if(_table.getModel() instanceof RowColumnMarkingTableModel)
        {
            recomputeMarks();
        }
    }

}
