package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.Component;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;

import javax.swing.DefaultCellEditor;
import javax.swing.JFormattedTextField;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.table.TableCellRenderer;
import javax.swing.text.DefaultFormatterFactory;
import javax.swing.text.NumberFormatter;

import ohd.hseb.hefs.utils.tools.NumberTools;

import org.jdesktop.swingx.renderer.DefaultTableRenderer;

/**
 * Designed for numeric cell editing, the format of the number may be passed into the constructor or a default
 * {@link DecimalFormat} instance will be used. This component will select all upon gaining focus after only one click.
 * A corresponding renderer, one that uses the same format, can be accessed via the
 * {@link #constructCorrespondingRenderer()} method.<br>
 * <br>
 * Note that the editor will not allow editing to end upon hitting the enter key unless the text is a valid number.
 * However, if you click outside the cell, it will revert invalid numbers to the last valid number prior to allowing a
 * cell switch.
 * 
 * @author hank.herr
 */
public class NumberCellEditor extends DefaultCellEditor
{
    private static final long serialVersionUID = 1L;

    /**
     * Formatter underlying the editor and renderer.
     */
    private NumberFormatter _formatter = null;

    /**
     * Default calls the super constructor with a {@link JFormattedTextField} using a regular {@link DecimalFormat}. The
     * {@link JFormattedTextField#commitEdit()} method is overridden to check that the text value is a number before
     * committing.
     */
    public NumberCellEditor()
    {
        //Commit edit is overridden because the default commitEdit will allow some values,
        //such as '75z', to make it to the setValueAt method in the table model upon hitting
        //<enter>.  However, other values, such as 'hank', will not (the JFormattedTextField 
        //rejects them). To make it consistent, I do a Double.parseDouble on the String to 
        //make sure it is a valid number of some kind and throw a ParseException if not.  Then
        //I commitEdit.
        super(new JFormattedTextField(new DecimalFormat())
        {
            private static final long serialVersionUID = 1L;

            @Override
            public void commitEdit() throws ParseException
            {
                try
                {
                    Double.parseDouble(getText());
                }
                catch(final NumberFormatException e)
                {
                    throw new ParseException("Unable to parse number.", 0);
                }
                super.commitEdit();
            }
        });
        setClickCountToStart(1);
        buildFormatter("");
    }

    /**
     * @param numberFormatPattern The number format pattern to use when editing and displaying numbers.
     */
    public NumberCellEditor(final String numberFormatPattern)
    {
        this();
        buildFormatter(numberFormatPattern);
    }

    public NumberFormatter getFormatter()
    {
        return _formatter;
    }

    @Override
    public Object getCellEditorValue()
    {
        Object value = super.getCellEditorValue();
        if(value instanceof String)
        {
            try
            {
                value = getFormatter().getFormat().parseObject((String)value);
            }
            catch(final ParseException e)
            {
            }
        }
        return value;
    }

    private void buildFormatter(final String numberFormatPattern)
    {
        if(numberFormatPattern == null)
        {
            _formatter = new NumberFormatter();
        }
        else
        {
            _formatter = new NumberFormatter(new DecimalFormat(numberFormatPattern));
        }
    }

    @Override
    public Component getTableCellEditorComponent(final JTable table,
                                                 final Object value,
                                                 final boolean isSelected,
                                                 final int row,
                                                 final int column)
    {
        final JFormattedTextField editor = (JFormattedTextField)super.getTableCellEditorComponent(table,
                                                                                                  value,
                                                                                                  isSelected,
                                                                                                  row,
                                                                                                  column);

        if(value != null)
        {
            editor.setFormatterFactory(new DefaultFormatterFactory(getFormatter()));
//            Number num = (Number)value;
//            String text = numberFormat.format(num);
            editor.setHorizontalAlignment(SwingConstants.LEFT);
//            editor.setText(text);
            editor.setValue(value);

            //Invoke later must be used because, after this is called, the formatter is reset on the field
            //causing the caret to be reset again and everything to be deselected.
            editor.addFocusListener(new FocusAdapter()
            {
                @Override
                public void focusGained(final FocusEvent e)
                {
                    SwingUtilities.invokeLater(new Runnable()
                    {
                        @Override
                        public void run()
                        {
                            editor.selectAll();
                        }
                    });
                }
            });

        }
        return editor;
    }

    /**
     * @return A {@link DefaultTableRenderer} (SwingX) that uses the _num
     */
    public TableCellRenderer constructCorrespondingRenderer()
    {
        return new DefaultTableRenderer()
        {
            private static final long serialVersionUID = 1L;

            @Override
            public Component getTableCellRendererComponent(final JTable table,
                                                           final Object value,
                                                           final boolean isSelected,
                                                           final boolean hasFocus,
                                                           final int row,
                                                           final int column)
            {
                // Exit out early if not a number.
                if(value == null || !(value instanceof Number))
                {
                    return super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
                }

                final String valueStr = NumberTools.formatNumber((NumberFormat)getFormatter().getFormat(),
                                                                 (Number)value);
                return super.getTableCellRendererComponent(table, valueStr, isSelected, hasFocus, row, column);
            }

        };
    }

    @Override
    public boolean stopCellEditing()
    {
        final Object value = ((JFormattedTextField)this.getComponent()).getValue();
        try
        {
            ((JFormattedTextField)this.getComponent()).commitEdit();
        }
        catch(final ParseException e)
        {
            ((JFormattedTextField)this.getComponent()).setValue(value);
        }
        return super.stopCellEditing();
    }
}
