package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;

import ohd.hseb.hefs.utils.arguments.ArgumentInsertingTextFieldComboBoxEditor;
import ohd.hseb.hefs.utils.gui.jtable.models.RowCheckableTableModel;
import ohd.hseb.hefs.utils.gui.jtable.renderers.JLabelTableCellRenderer;
import ohd.hseb.hefs.utils.gui.tools.ComboBoxTools;
import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.tools.IconTools;

/**
 * General utilities designed to work with the JTable and the other contents of this package.
 * 
 * @author hank.herr
 * @author Alexander.Garbarino
 */
public abstract class TableTools
{

    public static final Integer DEFAULT_DELETE_BUTTON_WIDTH = 16;
    public static final Integer DEFAULT_INDEX_COLUMN_WIDTH = 25;
    public static final String DELETE_JLABEL_NAME = "DELETE ICON LABEL";
    public static final String ADD_JLABEL_NAME = "ADD ICON LABEL";
    private static Integer _deleteButtonFixedWidth = null;
    private static Integer _indexColumnFixedWidth = null;

    /**
     * Calls {@link #createAddIconLabel(String)} passing in null for no tool tip.
     * 
     * @return JLabel displaying delete icon.
     */
    public static JLabel createDeleteIconLabel()
    {
        return createDeleteIconLabel(null);
    }

    /**
     * @param toolTipText Tool tip text for the {@link JLabel}.
     * @return JLabel displaying a delete icon.
     */
    public static JLabel createDeleteIconLabel(final String toolTipText)
    {
        final String iconImageFile = "hefsIcons/fewsTableDeleteRow.png";

        final URL url = ClassLoader.getSystemResource(iconImageFile);
        JLabel deleteLabel;
        if(url != null)
        {
            final ImageIcon icon = new ImageIcon(url);
            deleteLabel = new JLabel(icon);
        }
        else
        {
            deleteLabel = new JLabel("Del");
        }

        deleteLabel.setName(DELETE_JLABEL_NAME);
        deleteLabel.setToolTipText(toolTipText);
        return deleteLabel;
    }

    /**
     * @param toolTipText Tool tip text for the {@link JLabel}.
     * @return JLabel displaying a add icon.
     */
    public static JLabel createAddIconLabel(final String toolTipText)
    {
        final String iconImageFile = "hefsIcons/fewsTableAddRow.png";

        final URL url = ClassLoader.getSystemResource(iconImageFile);
        JLabel deleteLabel;
        if(url != null)
        {
            final ImageIcon icon = new ImageIcon(url);
            deleteLabel = new JLabel(icon);
        }
        else
        {
            deleteLabel = new JLabel("Add");
        }

        deleteLabel.setName(ADD_JLABEL_NAME);
        deleteLabel.setToolTipText(toolTipText);
        return deleteLabel;
    }

    /**
     * @param label The {@link TableTools} to check.
     * @return True if the label is a add label created by createDeleteIconLabel in this class. It checks if the
     *         getName() method for the label returns {@link #ADD_JLABEL_NAME}.
     */
    public static boolean isJLabelAnAddLabel(final JLabel label)
    {
        return (label.getName() != null) && label.getName().equals(ADD_JLABEL_NAME);
    }

    /**
     * @param label The {@link TableTools} to check.
     * @return True if the label is a delete label created by createDeleteIconLabel in this class. It checks if the
     *         getName() method for hte label returns {@link #DELETE_JLABEL_NAME}.
     */
    public static boolean isJLabelADeleteLabel(final JLabel label)
    {
        return (label.getName() != null) && label.getName().equals(DELETE_JLABEL_NAME);
    }

    /**
     * @return A fixed width value to use for the delete icon.
     */
    public static Integer getDeleteButtonFixedWidth()
    {
        if(_deleteButtonFixedWidth == null)
        {
            _deleteButtonFixedWidth = DEFAULT_DELETE_BUTTON_WIDTH;
        }
        return _deleteButtonFixedWidth;
    }

    /**
     * @param deleteButtonFixedWidth Width to use for the delete icon buttons in a table. This will rarely need to be
     *            called.
     */
    public static void setDeleteButtonFixedWidth(final Integer deleteButtonFixedWidth)
    {
        _deleteButtonFixedWidth = deleteButtonFixedWidth;
    }

    /**
     * @param indexColumnFixedWidth A width to use for index columns. This will rarely need to be called.
     */
    public static void setIndexColumnFixedWidth(final Integer indexColumnFixedWidth)
    {
        _indexColumnFixedWidth = indexColumnFixedWidth;
    }

    /**
     * @return A width to use for index columns. Default should be big enough for two digits.
     */
    public static Integer getIndexColumnFixedWidth()
    {
        if(_indexColumnFixedWidth == null)
        {
            _indexColumnFixedWidth = DEFAULT_INDEX_COLUMN_WIDTH;
        }
        return _indexColumnFixedWidth;
    }

    /**
     * @return The fixed width to use for a status column, displaying a icon the same size as the delete icon.
     */
    public static Integer getStatusColumnFixedWidth()
    {
        return DEFAULT_DELETE_BUTTON_WIDTH;
    }

    /**
     * @param state Status to use: true for 'good' false for 'bad', or null for unknown.
     * @return An ImageIcon that displays the status as a check mark (true), x (false), or exclamation mark (null).
     */
    public static Icon createStatusIcon(final Boolean state)
    {
        if(state == null)
        {
            return IconTools.getHSEBIcon("yellowExclamation");
        }
        else if(state == false)
        {
            return IconTools.getHSEBIcon("redX");
        }
        else
        {
            return IconTools.getHSEBIcon("checkMark");
        }
    }

    /**
     * @param state Status to use: true for 'good' false for 'bad', or null for unknown.
     * @param toolTipText Text for the tool tip, or null if none.
     * @return A JLabel containing the image icon output by createStatusIcon. The name of the JLabel will be the value
     *         of state.toString() and can be used to determine the boolean value associated with the label.
     */
    public static JLabel createStatusIconLabel(final Boolean state, final String toolTipText)
    {
        final Icon icon = createStatusIcon(state);
        if(icon != null)
        {
            final JLabel label = new JLabel(icon);
            label.setToolTipText(toolTipText);
            label.setName("" + state);
            return label;
        }
        return null;
    }

    /**
     * Short cut to set a fixed width for a column. It sets the preferred width, min width, and max width on the passed
     * in column.
     * 
     * @param col The TableColumn to set.
     * @param width The width to use.
     */
    public static void setColumnFixedWidth(final TableColumn col, final int width)
    {
        col.setPreferredWidth(width);
        col.setMinWidth(width);
        col.setMaxWidth(width);
    }

    /**
     * @param table The table in which to scroll.
     * @param row The view row to scroll to.
     */
    public static void scrollToRow(final JTable table, final int row)
    {
        SwingUtilities.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                table.scrollRectToVisible(table.getCellRect(row, 0, true));
            }
        });
    }

    /**
     * @param table The table in which to scroll.
     * @param col The view column to scroll to.
     */
    public static void scrollToColumn(final JTable table, final int col)
    {
        SwingUtilities.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                table.scrollRectToVisible(table.getCellRect(0, col, true));
            }
        });
    }

    /**
     * This sets selection interval per row, which is why it can handle selecting any rows, contiguous or not.
     * {@link ListSelectionEvent} instances thrown as a result of this will have the value is adjusting flag set
     * appropriately.
     * 
     * @param table The table in which to select arbitrary rows.
     * @param rows The rows to select; may be contiguous or not. Negative values are skipped.
     */
    public static void selectRows(final JTable table, final int[] rows)
    {
        //Remove any negative rows.
        final List<Integer> trimmedRows = new ArrayList<Integer>();
        for(final int row: rows)
        {
            if(row >= 0)
            {
                trimmedRows.add(row);
            }
        }

        for(int i = 0; i < trimmedRows.size(); i++)
        {
            //First change -- set value is adjust to true.  For th last change, always make it false,
            //even if the last change is also the first change (i.e., one row to select).
            if(i == 0)
            {
                table.getSelectionModel().setValueIsAdjusting(true);
            }
            if(i == trimmedRows.size() - 1)
            {
                table.getSelectionModel().setValueIsAdjusting(false);
            }

            //Set the selection interval.
            if(i == 0)
            {
                table.setRowSelectionInterval(trimmedRows.get(i), trimmedRows.get(i));
            }
            else
            {
                table.addRowSelectionInterval(trimmedRows.get(i), trimmedRows.get(i));
            }
        }
    }

    /**
     * Selects a row and scrolls to it.
     * 
     * @param table The table to affect.
     * @param row The view row to select.
     */
    public static void selectRowAndScrollToIt(final JTable table, final int row)
    {
//XXX The two lines below used to be within an invoke later.
//Is it necessary to do so?  No, because scrollToRow is doing it already.
//        SwingUtilities.invokeLater(new Runnable()
//        {
//            @Override
//            public void run()
//            {
        table.getSelectionModel().setSelectionInterval(row, row);
        scrollToRow(table, row);
//            }
//        });
    }

    /**
     * Selects and scrolls to the last row of a table. Useful if you add a new row and want to select it.
     * 
     * @param table The table to affect.
     */
    public static void selectAndScrollToLastRow(final JTable table)
    {
        final int lastRowIndex = table.getRowCount() - 1;
        selectRowAndScrollToIt(table, lastRowIndex);
    }

    /**
     * Performs all tasks related to creating a delete column outside of the TableModel. The TableModel must return a
     * delete icon label for the delete column. To do so, have getValue return this:<br>
     * <br>
     * {@link #createDeleteIconLabel(String)} <br>
     * This sets the default renderer for JLabel cells to JLabelTableCellRenderer(). It also sets the column width to a
     * fixed width returned by {@link #getDeleteButtonFixedWidth()}.
     * 
     * @param table The table that possesses the delete column.
     * @param columnIndex The index of the delete column
     */
    public static void addAddDeleteColumn(final JTable table,
                                          final int columnIndex,
                                          final JTableAddProcessor addProcessor,
                                          final JTableDeleteProcessor deleteProcessor)
    {
        table.addMouseListener(new JTableAddDeleteMouseListener(table, deleteProcessor, addProcessor));
        table.setDefaultRenderer(JLabel.class, new JLabelTableCellRenderer());
        TableTools.setColumnFixedWidth(table.getColumnModel().getColumn(0), TableTools.getDeleteButtonFixedWidth());
    }

    /**
     * Creates a panel for a JTable, and puts both it and its header inside. Useful for when you don't want a scrolling
     * table, but still want the header.
     * 
     * @param table the table to encapsulate
     * @return a new panel holding the given table and its header
     */
    public static JPanel wrapInPanel(final JTable table)
    {
        return wrapInPanel(table, null);
    }

    /**
     * Creates a panel for a JTable, and puts both it and its header inside. Useful for when you don't want a scrolling
     * table, but still want the header. May also give it a titled border.
     * 
     * @param table the table to encapsulate
     * @param title the title of the border. If set to null, then there will be no border at all.
     * @return a new panel holding the given table and its header
     */
    public static JPanel wrapInPanel(final JTable table, final String title)
    {
        final JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        panel.add(table.getTableHeader(), BorderLayout.NORTH);
        panel.add(table, BorderLayout.CENTER);
        if(title != null)
        {
            panel.setBorder(HSwingFactory.createTitledBorder(BorderFactory.createEtchedBorder(1), title, null));
        }
        return panel;
    }

    /**
     * Reselect the currently selected rows in the table, essentially triggering a {@link ListSelectionEvent} for the
     * table selection model. The validIsAdjusting flag will be set appropriately.
     * 
     * @param table
     */
    public static void reselectRows(final JTable table)
    {
        final int[] rows = table.getSelectedRows();
        table.clearSelection();
        if(rows.length > 0)
        {
            selectRows(table, rows);
        }
    }

    /**
     * Creates a {@link JComboBox} table cell editor with a {@link ArgumentInsertingTextFieldComboBoxEditor} editor. The
     * {@link ArgumentInsertingTextFieldComboBoxEditor} editor is used because it has a built in focus listener. It is
     * given null arguments so that they are not applied.
     * 
     * @param comboBoxItems The items to add to the combo box.
     * @return A DefaultCellEditor with the JComboBox editor set.
     */
    public static DefaultCellEditor buildFocusListeningComboBoxEditor(final Collection comboBoxItems)
    {
        //The ArgumentInsertingTextFieldComboBoxEditor is used because it has a focus listener built in.
        //If null is given to it, it becomes a simple text field editor with a focus listener.
        final JComboBox comboBoxEditor = new JComboBox();
        comboBoxEditor.setEditable(true);
        ComboBoxTools.addAllItems(comboBoxEditor, comboBoxItems);
        final ArgumentInsertingTextFieldComboBoxEditor editor = new ArgumentInsertingTextFieldComboBoxEditor(null);
        comboBoxEditor.setEditor(editor);
        return new DefaultCellEditor(comboBoxEditor);
    }

    /**
     * Creates a {@link JComboBox} table cell editor.
     * 
     * @param comboBoxItems The items to add to the combo box.
     * @return A DefaultCellEditor with the JComboBox editor set.
     */
    public static DefaultCellEditor buildComboBoxEditor(final Collection comboBoxItems)
    {
        //The ArgumentInsertingTextFieldComboBoxEditor is used because it has a focus listener built in.
        //If null is given to it, it becomes a simple text field editor with a focus listener.
        final JComboBox comboBoxEditor = new JComboBox();
        comboBoxEditor.setEditable(true);
        ComboBoxTools.addAllItems(comboBoxEditor, comboBoxItems);
        final ArgumentInsertingTextFieldComboBoxEditor editor = new ArgumentInsertingTextFieldComboBoxEditor(null);
        comboBoxEditor.setEditor(editor);
        return new DefaultCellEditor(comboBoxEditor);
    }

    /**
     * Converts, in place, all of the rows given to the {@link TableModel} of the given {@link JTable}. It calls
     * {@link JTable#convertRowIndexToModel(int)}.
     * 
     * @param table The table.
     * @param rows The rows to convert. This array is modified by this call.
     * @return The converted rows. It is the same int[] as the rows given.
     */
    public static int[] convertAllRowsToModel(final JTable table, final int[] rows)
    {
        for(int i = 0; i < rows.length; i++)
        {
            rows[i] = table.convertRowIndexToModel(rows[i]);
        }
        return rows;
    }

    /**
     * @param table The table.
     * @return All selected rows converted to the model via {@link JTable#convertRowIndexToModel(int)}
     */
    public static int[] getSelectedRowsConvertedToModel(final JTable table)
    {
        return convertAllRowsToModel(table, table.getSelectedRows());
    }

    /**
     * For use with {@link RowCheckableTableModel}, this returns a button for checking all rows.
     * 
     * @param model The model to which this button applies.
     * @return A button ready for use.
     */
    public static JButton createCheckAllButton(final RowCheckableTableModel model, final String toolTip)
    {
        return HSwingFactory.createJButtonWithIcon("hefsIcons/checkAllRows20x20.png", toolTip, new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                model.checkAll();
            }
        });
    }

    /**
     * For use with {@link RowCheckableTableModel}, this returns a button for unchecking all rows.
     * 
     * @param model The model to which this button applies.
     * @return A button ready for use.
     */
    public static JButton createUncheckAllButton(final RowCheckableTableModel model, final String toolTip)
    {
        return HSwingFactory.createJButtonWithIcon("hefsIcons/uncheckAllRows20x20.png", toolTip, new ActionListener()
        {
            @Override
            public void actionPerformed(final ActionEvent e)
            {
                model.uncheckAll();
            }
        });
    }

    /**
     * For use with {@link RowCheckableTableModel}, this returns a button for checking all rows.
     * 
     * @param model The model to which this button applies.
     * @return A button ready for use.
     */
    public static JButton createCheckSelectedButton(final RowCheckableTableModel model, final String toolTip)
    {
        return HSwingFactory.createJButtonWithIcon("hefsIcons/checkSelectedRows20x20.png",
                                                   toolTip,
                                                   new ActionListener()
                                                   {
                                                       @Override
                                                       public void actionPerformed(final ActionEvent e)
                                                       {
                                                           model.checkSelected();
                                                       }
                                                   });
    }

    /**
     * For use with {@link RowCheckableTableModel}, this returns a button for checking all rows.
     * 
     * @param model The model to which this button applies.
     * @return A button ready for use.
     */
    public static JButton createUncheckSelectedButton(final RowCheckableTableModel model, final String toolTip)
    {
        return HSwingFactory.createJButtonWithIcon("hefsIcons/uncheckSelectedRows20x20.png",
                                                   toolTip,
                                                   new ActionListener()
                                                   {
                                                       @Override
                                                       public void actionPerformed(final ActionEvent e)
                                                       {
                                                           model.uncheckSelected();
                                                       }
                                                   });
    }

    /**
     * The entire table is dumped, from row 0 to the last row and column 0 to the last column, regardless of which
     * colums or rows are visible in the table.
     * 
     * @param table The {@link JTable} whose data is to be dumped.
     * @param file The file to create.
     * @param separator The separator character sequence to use.
     * @throws IOException Standard reasons.
     */
    public static void dumpTableToFile(final JTable table, final File file, final CharSequence separator) throws IOException
    {
        final FileWriter writer = new FileWriter(file);
        try
        {
            for(int row = 0; row < table.getModel().getRowCount(); row++)
            {
                String rowStr = "";
                for(int col = 0; col < table.getColumnCount(); col++)
                {
                    rowStr += "" + table.getValueAt(row, col);
                    if(col < table.getColumnCount() - 1)
                    {
                        rowStr += separator.toString();
                    }
                }
                rowStr += "\n";
                writer.write(rowStr);
            }
        }
        finally
        {
            writer.close();
        }
    }
}
