package ohd.hseb.hefs.utils.gui.jtable;

import java.awt.Component;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;

import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;

import ohd.hseb.hefs.utils.gui.jtable.models.ToolTipTableModel;

import org.jdesktop.swingx.JXTable;

import com.google.common.base.Strings;

/**
 * General table allows for tool tips to be provided by the underlying model. If the underlying table model does not
 * implement {@link ToolTipTableModel}, then this class does nothing unless
 * {@link #setReturnCellContentsByDefault(boolean)} is called with true passed in. In that case, the cell and header
 * contents are returned by default. If the model does implement {@link ToolTipTableModel} and
 * {@link #_returnCellContentsByDefault} is true, then the tool tip interface methods are given preference. If they
 * return null, then the cell and header contents will be returned. <br>
 * <br>
 * IMPORTANT NOTE: This class keeps track of the indices of the TableColumns in the original TableColumnModel (i.e.,
 * based on the the first time setColumnModel is called). This allows for columns to be removed from the table column
 * model and the tool tips to be kept in line since the {@link ToolTipTableModel} generates tool tips based on the
 * column index. After initialization, you should only change the {@link TableColumnModel} via removes; no other
 * modification is allowed. The copy of the {@link #_originalColumnToColumnIndexMap} is reset whenever setModel is
 * called.
 * 
 * @author hank.herr
 */
public class ToolTipJTable extends JXTable
{
    private static final long serialVersionUID = 1L;

    /**
     * If true the if the model does not implement {@link ToolTipTableModel} or the
     * {@link ToolTipTableModel#getColumnHeaderToolTip(int)} or {@link ToolTipTableModel#getCellToolTip(int, int)}
     * return null, then the cell/header's contents will displayed in the tool tip. This may be useful if the column
     * width can be too narrow to view the full header/cell contents.
     */
    private boolean _returnCellContentsByDefault = true;

    public ToolTipJTable()
    {
        super();
    }

    public ToolTipJTable(final TableModel model)
    {
        super(model);
    }

    /**
     * Default tip to display is the header text in full, which is useful if the columns are too narrow to view the
     * header.
     * 
     * @param header The header for which to display a tip.
     * @param col The table column for which to display a tip.
     */
    private void displayDefaultHeaderToolTip(final JTableHeader header, final TableColumn col)
    {
        //Get the default value if desired.
        if(col != null)
        {
            final Object colHeaderValue = col.getHeaderValue();
            if(colHeaderValue instanceof String)
            {
                header.setToolTipText((String)colHeaderValue);
            }
        }
    }

    /**
     * @param rowIndex
     * @param colIndex
     * @param cellValue
     * @return The default cell tool tip, which is just the contents of the cell, if the contents are displayed as a
     *         {@link JLabel}. Otherwise it returns null, indicating no tool tip.
     */
    private String retrieveDefaultCellTooTip(final int rowIndex, final int colIndex, final Object cellValue)
    {
        //Default tool tip will redisplay the contents of the cell if the cell renderer is a JLabel
        if(rowIndex >= 0)
        {
            //The modelColIndex only applies to the call to getValueAt.  The colIndex for getCellRenderer
            //and getTableCellRendererComponent are relative to the visible table (not the underlying model).
            final Component c = getCellRenderer(rowIndex, colIndex).getTableCellRendererComponent(this,
                                                                                                  cellValue,
                                                                                                  false,
                                                                                                  false,
                                                                                                  rowIndex,
                                                                                                  colIndex);
            if(c instanceof JLabel)
            {
                return ((JLabel)c).getText();
            }
        }

        return null;
    }

    public int getModelColumnIndex(final MouseEvent e)
    {
        final int index = columnAtPoint(e.getPoint());
        if(index != -1)
        {
            return convertColumnIndexToModel(index);
        }
        return index;
    }

    public int getModelRowIndex(final MouseEvent e)
    {
        final int index = rowAtPoint(e.getPoint());
        if(index != -1)
        {
            return convertRowIndexToModel(index);
        }
        return index;
    }

    /**
     * See {@link #_returnCellContentsByDefault}.
     */
    public void setReturnCellContentsByDefault(final boolean returnCellContentsByDefault)
    {
        _returnCellContentsByDefault = returnCellContentsByDefault;
    }

    @Override
    public JTableHeader createDefaultTableHeader()
    {
        final JTableHeader header = super.createDefaultTableHeader();

        //Mouse motion listener is used to display header tool tips.
        header.addMouseMotionListener(new MouseMotionAdapter()
        {
            private TableColumn _curCol = null;

            @Override
            public void mouseMoved(final MouseEvent evt)
            {
                final JTableHeader header = (JTableHeader)evt.getSource();
                final JTable table = header.getTable();
                final TableColumnModel colModel = table.getColumnModel();

                //Get the column clicked on and its index relative to the model.
                final int colIndex = columnAtPoint(evt.getPoint());
                if(colIndex < 0)
                {
                    return;
                }
                final TableColumn col = colModel.getColumn(colIndex);
                final int modelColIndex = col.getModelIndex();

                //If the column has changed...
                if(col != _curCol)
                {
                    _curCol = col;

                    if(getModel() instanceof ToolTipTableModel)
                    {
                        final ToolTipTableModel model = (ToolTipTableModel)getModel();
                        final String tip = model.getColumnHeaderToolTip(modelColIndex);
                        if(!Strings.isNullOrEmpty(tip))
                        {
                            header.setToolTipText(tip);
                            return;
                        }
                    }

                    if(_returnCellContentsByDefault)
                    {
                        displayDefaultHeaderToolTip(header, col);
                    }
                }
            }
        });
        return header;
    }

    /**
     * Normally, if set to null, this will disable tool tips. This cancels that behavior.
     */
    @Override
    public void setToolTipText(final String text)
    {
        super.setToolTipText(Strings.nullToEmpty(text));
    }

    @Override
    public String getToolTipText(final MouseEvent e)
    {
        final TableModel model = this.getModel();
        final int visRowIndex = rowAtPoint(e.getPoint());
        final int colIndex = columnAtPoint(e.getPoint());
        if(visRowIndex < 0)
        {
            return null;
        }
        final int modelRowIndex = convertRowIndexToModel(visRowIndex);

        // Return if not clicked on any column header 
        TableColumn col = null;
        if(colIndex >= 0)
        {
            col = getColumnModel().getColumn(colIndex);
        }

        //Find the column in the table model.
        final int modelColIndex = col.getModelIndex();

        //Get the cell value.
        final Object cellValue = model.getValueAt(modelRowIndex, modelColIndex);

        //Return the tool tip from the model if one is given.
        if(model instanceof ToolTipTableModel)
        {
            final String toolTip = ((ToolTipTableModel)model).getCellToolTip(modelRowIndex, modelColIndex);
            if(!Strings.isNullOrEmpty(toolTip))
            {
                return toolTip;
            }
        }

        //Otherwise, return the default tool tip which depends on the default flag.
        if(_returnCellContentsByDefault)
        {
            return retrieveDefaultCellTooTip(modelRowIndex, colIndex, cellValue);
        }
        else
        {
            return super.getToolTipText(e);
        }
    }

}
