package ohd.hseb.hefs.utils.gui.jtable.models;

import javax.swing.JLabel;
import javax.swing.table.TableModel;

import ohd.hseb.hefs.utils.gui.jtable.TableTools;

/**
 * Wraps a {@link AddDeleteRowTableModel}, adding the needed column for deleting and adding rows. Usage should be as
 * follows:<br>
 * <br>
 * 1. Create a subclass of {@link AddDeleteTableModelWrapper}.<br>
 * 2. Within it, create an inner class {@link TableModel} that implements {@link AddDeleteRowTableModel} and any other
 * specialized model interfaces needed.<br>
 * 3. In the constructor for the subclass of {@link AddDeleteTableModelWrapper}, pass an instance of the inner
 * {@link TableModel} to the constructor for this, and specify tool tips to use.<br>
 * <br>
 * Note that the delete icons are added for every row. However, the extra add row is only added if the tool tip provided
 * in the constructor is not null.
 * 
 * @author Hank.Herr
 */
@SuppressWarnings("serial")
public class AddDeleteTableModelWrapper extends ForwardingTableModel
{
    private final AddDeleteRowTableModel _delegate;
    private final String _addString;
    private final String _deleteString;

    /**
     * Create a wrapper around a table model to which adds an add/delete column and row.
     * 
     * @param delegate the table model to wrap
     * @param addToolTip The tool tip for the add button. If null is given, no add row is provided.
     * @param deleteToolTip The tool tip for the delete button
     */
    public AddDeleteTableModelWrapper(AddDeleteRowTableModel delegate, String addToolTip, String deleteToolTip)
    {
        _delegate = delegate;
        _addString = addToolTip;
        _deleteString = deleteToolTip;
    }

    /**
     * Create a wrapper around a table model to which adds a delete column (not an add row).
     * 
     * @param delegate the table model to wrap
     * @param deleteString the tool tip for the delete button
     */
    public AddDeleteTableModelWrapper(AddDeleteRowTableModel delegate, String deleteString)
    {
        this(delegate, null, deleteString);
    }

    @Override
    public AddDeleteRowTableModel getDelegate()
    {
        return _delegate;
    }

    @Override
    public int getRowCount()
    {
        if(_addString == null)
        {
            return _delegate.getRowCount();
        }
        return _delegate.getRowCount() + 1;
    }

    @Override
    public int getColumnCount()
    {
        return _delegate.getColumnCount() + 1;
    }

    @Override
    public String getColumnName(int columnIndex)
    {
        if(columnIndex == 0)
        {
            return "";
        }
        else
        {
            return _delegate.getColumnName(columnIndex - 1);
        }
    }

    @Override
    public Class<?> getColumnClass(int columnIndex)
    {
        if(columnIndex == 0)
        {
            return JLabel.class;
        }
        else
        {
            return _delegate.getColumnClass(columnIndex - 1);
        }
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex)
    {
        if(columnIndex == 0 || rowIndex == _delegate.getRowCount())
        {
            return false;
        }
        else
        {
            return _delegate.isCellEditable(rowIndex, columnIndex - 1);
        }
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex)
    {
        if(columnIndex == 0)
        {
            if(rowIndex == _delegate.getRowCount())
            {
                return TableTools.createAddIconLabel(_addString);
            }
            else
            {
                return TableTools.createDeleteIconLabel(_deleteString);
            }
        }
        else if(rowIndex == _delegate.getRowCount())
        {
            return null;
        }
        else
        {
            return _delegate.getValueAt(rowIndex, columnIndex - 1);
        }
    }

    @Override
    public void setValueAt(Object aValue, int rowIndex, int columnIndex)
    {
        if(rowIndex != _delegate.getRowCount() && columnIndex > 0)
        {
            _delegate.setValueAt(aValue, rowIndex, columnIndex - 1);
        }
    }

    @Override
    public String getColumnHeaderToolTip(int index)
    {
        if(index == 0)
        {
            return null;
        }
        else
        {
            return super.getColumnHeaderToolTip(index - 1);
        }
    }

    @Override
    public String getCellToolTip(int rowIndex, int columnIndex)
    {
        if(columnIndex == 0)
        {
            if(rowIndex == _delegate.getRowCount())
            {
                return _addString;
            }
            else
            {
                return _deleteString;
            }
        }
        else if(rowIndex == _delegate.getRowCount())
        {
            return null;
        }
        else
        {
            return super.getCellToolTip(rowIndex, columnIndex - 1);
        }
    }

    @Override
    public Integer getPreferredWidth(int column)
    {
        if(column == 0)
        {
            return TableTools.getDeleteButtonFixedWidth();
        }
        else
        {
            return super.getPreferredWidth(column - 1);
        }
    }

    @Override
    public Integer getMaxWidth(int column)
    {
        if(column == 0)
        {
            return TableTools.getDeleteButtonFixedWidth();
        }
        else
        {
            return super.getMaxWidth(column - 1);
        }
    }

    @Override
    public Integer getMinWidth(int column)
    {
        if(column == 0)
        {
            return TableTools.getDeleteButtonFixedWidth();
        }
        else
        {
            return super.getMinWidth(column - 1);
        }
    }
}
