package ohd.hseb.hefs.utils.gui.jtable.renderers;

import java.awt.Color;
import java.awt.Component;
import java.util.HashMap;
import java.util.List;

import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * Modifies an existing {@link TableCellRenderer} to color cells based on a sequence of coloring. This assumes all
 * columns are colored in one color, event columns preceding {@link #_firstColumnToColor}.
 * 
 * @author alexander.garbarino
 */
public class AlternatingColumnColorTableCellRenderer extends ForwardingTableCellRenderer
{
    private final List<Color> _colorSequence;
    private final int _numberOfColumnsPerColor;
    private final int _firstColumnToColor;

    /**
     * Because on {@link Component} is used to render all cells, I need to remember the appropriate color to use for
     * each column preceding the {@link #_firstColumnToColor}. This map stores that info.
     */
    private final HashMap<Integer, Color> _columnToDefaultColorMap = Maps.newHashMap();

    public AlternatingColumnColorTableCellRenderer(final TableCellRenderer delegate,
                                                   final int firstColumnToColor,
                                                   final int numberOfColumnsPerColor,
                                                   final Color... colors)
    {
        super(delegate);
        _firstColumnToColor = firstColumnToColor;
        _numberOfColumnsPerColor = numberOfColumnsPerColor;
        _colorSequence = Lists.newArrayList(colors);
    }

    @Override
    public Component getTableCellRendererComponent(final JTable table,
                                                   final Object value,
                                                   final boolean isSelected,
                                                   final boolean hasFocus,
                                                   final int row,
                                                   final int column)
    {
        final Component component = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
//        if(component == null) //return null for a null component so that the empty renderer can be applied.
//        {
//            return null;
//        }

        if(!isSelected)
        {
            if(column < _firstColumnToColor)
            {
                Color bgColor = _columnToDefaultColorMap.get(column);
                if(bgColor == null)
                {
                    bgColor = component.getBackground();
                    _columnToDefaultColorMap.put(column, bgColor);
                }
                component.setBackground(bgColor);
                return component;
            }

            final int workingColumn = column - _firstColumnToColor;
            final int sequenceIndex = (int)((double)(workingColumn) / (double)_numberOfColumnsPerColor);
            final int colorIndex = sequenceIndex % _colorSequence.size();
            component.setBackground(_colorSequence.get(colorIndex));
        }

        return component;
    }
}
