package ohd.hseb.hefs.utils.gui.jtable.renderers;

import java.awt.Color;
import java.awt.Component;
import java.util.List;

import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;

import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.Triad;
import ohd.hseb.hefs.utils.effect.Effect;
import ohd.hseb.hefs.utils.gui.tools.ColorTools;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;

/**
 * Renderer allows for a series of {@link Predicate}s to be applied to a component, each having an associated
 * {@link Effect} if the predicate is true and an effect if false. See {@link #addEffect(Predicate, Effect, Effect)} for
 * details about the predicate.
 * 
 * @author hankherr
 */
public class PredicateTableCellRenderer extends ForwardingTableCellRenderer
{

    private final List<Triad<Predicate<Dyad<Integer, Integer>>, Effect<Component>, Effect<Component>>> _conditions = Lists.newArrayList();

    public PredicateTableCellRenderer(final TableCellRenderer delegate)
    {
        super(delegate);
    }

    /**
     * If a provided {@link Effect} is null, then it will not be applied.
     * 
     * @param predicate Its method {@link Predicate#apply(Object)} is called to determine if the condition is true or
     *            false. The {@link Dyad} provided to the predicate is of two integers, the first being the view row and
     *            the second being the view column for the table. For example, call {@link Dyad#getFirst()} to get the
     *            row and use {@link JTable#convertRowIndexToModel(int)} to get the table model row.
     * @param trueEffect The {@link Effect} to apply if true. Null is allowed.
     * @param falseEffect The {@link Effect} to apply if false. Null is allowed.
     */
    public void addEffect(final Predicate<Dyad<Integer, Integer>> predicate,
                          final Effect<Component> trueEffect,
                          final Effect<Component> falseEffect)
    {
        _conditions.add(new Triad<Predicate<Dyad<Integer, Integer>>, Effect<Component>, Effect<Component>>(predicate,
                                                                                                           trueEffect,
                                                                                                           falseEffect));
    }

    /**
     * Calls {@link #addEffect(Predicate, Effect, Effect)} passing in null for the false effect.
     */
    public void addTrueEffect(final Predicate<Dyad<Integer, Integer>> predicate, final Effect<Component> trueEffect)
    {
        _conditions.add(new Triad<Predicate<Dyad<Integer, Integer>>, Effect<Component>, Effect<Component>>(predicate,
                                                                                                           trueEffect,
                                                                                                           null));
    }

    /**
     * Calls {@link #addEffect(Predicate, Effect, Effect)} passing in null for the true effect.
     */
    public void addFalseEffect(final Predicate<Dyad<Integer, Integer>> predicate, final Effect<Component> falseEffect)
    {
        _conditions.add(new Triad<Predicate<Dyad<Integer, Integer>>, Effect<Component>, Effect<Component>>(predicate,
                                                                                                           null,
                                                                                                           falseEffect));
    }

    @Override
    public Component getTableCellRendererComponent(final JTable table,
                                                   final Object value,
                                                   final boolean isSelected,
                                                   final boolean hasFocus,
                                                   final int row,
                                                   final int column)
    {
        final Component c = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

        if(c == null)
        {

        }
        for(final Triad<Predicate<Dyad<Integer, Integer>>, Effect<Component>, Effect<Component>> condition: _conditions)
        {
            if(condition.getFirst().apply(new Dyad<Integer, Integer>(row, column)))
            {
                if(condition.getSecond() != null)
                {
                    condition.getSecond().perform(c);
                }
            }
            else
            {
                if(condition.getThird() != null)
                {
                    condition.getThird().perform(c);
                }
            }
        }
        return c;
    }

    /**
     * Generic effect for coloring the foreground of a component. The {@link #_mix} attribute indicates if the applied
     * color will be mixed with the existing color (true) or applied to the component directly without mixing (false)
     * 
     * @author hankherr
     */
    public static class ForegroundColorEffect extends Effect<Component>
    {

        private final Color _c;
        private final boolean _mix;

        public ForegroundColorEffect(final Color c, final boolean mix)
        {
            _c = c;
            _mix = mix;
        }

        @Override
        public void perform(final Component input)
        {
            if(_mix)
            {
                input.setForeground(ColorTools.mixColors(input.getForeground(), _c));
            }
            else
            {
                input.setForeground(_c);
            }
        }
    }

    public static ForegroundColorEffect createForegroundEffect(final Color c, final boolean mix)
    {
        return new ForegroundColorEffect(c, mix);
    }

    /**
     * Generic effect for coloring the background of a component. The {@link #_mix} attribute indicates if the applied
     * color will be mixed with the existing color (true) or applied to the component directly without mixing (false)
     * 
     * @author hankherr
     */
    public static class BackgroundColorEffect extends Effect<Component>
    {

        private final Color _c;
        private final boolean _mix;

        public BackgroundColorEffect(final Color c, final boolean mix)
        {
            _c = c;
            _mix = mix;
        }

        @Override
        public void perform(final Component input)
        {
            if(_mix)
            {
                input.setBackground(ColorTools.mixColors(input.getBackground(), _c));
            }
            else
            {
                input.setBackground(_c);
            }
        }
    }

    public static BackgroundColorEffect createBackgroundEffect(final Color c, final boolean mix)
    {
        return new BackgroundColorEffect(c, mix);
    }

}
