package ohd.hseb.hefs.utils.gui.jtable.renderers;

import java.awt.Component;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.MouseEvent;
import java.util.EventObject;

import javax.swing.AbstractCellEditor;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.table.TableCellEditor;

/**
 * Table editor for {@link String} data (not numbers) that includes a {@link FocusAdapter} that highlights the text when
 * the focus is gained. It will also turn off cell editing when focus is lost, even if the focus is lost by clicking
 * outside the table.<br>
 * <br>
 * This class does allow for user specified editor components, but they must be subclasses of {@link JTextField}.
 * 
 * @author hank.herr
 */
public class StringCellEditor extends AbstractCellEditor implements TableCellEditor
{
    private static final long serialVersionUID = 5799252748973630853L;

    private static final int DEFAULT_CLICK_COUNT = 1;

    /**
     * Defines how many clicks will fire the editing mode Default to single click
     */
    private int _clickCount = DEFAULT_CLICK_COUNT;

    //XXX I'm not sure why I feel the need to track the _currentValue based on how it is used below.
    //But I don't want to change it for now.
    private String _currentValue;

    /**
     * This is the component that will handle the editing of the cell value. Initialized to a {@link JTextField}.
     */
    private JTextField _component = new JTextField();

    /**
     * Focus listener is added when editing and removed when not editing.
     */
    private FocusAdapter _generalFocusListener;

    /**
     * This flag addresses the following problem: the getTableCellEditorComponent(...) method is called before the
     * displayed component loses focus. Hence, when its focusLost is triggered, it is actually triggering it relative to
     * the cell just clicked on. Thus, stop cell editing is called immediately after the component is returned, making
     * it so that the component is never actually edited. By using this flag, we can control when a focusLost triggers a
     * stopCellEditing. Specifically, it is triggered only after the focus has been gained.
     */
    private boolean _focusLostTriggerStopFlag = true;

    /**
     * Sets up the editor such that one click is required to start editing. Also sets up focus listener so that the
     * entire cell is selected when focus is gained.
     */
    public StringCellEditor()
    {
        super();
        setupFocusListener();
    }

    /**
     * Sets up the editor such that clickCount clicks are required to enter edit mode. Also sets up focus listener so
     * that the entire cell is selected when focus is gained.
     * 
     * @param clickCount Number of clicks.
     */
    public StringCellEditor(int clickCount)
    {
        super();
        this._clickCount = clickCount;
        setupFocusListener();
    }

    /**
     * Uses a provided editor component and sets it up so that clickCount clicks are required to enter edit mode. This
     * does not include a focus listener; if you provide your own component, you must decide how to react to a focus
     * gained or lost event.
     * 
     * @param editorComp Editor component, which must be an instance/subclass of JTextField.
     * @param clickCount Number of clicks required to start editing.
     */
    public StringCellEditor(JTextField editorComp, int clickCount)
    {
        super();
        _component = editorComp;
        _clickCount = clickCount;
        setupFocusListener();
    }

    private void setupFocusListener()
    {
        _generalFocusListener = new FocusAdapter()
        {
            @Override
            public void focusGained(FocusEvent e)
            {
                _component.selectAll();
                _focusLostTriggerStopFlag = true;
            }

            @Override
            public void focusLost(FocusEvent e)
            {
                if(_focusLostTriggerStopFlag)
                {
                    stopCellEditing();
                }
            }
        };
    }

    @Override
    public boolean stopCellEditing()
    {
        _component.removeFocusListener(_generalFocusListener);
        _focusLostTriggerStopFlag = false;
        return super.stopCellEditing();
    }

    @Override
    public Object getCellEditorValue()
    {
        return (_component).getText();
    }

    public int getClickCount()
    {
        return _clickCount;
    }

    @Override
    public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column)
    {
        if(value == null)
        {
            value = "";
        }
        _currentValue = value.toString();

        _component.setText(_currentValue);
        _component.selectAll();

        _component.addFocusListener(_generalFocusListener);
        return _component;
    }

    @Override
    public boolean isCellEditable(EventObject evt)
    {
        if(evt instanceof MouseEvent)
        {
            return ((MouseEvent)evt).getClickCount() >= getClickCount();
        }

        return true;
    }

    public void setClickCount(int clickCount)
    {
        this._clickCount = clickCount;
    }
}
