/*
 * Created on Mar 7, 2007 To change the template for this generated file go to Window&gt;Preferences&gt;Java&gt;Code
 * Generation&gt;Code and Comments
 */
package ohd.hseb.hefs.utils.gui.listpanels;

import java.awt.Color;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.util.data.DataSet;
import ohd.hseb.util.misc.MiscTools;

/**
 * Displays a JList inside of a JScrollPane with a title border around it. Any owning class must implement
 * ListPanelOwner and pass itself into the constructor if it needs to hear when clicks happen. <br>
 * <br>
 * Optionally, a JPopupMenu can be associated with right clicks within the list. Pass in true into the appropriate
 * argument within the constructor and define the popup menu within the method getListPanelPopupMenu. When a menu item
 * is selected, the method listPanelPopupMenuItemSelected will be called with the title of the popup menu. Always assume
 * that the currently selected list item(s) are what is to be impacted by the popup menu click.
 * 
 * @author hank
 */
public class JListPanel extends JPanel implements ListSelectionListener, MouseListener, MouseMotionListener,
ActionListener
{
    private static final long serialVersionUID = 1L;
    final static String CLASSNAME = "JListPanel";
    public final static String NONE_STR = "NONE";

    private Font _listFont;

    /**
     * The managed JList.
     */
    private JList _theJList;

    /**
     * List of Strings making up _theJList.
     */
    private List<String> _listData;

    /**
     * Contains _theJList.
     */
    private JScrollPane _listSP;

    /**
     * Popup menu that is created whenever an item is right clicked. Acquired from the owner.
     */
    private JPopupMenu _popupMenu;

    /**
     * If false, _popupMenu is null and never pops up.
     */
    private final boolean _includePopupMenu;

    /**
     * The owner of this panel. Must implement the interface.
     */
    private final ListPanelOwner _owner;

    /**
     * Background color for the panel.
     */
    private final Color _bgColor;

    /**
     * If true, then NONE is to be at the top of the list at all times. If false, then NONE will only be included when
     * there are no list items.
     */
    private final boolean _includeNone;

    /**
     * The text to put in the titled border for this panel.
     */
    private final String _title;

    /**
     * When true, change is selected items is ignored.
     */
    private boolean _ignoreSelections;

    /**
     * The index of the item over which the mouse was located when the right mouse button was clicked (this is the
     * action that opens up the optional popup menu!).
     */
    private int _rightClickedOnListIndex;

    /**
     * The index of the last item over which the mouse pointed.
     */
    private int _mouseOverItemIndex;

    /**
     * If true, then the list, when updated, will always be sorted.
     */
    private boolean _sortListForDisplay;

    /**
     * @param title The border title for this panel.
     * @param listData The list data, which must be a List of Strings. The list will be sorted alphabetically upon
     *            construction. This list is copied by the class.
     * @param sortForDisplay sort the listData prior to displaying them.
     * @param bgColor The background color for the list and embedded panel.
     * @param includeNone True if NONE is to be part of the list. If NONE is not found in the list, it will be added at
     *            the top.
     * @param includePopupMenu True if a popup menu can be created and the owner provides it.
     * @param owner The owner, which is called whenever a selection is made from the list.
     */
    public JListPanel(final String title,
                      final List<String> listData,
                      final boolean sortForDisplay,
                      final Color bgColor,
                      final boolean includeNone,
                      final boolean includePopupMenu,
                      final ListPanelOwner owner)
    {
        _title = title;
        _bgColor = bgColor;
        _owner = owner;
        _includeNone = includeNone;
        _includePopupMenu = includePopupMenu;
        _ignoreSelections = false;
        _rightClickedOnListIndex = -1;
        _mouseOverItemIndex = -1;
        _listData = new ArrayList<String>();
        _sortListForDisplay = sortForDisplay;
        buildJList();
        updateListData(listData);
        createDisplay();
        createPopupMenu();
    }

    /**
     * @param title The border title for this panel.
     * @param listData The list data, which must be a List of Strings. The list will be sorted alphabetically upon
     *            construction. This list is copied by the class.
     * @param bgColor The background color for the list and embedded panel.
     * @param includeNone True if NONE is to be part of the list. If NONE is not found in the list, it will be added at
     *            the top.
     * @param includePopupMenu True if a popup menu can be created and the owner provides it.
     * @param owner The owner, which is called whenever a selection is made from the list.
     */
    public JListPanel(final String title,
                      final List<String> listData,
                      final Color bgColor,
                      final boolean includeNone,
                      final boolean includePopupMenu,
                      final ListPanelOwner owner)
    {
        _title = title;
        _bgColor = bgColor;
        _owner = owner;
        _includeNone = includeNone;
        _includePopupMenu = includePopupMenu;
        _ignoreSelections = false;
        _rightClickedOnListIndex = -1;
        _mouseOverItemIndex = -1;
        _listData = new ArrayList<String>();
        _sortListForDisplay = true;
        buildJList();
        updateListData(listData);
        createDisplay();
        createPopupMenu();
    }

    private void createDisplay()
    {
        //This is the interior panel
        final JPanel panel = new JPanel(new GridBagLayout());

        //Fill in the interior panel.
        GridBagConstraints constraints;
        Insets theinsets = new Insets(5, 5, 5, 5);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        panel.add(_listSP, constraints);
        final TitledBorder tb = new TitledBorder(new LineBorder(Color.GRAY), _title);
        panel.setBorder(tb);

        //Put the interior panel within this panel so that I can have some insets.
        setLayout(new GridBagLayout());
        theinsets = new Insets(2, 2, 2, 2);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          0,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.WEST,
                                                          GridBagConstraints.BOTH,
                                                          theinsets,
                                                          0,
                                                          0);
        add(panel, constraints);

        //Set the backgrounds.
        if(_bgColor != null)
        {
            panel.setBackground(_bgColor);
            setBackground(_bgColor);
        }

        _theJList.addMouseListener(this);
        _theJList.addMouseMotionListener(this);
    }

    /**
     * Gets the popup menu from the owner and adds this as an action listener to every menu item.
     */
    private void createPopupMenu()
    {
        if(!_includePopupMenu)
        {
            _popupMenu = null;
            return;
        }
        _popupMenu = _owner.getListPanelPopupMenu(this);
        int i;
        for(i = 0; i < _popupMenu.getComponents().length; i++)
        {
            if(_popupMenu.getComponents()[i] instanceof JMenuItem)
            {
                ((JMenuItem)_popupMenu.getComponents()[i]).addActionListener(this);
            }
        }
    }

    /**
     * Create a JList given a list of VerificationLocations.
     * 
     * @param locations List of VerificationLocation objects.
     */
    private void buildJList()
    {
        //Create the scroll pane.
        _listSP = HSwingFactory.createScrollPanedJList(_listFont, _listData);
        _listSP.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        _listSP.setWheelScrollingEnabled(true);
        _theJList = (JList)(_listSP.getViewport().getView());
        _theJList.setSelectedIndex(0);
        _theJList.addListSelectionListener(this);

    }

    /**
     * Update the list.
     */
    public void updateListData(final List<String> listData)
    {
        _listData = listData;

        if((listData == null) || (listData.size() == 0))
        {
            _listData = new ArrayList<String>();
            _listData.add(NONE_STR);
        }
        else
        {
            _listData = new ArrayList<String>();
            _listData.addAll(listData);
            if(_sortListForDisplay)
            {
                Collections.sort(_listData);
            }
            if(_includeNone)
            {
                _listData.add(0, NONE_STR);
            }
        }

        //Create the scroll pane.
        _theJList.removeAll();
        refreshJList();
        _theJList.setSelectedIndex(0);
    }

    /**
     * Refreshes the JList to match the _listData data.
     */
    public void refreshJList()
    {
        _theJList.setListData(MiscTools.stringArrayFromList(_listData));
    }

    public void setSortListForDisplay(final boolean b)
    {
        _sortListForDisplay = b;
    }

    public JList getTheJList()
    {
        return _theJList;
    }

    public List<String> getListData()
    {
        return _listData;
    }

    public int getSelectedIndex()
    {
        return _theJList.getSelectedIndex();
    }

    public String getSelectedItem()
    {
        return (String)_theJList.getSelectedValue();
    }

    public void setSelectedItem(final String item)
    {
        if(item != null)
        {
            _theJList.setSelectedValue(item, true);
        }
    }

    public void setSelectedIndex(final int index)
    {
        if(index < 0)
        {
            _theJList.clearSelection();
        }
        else
        {
            _theJList.setSelectedIndex(index);
        }
    }

    public List getSelectedItems()
    {
        return _theJList.getSelectedValuesList();
    }

    public void setSelectedItems(final List items)
    {
        if(items == null)
        {
            return;
        }
        int i;
        for(i = 0; i < items.size(); i++)
        {
            _theJList.setSelectedValue(items.get(i), true);
        }
    }

    /**
     * Returns the text of the item at the specified point in _theJList.
     * 
     * @param pt Point in pixels relative to _theJList.
     */
    public int getItemIndexAt(final Point pt)
    {
        int i;
        Rectangle rect;
        for(i = 0; i < _listData.size(); i++)
        {
            rect = _theJList.getCellBounds(i, i);
            if((rect.getX() < pt.getX()) && (rect.getX() + rect.getWidth() >= pt.getX()) && (rect.getY() < pt.getY())
                && (rect.getY() + rect.getHeight() >= pt.getY()))
            {
                return i;
            }
        }
        return (int)DataSet.MISSING;
    }

    public void setIgnoreSelection(final boolean b)
    {
        _ignoreSelections = b;
    }

    public boolean getIgnoreSelections()
    {
        return _ignoreSelections;
    }

    public int getIndexOfItemMouseIsOver()
    {
        return _mouseOverItemIndex;
    }

    public String getItemRightClickedOn()
    {
        if(_rightClickedOnListIndex > 0)
        {
            return _listData.get(_rightClickedOnListIndex);
        }
        return null;
    }

    public int getIndexOfItemRightClickedOn()
    {
        return _rightClickedOnListIndex;
    }

    @Override
    public void valueChanged(final ListSelectionEvent arg0)
    {
        //If this is a sequence of rapidfire calls, then I only want to react to one of the calls.
        //If this method returns true, then it is part of a sequence of events.  It will return false
        //for the last event in the sequence.  This is necessary to avoid updating the owner twice for
        //each list selection (once for deselecting old, once for selecting new).
        if((arg0 != null) && (arg0.getValueIsAdjusting()))
        {
            return;
        }

        if((_owner != null) && (!_ignoreSelections))
        {
            _owner.listItemSelected(this);
        }

    }

    @Override
    public void mouseClicked(final MouseEvent arg0)
    {
    }

    @Override
    public void mouseEntered(final MouseEvent arg0)
    {
        _mouseOverItemIndex = getItemIndexAt(arg0.getPoint());
        _theJList.repaint();
    }

    @Override
    public void mouseExited(final MouseEvent arg0)
    {
    }

    @Override
    public void mousePressed(final MouseEvent arg0)
    {
        //Record the index of the item clicked.
        _rightClickedOnListIndex = getItemIndexAt(arg0.getPoint());

        if((arg0.getButton() != MouseEvent.BUTTON1) && (_popupMenu != null) && (arg0.isPopupTrigger())
            && (!getItemRightClickedOn().equalsIgnoreCase(NONE_STR)))
        {
            _popupMenu.show(this,
                            (int)(arg0.getX() - _listSP.getViewport().getViewRect().getX()),
                            (int)(arg0.getY() - _listSP.getViewport().getViewRect().getY()));
        }
    }

    @Override
    public void mouseReleased(final MouseEvent arg0)
    {
        if((arg0.getButton() != MouseEvent.BUTTON1) && (_popupMenu != null) && (arg0.isPopupTrigger())
            && (!getItemRightClickedOn().equalsIgnoreCase(NONE_STR)))
        {
            _popupMenu.show(this, arg0.getX(), arg0.getY());
        }
    }

    @Override
    public void actionPerformed(final ActionEvent arg0)
    {
        _owner.listPanelPopupMenuItemSelected(this, ((JMenuItem)arg0.getSource()).getText());
    }

    @Override
    public void mouseDragged(final MouseEvent arg0)
    {
    }

    @Override
    public void mouseMoved(final MouseEvent arg0)
    {
        _mouseOverItemIndex = getItemIndexAt(arg0.getPoint());
        _theJList.repaint();
    }

}
