package ohd.hseb.hefs.utils.gui.tools;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JFrame;
import javax.swing.Timer;

/**
 * This special listener is designed to catch single clicks versus double clicks through the use of a {@link Timer}. If
 * a single click is performed, then the {@link #singleClick(MouseEvent)} method is called. For double clicks
 * {@link #doubleClick(MouseEvent)} is called. See the {@link #main(String[])} for an example of how to apply it.
 * 
 * @author hankherr
 */
public abstract class ClickListener extends MouseAdapter implements ActionListener
{
    private final static int clickInterval = (Integer)Toolkit.getDefaultToolkit()
                                                             .getDesktopProperty("awt.multiClickInterval");

    private MouseEvent lastEvent;
    private final Timer timer;

    public ClickListener()
    {
        this(clickInterval);
    }

    public ClickListener(final int delay)
    {
        timer = new Timer(delay, this);
    }

    @Override
    public void mouseClicked(final MouseEvent e)
    {
        if(e.getClickCount() > 2)
            return;

        lastEvent = e;

        if(timer.isRunning())
        {
            timer.stop();
            doubleClick(lastEvent);
        }
        else
        {
            timer.restart();
        }
    }

    public void actionPerformed(final ActionEvent e)
    {
        timer.stop();
        singleClick(lastEvent);
    }

    /**
     * Override to specify the action for a single click.
     * 
     * @param e The {@link MouseEvent} associated with the click.
     */
    public abstract void singleClick(final MouseEvent e);

    /**
     * Override to specify the action for a double click.
     * 
     * @param e The {@link MouseEvent} associated with the double click.
     */
    public abstract void doubleClick(final MouseEvent e);

    /**
     * Useful as an example!!!
     * 
     * @param args
     */
    public static void main(final String[] args)
    {
        final JFrame frame = new JFrame("Double Click Test");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.addMouseListener(new ClickListener()
        {
            @Override
            public void singleClick(final MouseEvent e)
            {
                System.out.println("single");
            }

            @Override
            public void doubleClick(final MouseEvent e)
            {
                System.out.println("double");
            }
        });
        frame.setSize(200, 200);
        frame.setVisible(true);
    }
}
