package ohd.hseb.hefs.utils.gui.tools;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;

import javax.swing.JColorChooser;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.colorchooser.AbstractColorChooserPanel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import ohd.hseb.hefs.utils.gui.colors.X11ColorStrings;
import ohd.hseb.hefs.utils.gui.components.AlphaColorChooserPanel;
import ohd.hseb.util.misc.SegmentedLine;

public abstract class ColorTools
{

    /**
     * A transparent color with RGB indicating black.
     */
    public final static Color TRANSPARENT_BLACK = new Color(0, 0, 0, 0);

    /**
     * A transparent color with RGB indicating white. I'm not sure why I need two transparent colors. I tend to favor
     * the idea of white, but, until I have time for thorough testing to see if black can be removed, I'll leave the
     * both here and in place.
     */
    public final static Color TRANSPARENT_WHITE = new Color(255, 255, 255, 0);

    /**
     * This processes a color string. The format of the String must be:<br>
     * <br>
     * [known color]<br>
     * -or-<br>
     * [red][separator][green][separator][blue] <br>
     * <br>
     * The method first checks to the see if the string corresponds to a recognized color: black, blue, cyan, dark_gray,
     * gray, green, light_gray, magenta, orange, pink, red, white, and yellow. If none of these match, it will ASSUME
     * the string is of the second format and translate it to a color. If any component is not a number of if there are
     * not three components, null is returned. <br>
     * <br>
     * 
     * @param colorStr
     * @param separator
     * @return Color
     */
    public static Color processColorString(final String colorStr, final char separator)
    {
        //Process Java color strings.
        if(colorStr.equalsIgnoreCase("black"))
        {
            return Color.black;
        }
        else if(colorStr.equalsIgnoreCase("blue"))
        {
            return Color.blue;
        }
        else if(colorStr.equalsIgnoreCase("cyan"))
        {
            return Color.cyan;
        }
        else if(colorStr.equalsIgnoreCase("dark_gray"))
        {
            return Color.DARK_GRAY;
        }
        else if(colorStr.equalsIgnoreCase("gray"))
        {
            return Color.gray;
        }
        else if(colorStr.equalsIgnoreCase("green"))
        {
            return Color.green;
        }
        else if(colorStr.equalsIgnoreCase("light_gray"))
        {
            return Color.LIGHT_GRAY;
        }
        else if(colorStr.equalsIgnoreCase("magenta"))
        {
            return Color.magenta;
        }
        else if(colorStr.equalsIgnoreCase("orange"))
        {
            return Color.orange;
        }
        else if(colorStr.equalsIgnoreCase("pink"))
        {
            return Color.pink;
        }
        else if(colorStr.equalsIgnoreCase("red"))
        {
            return Color.red;
        }
        else if(colorStr.equalsIgnoreCase("white"))
        {
            return Color.white;
        }
        else if(colorStr.equalsIgnoreCase("yellow"))
        {
            return Color.yellow;
        }
        else if(X11ColorStrings.COLOR_STRING_SPECIFIER.retrieveX11Color(colorStr) != null)
        {
            return X11ColorStrings.COLOR_STRING_SPECIFIER.retrieveX11Color(colorStr);
        }

        return processRGBString(colorStr, separator);
    }

    /**
     * Process an RGB string with a given separator in order to acquire a color.
     * 
     * @param rgbString
     * @param separator
     * @return The color or black if the RGB string is invalid.
     */
    public static Color processRGBString(final String rgbString, final char separator)
    {
        final SegmentedLine segLine = new SegmentedLine(rgbString, "" + separator, SegmentedLine.MODE_ALLOW_EMPTY_SEGS);
        if((segLine.getNumberOfSegments() != 3) && (segLine.getNumberOfSegments() != 4))
        {
            return null;
        }
        try
        {
            final int red = Integer.parseInt(segLine.getSegment(0));
            final int green = Integer.parseInt(segLine.getSegment(1));
            final int blue = Integer.parseInt(segLine.getSegment(2));
            int alpha = 255;
            if(segLine.getNumberOfSegments() == 4)
            {
                alpha = Integer.parseInt(segLine.getSegment(3));
            }

            if(((red < 0) && (red >= 256)) || ((green < 0) && (green >= 256)) || ((blue < 0) && (blue >= 256))
                || ((alpha < 0) && (alpha >= 256)))
            {
                return Color.BLACK;
            }

            return new Color(red, green, blue, alpha);
        }
        catch(final NumberFormatException nfe)
        {
            return Color.BLACK;
        }

    }

    /**
     * Generate a color string of the correct format matching processColorString above.
     * 
     * @param color To be translated into a string.
     * @param separator The separator used between the red/green/blue components
     * @return String that can be understood by processColorString.
     */
    public static String generateColorString(final Color color, final char separator)
    {
        return "" + color.getRed() + separator + color.getGreen() + separator + color.getBlue();
    }

    /**
     * Wraps the standard {@link JColorChooser} inside of a confirmation dialog with an added label indicating if the
     * current selection is completely transparent (warning the user that it will not be visible).
     * 
     * @param parent Parent component for JDialog.
     * @param initialColor The initial color.
     * @return New color or null if cancel is clicked.
     */
    public static Color chooseColor(final Component parent, Color initialColor)
    {
        if(initialColor == null)
        {
            initialColor = Color.BLACK;
        }
        final JColorChooser colorChooser = new JColorChooser();

        //This label will warn users if the color is completely transparent and therefore not visible.
        final JLabel transparencyAlertLabel = new JLabel("");
        colorChooser.getSelectionModel().addChangeListener(new ChangeListener()
        {
            @Override
            public void stateChanged(final ChangeEvent e)
            {
                if(colorChooser.getSelectionModel().getSelectedColor().getAlpha() == 0)
                {
                    transparencyAlertLabel.setText("Warning: Selected color is completely transparent!");
                }
                else if(colorChooser.getSelectionModel().getSelectedColor().getAlpha() == 255)
                {
                    transparencyAlertLabel.setText("Selected color is opaque");
                }
                else
                {
                    transparencyAlertLabel.setText("Selected color is partially transparent");
                }
            }
        });
        colorChooser.getSelectionModel().setSelectedColor(initialColor);

        //The transparency message goes below the color chooser.
        final JPanel displayedPanel = new JPanel(new BorderLayout());
        displayedPanel.add(colorChooser, BorderLayout.CENTER);
        displayedPanel.add(transparencyAlertLabel, BorderLayout.SOUTH);

        final int returnValue =
                              JOptionPane.showConfirmDialog(parent,
                                                            displayedPanel,
                                                            "Choose Color",
                                                            JOptionPane.OK_CANCEL_OPTION,
                                                            JOptionPane.QUESTION_MESSAGE);

        if(returnValue == JOptionPane.OK_OPTION)
        {
            final Color chosenColor = new Color(colorChooser.getSelectionModel().getSelectedColor().getRed(),
                                                colorChooser.getSelectionModel().getSelectedColor().getGreen(),
                                                colorChooser.getSelectionModel().getSelectedColor().getBlue(),
                                                colorChooser.getSelectionModel().getSelectedColor().getAlpha());
            return chosenColor;
        }
        return null;
    }

    /**
     * A {@link JColorChooser} that only chooses the alpha level, and nothing else. The returned color will have the
     * same RGB value, but the new chosen alpha.
     * 
     * @param parent The parent of the {@link JDialog}.
     * @param initialColor The initial color.
     * @return New color or null if cancel is clicked.
     */
    public static Color chooseAlpha(final Component parent, final Color initialColor)
    {
        final JColorChooser colorChooser = new JColorChooser(initialColor);
        final AbstractColorChooserPanel[] panels = colorChooser.getChooserPanels();
        for(int i = 0; i < panels.length; i++)
        {
            colorChooser.removeChooserPanel(panels[i]);
        }
        colorChooser.addChooserPanel(new AlphaColorChooserPanel(initialColor));

        final int returnValue =
                              JOptionPane.showConfirmDialog(parent,
                                                            colorChooser,
                                                            "Choose Transparency (0 transparent, 255 opaque)",
                                                            JOptionPane.OK_CANCEL_OPTION,
                                                            JOptionPane.QUESTION_MESSAGE);

        if(returnValue == JOptionPane.OK_OPTION)
        {
            return colorChooser.getSelectionModel().getSelectedColor();
        }
        return null;
    }

    /**
     * Distance from one color to another. Computed using distances in 3 dimensions. Variation on Pythagorean theorem.
     * 
     * @param color1 First color.
     * @param color2 Second color.
     * @return distance.
     */
    private static double computeDistanceBetweenColors(final Color color1, final Color color2)
    {
        return Math.sqrt(Math.pow(((double)color1.getRed() - (double)color2.getRed()), 2.0D)
            + Math.pow(((double)color1.getBlue() - (double)color2.getBlue()), 2.0D)
            + Math.pow(((double)color1.getGreen() - (double)color2.getGreen()), 2.0D)
            + Math.pow(((double)color1.getAlpha() - (double)color2.getAlpha()), 2.0D));
    }

    /**
     * @param color Color to check.
     * @return Return true if the color is dark; i.e. it is closer to black than white, according to
     *         computeDistanceBetweenColors.
     */
    public static boolean isColorDark(final Color color)
    {
        if(color.getAlpha() < 128)
        {
            return false;
        }
        final Color lightestWhite = new Color(255, 255, 255, 0);
        if(ColorTools.computeDistanceBetweenColors(color,
                                                   Color.BLACK) >= ColorTools.computeDistanceBetweenColors(color,
                                                                                                           lightestWhite))
        {
            return false;
        }
        return true;
    }

    /**
     * Mix two colors evenly. Calls {@link #mixColors(Color, Color, double)} with 0.5 as the distance fraction.
     * 
     * @param color1 First color in the mix.
     * @param color2 Second color in the mix.
     * @return Mixed color in which each of the r, g, b components, and alpha, are averaged.
     */
    public static Color mixColors(final Color color1, final Color color2)
    {
        return mixColors(color1, color2, 0.5d);

    }

    /**
     * @param color1 The starting color.
     * @param color2 The ending color.
     * @param distanceFraction The fraction (between 0 and 1) of the difference (distance) between the two colors to
     *            travel to create the mixed color. Passing in 0.5 is equivalent to {@link #mixColors(Color, Color)}. A
     *            value of 0.0 returns the first color, and 1.0 returns the second color.
     * @return A mixed color with the distanceFraction dictating the mix weight.
     */
    public static Color mixColors(final Color color1, final Color color2, final double distanceFraction)
    {
        return new Color((int)(color1.getRed() + distanceFraction * (color2.getRed() - color1.getRed())),
                         (int)(color1.getGreen() + distanceFraction * (color2.getGreen() - color1.getGreen())),
                         (int)(color1.getBlue() + distanceFraction * (color2.getBlue() - color1.getBlue())),
                         (int)(color1.getAlpha() + distanceFraction * (color2.getAlpha() - color1.getAlpha())));
    }

    /**
     * @param color1 The starting color.
     * @param colorMid The middle color
     * @param color2 The ending color.
     * @param distanceFraction The fraction (between 0 and 1) of the difference (distance) between color1 and color2 to
     *            travel. If the fraction is smaller than 0.5, then this returns
     *            {@link #mixColors(Color, Color, double)} using color1 and colorMid. If the fraction is larger than
     *            0.5, this this does the same with colorMid and color2. At 0.5, it returns colorMid.
     * @return A mixed color with the distanceFraction dictating the mix weight.
     */
    public static Color mixColors(final Color color1,
                                  final Color colorMid,
                                  final Color color2,
                                  final double distanceFraction)
    {
        if(distanceFraction < 0.5)
        {
            return mixColors(color1, colorMid, distanceFraction / 0.5D);
        }
        else if(distanceFraction > 0.5)
        {
            return mixColors(colorMid, color2, (distanceFraction - 0.5D) / 0.5D);
        }
        return colorMid;
    }

    /**
     * Builds an array of colors, stepping from the first color provided to the last color provided through each of the
     * intermediary colors. This algorithm cannot guarantee that the intermediary colors will be in the list, since the
     * steps are all equal sized, meaning that colors can be stepped over by this algorithm. Only the first and last
     * colors are guaranteed to be in the returned array.
     * 
     * @param numberOfColors The number of colors to create.
     * @param baseColors The Colors that dictate whats in the palette.
     * @return A palette of colors.
     */
    public static Color[] buildColorPalette(final int numberOfColors, final Color... baseColors)
    {
        final int numberOfBaseColors = baseColors.length;
        final Color[] palette = new Color[numberOfColors];

        //No shade computations needed.
        if(numberOfColors <= numberOfBaseColors)
        {
            for(int i = 0; i < palette.length; i++)
            {
                palette[i] = baseColors[i];
            }
        }
        else
        {
            int baseColorIndex = 0;
            double baseColorFractionalCounter = 0.0d;
            double mixingFraction = 0.0d;

            //This algorithms 'walks' from 0 to the number of base colors such that the number of steps
            //taken equals the number of colors to create.
            for(int paletteIndex = 0; paletteIndex < palette.length; paletteIndex++)
            {
                //Base color index is used to identify the first of the two colors.  It is the truncation
                //of the fractional counter.
                baseColorIndex = (int)baseColorFractionalCounter;
                mixingFraction = (baseColorFractionalCounter - baseColorIndex);
                if(baseColorIndex == numberOfBaseColors - 1) //The last color is called for, which causes an index error
                {
                    baseColorIndex--;
                    mixingFraction = 1.0d;
                }

                //Mix them and set the palette color.
                palette[paletteIndex] = ColorTools.mixColors(baseColors[baseColorIndex],
                                                             baseColors[baseColorIndex + 1],
                                                             mixingFraction);

                //Increment the counter.  The step size is the number of total colors minus 1.  This is
                //because if we start counting at 0, the number of steps taken is actually numberOfColors - 1. 
                baseColorFractionalCounter += (numberOfBaseColors - 1.0d) / (numberOfColors - 1.0d);
            }
        }

        return palette;
    }

    /**
     * @return A deep copy of the provided color. With return null if given null.
     */
    public static Color deepCopy(final Color c)
    {
        if(c == null)
        {
            return null;
        }
        return new Color(c.getRed(), c.getGreen(), c.getBlue(), c.getAlpha());
    }
}
