package ohd.hseb.hefs.utils.gui.tools;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.LayoutManager;
import java.awt.Point;
import java.awt.event.ActionListener;
import java.awt.event.ItemListener;
import java.io.File;
import java.net.URL;
import java.util.Collection;
import java.util.List;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSpinner;
import javax.swing.JSpinner.NumberEditor;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.KeyStroke;
import javax.swing.ScrollPaneConstants;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingConstants;
import javax.swing.border.BevelBorder;
import javax.swing.border.Border;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;

import ohd.hseb.hefs.utils.gui.components.ComponentTitledBorder;
import ohd.hseb.hefs.utils.gui.components.MenuItemTriggeringButton;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * A collection of static methods that build and return GUI components.
 * 
 * @author hank
 */
public abstract class HSwingFactory
{

    /**
     * @param label
     * @param font Can be null to use default font.
     * @param selected
     * @return Labeled JCheckBox initialized to selected.
     */
    public static JCheckBox createJCheckBox(final String label, final Font font, final boolean selected)
    {
        final JCheckBox working = new JCheckBox(label, selected);
        if(font != null)
        {
            working.setFont(font);
        }
        return working;
    }

    /**
     * Creates a formatted {@link JLabel} using {@link String#format(String, Object...)}. This is similar to C printf.
     * 
     * @param format The format to follow.
     * @param formatArgs Variable length list of formatter arguments.
     * @return JLabel with default font.
     */
    public static JLabel createJLabel(final String format, final Object... formatArgs)
    {
        return new JLabel(String.format(format, formatArgs));
    }

    /**
     * @param label
     * @param font Null is allowed.
     * @param fg Null is allowed.
     * @return A {@link JLabel} with the font and fg set, if specified, and horizontal alignment
     *         {@link SwingConstants#LEADING}.
     */
    public static JLabel createJLabel(final String label, final Font font, final Color fg)
    {
        return createJLabel(label, font, fg, JLabel.LEADING);
    }

    /**
     * @param label
     * @param font Null is allowed.
     * @param fg Null is allowed.
     * @param horizontal
     * @return JLabel with the specified horizontal alignment; see SwingConstants.
     */
    public static JLabel createJLabel(final String label, final Font font, final Color fg, final int horizontal)
    {
        final JLabel working = new JLabel(label, horizontal);
        if(font != null)
        {
            working.setFont(font);
        }
        if(fg != null)
        {
            working.setForeground(fg);
        }
        return working;
    }

    /**
     * @param text The radio button label
     * @param font The font. Null is allowed.
     * @param listener the ItemListener that listens to this. Null is allowed.
     * @param initialState
     * @return {@link JRadioButton} with the font set if given and ItemListener attached if given.
     */
    public static JRadioButton createJRadioButton(final String text,
                                                  final Font font,
                                                  final ItemListener listener,
                                                  final boolean initialState)
    {
        final JRadioButton radioButton = new JRadioButton(text, initialState);
        if(font != null)
        {
            radioButton.setFont(font);
        }
        if(listener != null)
        {
            radioButton.addItemListener(listener);
        }
        return radioButton;
    }

    /**
     * @param items
     * @param font Font to use. Null is allowed.
     * @return
     */
    @SuppressWarnings("rawtypes")
    public static JComboBox createJComboBox(final Collection items, final Font font)
    {
        @SuppressWarnings("unchecked")
        final JComboBox working = new JComboBox(new Vector(items));
        if(font != null)
        {
            working.setFont(font);
        }
        return working;
    }

    /**
     * @param items
     * @param font Null is allowed.
     * @return
     */
    public static JComboBox createJComboBox(final String[] items, final Font font)
    {
        final JComboBox working = new JComboBox(items);
        if(font != null)
        {
            working.setFont(font);
        }
        return working;
    }

    /**
     * @param text
     * @param font Null is allowed.
     * @return {@link JTextField} with text initialized. Columns is not set.
     */
    public static JTextField createJTextField(final String text, final Font font)
    {
        return createJTextField(text, font, -1);
    }

    /**
     * @param text
     * @param font Null is allowed.
     * @param columns If negative, column width is not set.
     * @return {@link JTextField} with text initialized. Columns is not set.
     */
    public static JTextField createJTextField(final String text, final Font font, final int columns)
    {
        final JTextField working = new JTextField(text);
        if(font != null)
        {
            working.setFont(font);
        }
        if(columns > 0)
        {
            working.setColumns(columns);
        }
        return working;
    }

    /**
     * It may be better to define {@link JButton} instances via {@link Action} as opposed to this.
     * 
     * @param text
     * @param font Can be null.
     * @param listener Can be null.
     * @return
     */
    public static JButton createJButton(final String text, final Font font, final ActionListener listener)
    {
        final JButton working = new JButton(text);
        if(font != null)
        {
            working.setFont(font);
        }
        if(listener != null)
        {
            working.addActionListener(listener);
        }
        return working;
    }

    /**
     * It may be better to define {@link JButton} instances via {@link Action} as opposed to this.
     * 
     * @param icon Icon to use.
     * @param listener Can be null.
     * @return
     */
    public static JButton createJButton(final Icon icon, final ActionListener listener)
    {
        final JButton working = new JButton(icon);
        if(listener != null)
        {
            working.addActionListener(listener);
        }
        return working;
    }

    /**
     * @param iconImageName Name of an icon file to use, which is loaded as a system resource. If the icon is not found,
     *            the button will have a label with the icon file name in it.
     * @param toolTip Tool tip to apply. Null is allowed.
     * @param listener Null is allowed.
     * @return A JButton displaying the specified icon.
     */
    public static JButton createJButtonWithIcon(final String iconImageName,
                                                final String toolTip,
                                                final ActionListener listener)
    {
        final URL url = ClassLoader.getSystemResource(iconImageName);
        ImageIcon icon = null;
        JButton working = null;
        if(url != null)
        {
            icon = new ImageIcon(url);
            working = new JButton(icon);
        }
        else
        {
            working = new JButton(iconImageName);
        }
        if(listener != null)
        {
            working.addActionListener(listener);
        }
        if(toolTip != null)
        {
            working.setToolTipText(toolTip);
        }
        return working;
    }

    /**
     * Creates a {@link JButton} that triggers a {@link JMenuItem} when clicked. Makes it easy to tie a tool bar button
     * to a menu item.
     * 
     * @param icon Icon to use.
     * @param menuItem Menu item to activate.
     * @return
     */
    public static JButton createMenuItemTriggeringToolBarButton(final Icon icon, final JMenuItem menuItem)
    {
        final JButton button = new MenuItemTriggeringButton(icon, menuItem);
        return button;
    }

    /**
     * Creates a {@link JButton} that triggers a {@link JMenuItem} when clicked. Makes it easy to tie a tool bar button
     * to a menu item.
     * 
     * @param iconImageFile Name of a resource providing an icon (.png file, for example).
     * @param menuItem Menu item to activate.
     * @return {@link MenuItemTriggeringButton} instance or null if the icon cannot be loaded.
     */
    public static MenuItemTriggeringButton createMenuItemTriggeringToolBarButton(final String iconImageFile,
                                                                                 final JMenuItem menuItem)
    {
        if(ClassLoader.getSystemResource(iconImageFile) == null)
        {
            return null;
        }
        final ImageIcon icon = new ImageIcon(ClassLoader.getSystemResource(iconImageFile));
        final MenuItemTriggeringButton button = new MenuItemTriggeringButton(icon, menuItem);
        return button;
    }

    /**
     * @param names Button labels.
     * @param font Null is allowed.
     * @param listener Null is allowed.
     * @return Array of {@link JButton} instances, all with the same {@link ActionListener}.
     */
    public static JButton[] createJButtons(final String[] names, final Font font, final ActionListener listener)
    {
        int i;

        //Create the buttons
        final JButton[] buttons = new JButton[names.length];
        for(i = 0; i < names.length; i++)
        {
            buttons[i] = HSwingFactory.createJButton(names[i], font, listener);
        }
        return buttons;
    }

    /**
     * @param buttons Buttons to add to a panel.
     * @param layoutMgr {@link LayoutManager} to follow.
     * @return {@link JPanel} with all buttons added via the {@link JPanel#add(Component)} method using the given
     *         LayoutManager.
     */
    public static JPanel createJButtonPanel(final JButton[] buttons, final LayoutManager layoutMgr)
    {
        final JPanel panel = new JPanel(layoutMgr);
        int i;
        for(i = 0; i < buttons.length; i++)
        {
            panel.add(buttons[i]);
        }
        return panel;
    }

    /**
     * @param buttons
     * @param groupsizes
     * @param boxlayout
     * @return Box containing buttons created by calling
     *         {@link #createJButtonBox(JButton[], int[], int, boolean, boolean, boolean)} with true, false, and false
     *         passed in.
     */
    public static Box createJButtonBox(final JButton[] buttons, final int[] groupsizes, final int boxlayout)
    {
        return createJButtonBox(buttons, groupsizes, boxlayout, true, false, false);
    }

    /**
     * Create a Box containing the buttons given as the first parameter. <br>
     * <br>
     * - The buttons will be grouped with borders (optional) surrounding the groups. The parameter groupsizes specifies
     * the number of groups (length of array) and number of buttons in each group. The buttons are assumed to be ordered
     * by group. <br>
     * - The boxlayout parameter must be BoxLayout.X_AXIS for horizontal buttons or BoxLayout.Y_AXIS for vertical
     * buttons.<br>
     * - If the boolean borderon is true, then there will be a border placed around the buttons.<br>
     * - If the boolean outsidepanels is true, then there will be filler JPanels placed below above the buttons within
     * the box.<br>
     * <br>
     * The result is that the array passed in to buttons will contain the created buttons and the returned panel will be
     * used for display. Each group will correspond to a panel within the box with a GridLayout evenly spacing the
     * buttons.<br>
     * <br>
     * 
     * @param buttons
     * @param groupsizes
     * @param boxDirection
     * @param borderon
     * @param outsidepanels
     * @return
     */
    public static Box createJButtonBox(final JButton[] buttons,
                                       final int[] groupsizes,
                                       final int boxDirection,
                                       final boolean borderon,
                                       final boolean outsidepanels)
    {
        //TRUE is for internal grid layout.
        return createJButtonBox(buttons, groupsizes, boxDirection, true, borderon, outsidepanels);
    }

    /**
     * Create a Box containing the buttons given as the first parameter. <br>
     * <br>
     * - The buttons will be grouped with borders (optional) surrounding the groups. The parameter groupsizes specifies
     * the number of groups (length of array) and number of buttons in each group. The buttons are assumed to be ordered
     * by group. <br>
     * - The boxlayout parameter must be BoxLayout.X_AXIS for horizontal buttons or BoxLayout.Y_AXIS for vertical
     * buttons.<br>
     * - If the boolean borderon is true, then there will be a border placed around the buttons.<br>
     * - If the boolean outsidepanels is true, then there will be filler JPanels placed below above the buttons within
     * the box.<br>
     * <br>
     * The result is that the array passed in to buttons will contain the created buttons and the returned panel will be
     * used for display.<br>
     * <br>
     * 
     * @param buttons
     * @param groupsizes
     * @param boxDirection
     * @param useInternalGridLayout if true each group's internal panel will be layed out using GridLayout. Otherwise,
     *            flow layout will be used. This only matters if boxDirection is X_AXIS. Y_AXIS always uses GridLayout
     *            for internal group panels.
     * @param borderon
     * @param outsidepanels
     * @return
     */
    public static Box createJButtonBox(final JButton[] buttons,
                                       final int[] groupsizes,
                                       final int boxDirection,
                                       final boolean useInternalGridLayout,
                                       final boolean borderon,
                                       final boolean outsidepanels)
    {
        int i, j, basenum = 0;

        //I need one panel for each button, each with a FlowLayout
        final JPanel buttonwrappanel[] = new JPanel[buttons.length];
        for(i = 0; i < buttons.length; i++)
        {
            buttonwrappanel[i] = new JPanel();
            buttonwrappanel[i].add(buttons[i]);
        }

        //Now, I'll construct some fancy panels to contain the button wrap panels in 
        //button groups.
        final JPanel fancywrappanel[] = new JPanel[groupsizes.length];
        for(i = 0; i < groupsizes.length; i++)
        {
            //The GridLayout depends on the boxlayout.
            if(boxDirection == BoxLayout.X_AXIS)
            {
                if(useInternalGridLayout)
                {
                    fancywrappanel[i] = new JPanel(new GridLayout(1, groupsizes[i]));
                }
                else
                {
                    fancywrappanel[i] = new JPanel(new FlowLayout());
                }
            }
            else
            {
                fancywrappanel[i] = new JPanel(new GridLayout(groupsizes[i], 1));
            }

            //Only turn on the bevel border if borderon is true!
            if(borderon)
            {
                fancywrappanel[i].setBorder(new BevelBorder(BevelBorder.RAISED));
            }

            //Add the buttons to this group.  Note that I am assuming buttonwrappanel was 
            //created in order of button grouping and display!
            for(j = 0; j < groupsizes[i]; j++)
            {
                fancywrappanel[i].add(buttonwrappanel[basenum + j]);
            }

            //Increment basenum by the number of pieces just added so that we add the next
            //group starting from the correct position in buttonwrappanel[].
            basenum += groupsizes[i];
        }

        //Create the button panel using the fancywrappanel array.
        Box buttonpanel;
        buttonpanel = new Box(boxDirection);
        if(outsidepanels)
        {
            buttonpanel.add(HSwingFactory.createFillerJPanel());
        }
        for(i = 0; i < groupsizes.length; i++)
        {
            buttonpanel.add(fancywrappanel[i]);
        }
        if(outsidepanels)
        {
            buttonpanel.add(HSwingFactory.createFillerJPanel());
        }

        //Return buttonpanel.
        return buttonpanel;
    }

    /**
     * Creata a {@link JList} and put it inside a {@link JScrollPane}. Return the JScrollPane. To get the JList that is
     * created within this method, call...<br>
     * <br>
     * (JList)(<returned pane>.getViewport().getView()); <br>
     * <br>
     * where <returned pane> is the JScrollPane returned by this function. <br>
     * <br>
     * 
     * @param font Null is allowed.
     * @return JScrollPane containing an embedded JList.
     */
    public static JScrollPane createScrollPanedJList(final Font font)
    {
        final JList list = new JList();
        if(font != null)
        {
            list.setFont(font);
        }
        return new JScrollPane(list);
    }

    /**
     * As {@link #createScrollPanedJList(Font)}, but the contents of the list are initialized to the passed in List
     * values.
     * 
     * @param font Null is allowed.
     * @param listdata
     * @return JScrollPane containing an embedded JList.
     */
    public static JScrollPane createScrollPanedJList(final Font font, final List listdata)
    {
        final JList list = new JList();
        if(font != null)
        {
            list.setFont(font);
        }
        list.setListData(listdata.toArray());
        return new JScrollPane(list);
    }

    /**
     * Creata a {@link JTextArea} and put it inside a {@link JScrollPane}. Return the JScrollPane. To get the JList that
     * is created within this method, call...<br>
     * <br>
     * (JTextArea)(<returned pane>.getViewport().getView()); <br>
     * <br>
     * where <returned pane> is the JScrollPane returned by this function. <br>
     * 
     * @param font Null is allowed.
     * @param displayedtext Initial text.
     * @param rows
     * @param columns
     * @param wrap
     * @param editable
     * @return JScrollPane with an embedded text area.
     */
    public static JScrollPane createScrollPanedJTextArea(final Font font,
                                                         final String displayedtext,
                                                         final int rows,
                                                         final int columns,
                                                         final boolean wrap,
                                                         final boolean editable)
    {
        final JTextArea text = new JTextArea(displayedtext, rows, columns);
        text.setLineWrap(wrap);
        text.setWrapStyleWord(true);
        text.setEditable(editable);
        if(font != null)
        {
            text.setFont(font);
        }
        return new JScrollPane(text);
    }

    /**
     * Creata a {@link JTextField} and put it inside a JScrollPane. Return the {@link JScrollPane}. To get the
     * JTextField that is created within this method, call...<br>
     * <br>
     * (JTextField)(<returned pane>.getViewport().getView()); <br>
     * <br>
     * where <returned pane> is the JScrollPane returned by this function. <br>
     * 
     * @param font
     * @param displayedtext
     * @param columns A negative or zero value implies columns are not set.
     * @param editable
     * @return JScrollPane around a text field with the vertical scroll bar turned off.
     */
    public static JScrollPane createScrollPanedJTextField(final Font font,
                                                          final String displayedtext,
                                                          final int columns,
                                                          final boolean editable)
    {
        final JTextField text = createJTextField(displayedtext, font, columns);
        text.setEditable(editable);

        final JScrollPane jsp = new JScrollPane(text);
        jsp.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER);
        return jsp;
    }

    /**
     * See {@link #createScrollPanedJTextField(Font, String, boolean)}, where columns is not set. It calls the
     * createJTextField that does not use the columns variable.
     * 
     * @param font
     * @param displayedtext
     * @param editable
     * @return JScrollPane around a text field with columns not set.
     */
    public static JScrollPane createScrollPanedJTextField(final Font font,
                                                          final String displayedtext,
                                                          final boolean editable)
    {
        return createScrollPanedJTextField(font, displayedtext, -1, editable);
    }

    /**
     * @param label
     * @param font Null is allowed.
     * @param listener Null is allowed.
     * @param ks Null is allowed.
     * @return {@link JMenuItem} with the given label, font (if set), listener (if set) and {@link KeyStroke} (if set).
     */
    public static JMenuItem createJMenuItem(final String label,
                                            final Font font,
                                            final ActionListener listener,
                                            final KeyStroke ks)
    {
        final JMenuItem item = new JMenuItem(label);
        if(font != null)
        {
            item.setFont(font);
        }
        if(listener != null)
        {
            item.addActionListener(listener);
        }
        if(ks != null)
        {
            item.setAccelerator(ks);
            //item.setText
        }

        return item;
    }

    /**
     * @param label
     * @param font Null is allowed.
     * @param listener Null is allowed.
     * @param ks Null is allowed.
     * @return A {@link JCheckBoxMenuItem} with the given label, font (if set), listener (if set) and {@link KeyStroke}
     *         (if set).
     */
    public static JCheckBoxMenuItem createJCheckBoxMenuItem(final String label,
                                                            final Font font,
                                                            final ActionListener listener,
                                                            final KeyStroke ks)
    {
        final JCheckBoxMenuItem item = new JCheckBoxMenuItem(label);
        if(font != null)
        {
            item.setFont(font);
        }
        if(listener != null)
        {
            item.addActionListener(listener);
        }
        if(ks != null)
        {
            item.setAccelerator(ks);
            //item.setText
        }
        item.setSelected(false);

        return item;
    }

    /**
     * @param label
     * @param font Null is allowed.
     * @param listener Null is allowed.
     * @param ks Null is allowed.
     * @return A {@link JRadioButtonMenuItem} with the given label, font (if set), listener (if set) and
     *         {@link KeyStroke} (if set).
     */
    public static JRadioButtonMenuItem createJRadioButtonMenuItem(final String label,
                                                                  final Font font,
                                                                  final ActionListener listener,
                                                                  final KeyStroke ks)
    {
        final JRadioButtonMenuItem item = new JRadioButtonMenuItem(label);
        if(font != null)
        {
            item.setFont(font);
        }
        if(listener != null)
        {
            item.addActionListener(listener);
        }
        if(ks != null)
        {
            item.setAccelerator(ks);
            //item.setText
        }
        item.setSelected(false);

        return item;
    }

    /**
     * Create a JPanel which contains nothing... useful for filler.
     * 
     * @return
     */
    public static JPanel createFillerJPanel()
    {
        final Canvas filler = new Canvas();
        final JPanel fillerpanel = new JPanel();
        fillerpanel.add(filler);
        return fillerpanel;
    }

    /**
     * Create a JPanel which contains nothing... useful for filler.
     * 
     * @return
     */
    public static JPanel createFillerJPanel(final Color backgroundColor)
    {
        final Canvas filler = new Canvas();
        filler.setBackground(backgroundColor);
        final JPanel fillerpanel = new JPanel();
        fillerpanel.setBackground(backgroundColor);
        fillerpanel.add(filler);
        return fillerpanel;
    }

    /**
     * @return A {@link JTextPane} which contains error message. HTML can be used.
     */
    public static JTextPane createErrorMessagePane(final String errMessage)
    {
        return createErrorMessagePane(null, errMessage);
    }

    /**
     * @return A {@link JTextPane} that displays a header in vold and a message in plain text.
     */
    public static JTextPane createErrorMessagePane(final String headerMessage, final String errMessage)
    {
        if(errMessage != null)
        {
            errMessage.replaceAll("\n", "<br>");
        }
        if(headerMessage != null)
        {
            headerMessage.replaceAll("\n", "<br>");
        }

        final StringBuffer sb = new StringBuffer("<HTML><div>");
        if(headerMessage != null && headerMessage.length() > 0)
        {
            sb.append("<h3>");
            sb.append(headerMessage);
            sb.append("</h3>");
        }
        sb.append("<p>" + errMessage + "</p>");
        sb.append("</div></HTML>");

        final JTextPane textPane = new JTextPane();
        textPane.setContentType("text/html");
        textPane.setText(sb.toString().trim());
        textPane.setEditable(false);

        return textPane;

    }

    /**
     * Add the {@link JLabel} and {@link JComboBox} to a {@link JPanel}, with a FlowLayout(FlowLayout.LEFT) layout.
     * 
     * @param label
     * @param combobox
     * @return JPanel containing a JLabel and JComboBox in a flow layout.
     */
    public static JPanel createLabeledComboBox(final JLabel label, final JComboBox combobox)
    {
        final JPanel panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        panel.add(label);
        panel.add(combobox);
        return panel;
    }

    /**
     * Add the {@link JLabel} and {@link JComboBox} to a JPanel, with a FlowLayout(FlowLayout.LEFT) layout. The
     * JComboBox is initialize to the select the provided value.
     * 
     * @param label
     * @param combobox
     * @param initvalue The initial value for the JComboBox.
     * @return JPanel containing a JLabel and JComboBox in a flow layout.
     */
    public static JPanel createLabeledComboBox(final JLabel label, final JComboBox combobox, final Object initvalue)
    {
        final JPanel panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        panel.add(label);
        panel.add(combobox);
        combobox.setSelectedItem(initvalue);
        return panel;
    }

    /**
     * Add the {@link JLabel} and {@link JTextField} to a {@link JPanel}, with a FlowLayout(FlowLayout.LEFT) layout.
     * 
     * @param label
     * @param text
     * @return JPanel containing a JLabel and JTextField in a flow layout.
     */
    public static JPanel createLabeledJTextField(final JLabel label, final JTextField text)
    {
        final JPanel panel = new JPanel(new FlowLayout(FlowLayout.LEFT));
        panel.add(label);
        panel.add(text);
        return panel;
    }

    /**
     * See {@link #createJSpinner(Font, int, String, Integer, Integer, Integer, Integer)}.
     * 
     * @param font Can be null.
     * @param numberOfColumns
     * @param initialValue
     * @param min Can be null (no min).
     * @param max Can be null (no max).
     * @param step
     * @return A {@link JSpinner} initialized for use.
     */
    public static JSpinner createJSpinner(final Font font,
                                          final int numberOfColumns,
                                          final Integer initialValue,
                                          final Integer min,
                                          final Integer max,
                                          final Integer step)
    {
        return createJSpinner(font, numberOfColumns, null, initialValue, min, max, step);
    }

    /**
     * Initialize a {@link JSpinner} using a NumberEditor with the given range.
     * 
     * @param font Can be null.
     * @param numberOfColumns
     * @param numberFormatString Can be null.
     * @param initialValue
     * @param min Can be null (no min).
     * @param max Can be null (no max).
     * @param step
     * @return
     */
    public static JSpinner createJSpinner(final Font font,
                                          final int numberOfColumns,
                                          final String numberFormatString,
                                          final Integer initialValue,
                                          final Integer min,
                                          final Integer max,
                                          final Integer step)
    {
        final SpinnerNumberModel spinnerModel = new SpinnerNumberModel(initialValue, min, max, step);
        final JSpinner spinner = new JSpinner(spinnerModel);
        if(numberFormatString != null)
        {
            //Work around for an annoying inconsistent font problem
            final Font oldFont = ((NumberEditor)spinner.getEditor()).getTextField().getFont();
            spinner.setEditor(new NumberEditor(spinner, numberFormatString));
            ((NumberEditor)spinner.getEditor()).getTextField().setFont(oldFont);
        }
        if(font != null)
        {
            ((NumberEditor)spinner.getEditor()).getTextField().setFont(font);
        }
        ((NumberEditor)spinner.getEditor()).getTextField().setColumns(numberOfColumns);
        return spinner;
    }

    /**
     * See {@link #createJSpinner(Font, int, String, Double, Double, Double, Double)}.
     * 
     * @param font Can be null.
     * @param numberOfColumns
     * @param initialValue
     * @param min Can be null (no min).
     * @param max Can be null (no max).
     * @param step
     * @return A {@link JSpinner} initialized for use.
     */
    public static JSpinner createJSpinner(final Font font,
                                          final int numberOfColumns,
                                          final Double initialValue,
                                          final Double min,
                                          final Double max,
                                          final Double step)
    {
        return createJSpinner(font, numberOfColumns, null, initialValue, min, max, step);
    }

    /**
     * Initialize a {@link JSpinner} using a NumberEditor with the given range.
     * 
     * @param font Can be null.
     * @param numberOfColumns
     * @param numberFormatString Can be null.
     * @param initialValue
     * @param min Can be null (no min).
     * @param max Can be null (no max).
     * @param step
     * @return
     */
    public static JSpinner createJSpinner(final Font font,
                                          final int numberOfColumns,
                                          final String numberFormatString,
                                          final Double initialValue,
                                          final Double min,
                                          final Double max,
                                          final Double step)
    {
        final SpinnerNumberModel spinnerModel = new SpinnerNumberModel(initialValue, min, max, step);
        final JSpinner spinner = new JSpinner(spinnerModel);
        if(numberFormatString != null)
        {
            final Font oldFont = ((NumberEditor)spinner.getEditor()).getTextField().getFont();
            spinner.setEditor(new NumberEditor(spinner, numberFormatString));
            ((NumberEditor)spinner.getEditor()).getTextField().setFont(oldFont);
        }
        if(font != null)
        {
            ((NumberEditor)spinner.getEditor()).getTextField().setFont(font);
        }
        ((NumberEditor)spinner.getEditor()).getTextField().setColumns(numberOfColumns);
        return spinner;
    }

    /**
     * See {@link #createJSpinner(Font, int, String, Float, Float, Float, Float)}.
     * 
     * @param font Can be null.
     * @param numberOfColumns
     * @param initialValue
     * @param min Can be null (no min).
     * @param max Can be null (no max).
     * @param step
     * @return A {@link JSpinner} initialized for use.
     */
    public static JSpinner createJSpinner(final Font font,
                                          final int numberOfColumns,
                                          final Float initialValue,
                                          final Float min,
                                          final Float max,
                                          final Float step)
    {
        return createJSpinner(font, numberOfColumns, null, initialValue, min, max, step);
    }

    /**
     * Initialize a {@link JSpinner} using a NumberEditor with the given range.
     * 
     * @param font Can be null.
     * @param numberOfColumns
     * @param numberFormatString Can be null.
     * @param initialValue
     * @param min Can be null (no min).
     * @param max Can be null (no max).
     * @param step
     * @return
     */
    public static JSpinner createJSpinner(final Font font,
                                          final int numberOfColumns,
                                          final String numberFormatString,
                                          final Float initialValue,
                                          final Float min,
                                          final Float max,
                                          final Float step)
    {
        final SpinnerNumberModel spinnerModel = new SpinnerNumberModel(initialValue, min, max, step);
        final JSpinner spinner = new JSpinner(spinnerModel);
        if(numberFormatString != null)
        {
            final Font oldFont = ((NumberEditor)spinner.getEditor()).getTextField().getFont();
            spinner.setEditor(new NumberEditor(spinner, numberFormatString));
            ((NumberEditor)spinner.getEditor()).getTextField().setFont(oldFont);
        }
        if(font != null)
        {
            ((NumberEditor)spinner.getEditor()).getTextField().setFont(font);
        }
        ((NumberEditor)spinner.getEditor()).getTextField().setColumns(numberOfColumns);
        return spinner;
    }

    /**
     * Creates a {@link JSplitPane} with the two passed in components. By default, the two components will have a
     * minimum size set to 100x100. If this is not a good number, then you need to set the minimum size outside of this
     * method after the pane is created. The divider size is set to 8.
     * 
     * @param splitOrientation Either JSplitPane.VERTICAL_SPLIT or JSplitPane.HORIZONTAL_SPLIT.
     * @param topOrLeftPanel The top or left side panel.
     * @param bottomOrRightPanel The bottom or right side panel.
     * @param continuousRedrawing IF true, then the component panels will be redrawn as the divider is moved. IF false,
     *            after the divider is moved.
     */
    public static JSplitPane createJSPlitPane(final int splitOrientation,
                                              final Component topOrLeftPanel,
                                              final Component bottomOrRightPanel,
                                              final boolean continuousRedrawing)
    {
        topOrLeftPanel.setMinimumSize(new Dimension(100, 100)); //100x100 is arbitrary
        bottomOrRightPanel.setMinimumSize(new Dimension(100, 100));

        final JSplitPane splitPane = new JSplitPane(splitOrientation);
        splitPane.setTopComponent(topOrLeftPanel);
        splitPane.setBottomComponent(bottomOrRightPanel);
        splitPane.setContinuousLayout(continuousRedrawing);
        splitPane.setDividerSize(8);
        return splitPane;
    }

    /**
     * Creates a {@link TitledBorder} with the specified parameters. The {@link LineBorder} drawing the line has the
     * specified color.
     * 
     * @param text
     * @param font Can be null.
     * @param lineColor CanNOT be null!
     * @return
     */
    public static TitledBorder createTitledBorder(final String text, final Font font, final Color lineColor)
    {
        final TitledBorder tb = new TitledBorder(new LineBorder(lineColor), text);
        if(font != null)
        {
            tb.setTitleFont(font);
        }
        return tb;
    }

    /**
     * Creates a {@link TitledBorder} with the specified parameters. The baseBorder provides the basis for the title
     * border. See {@link BorderFactory}.
     * 
     * @param baseBorder
     * @param text
     * @param font Can be null.
     * @return
     */
    public static TitledBorder createTitledBorder(final Border baseBorder, final String text, final Font font)
    {
        final TitledBorder tb = new TitledBorder(baseBorder, text);
        if(font != null)
        {
            tb.setTitleFont(font);
        }
        return tb;
    }

    /**
     * Creates a {@link ComponentTitledBorder} with the specified parameters.
     * 
     * @param borderedComponent {@link Component} to which the border is added.
     * @param baseBorder The base border.
     * @param componentInBorder The component to be within the border, usually a {@link JCheckBox} or {@link JLabel}.
     * @return A ComponentTitledBorder which extends Border.
     */
    public static ComponentTitledBorder createComponentTitledBorder(final JComponent borderedComponent,
                                                                    final Border baseBorder,
                                                                    final Component componentInBorder)
    {
        final ComponentTitledBorder tb = new ComponentTitledBorder(componentInBorder, borderedComponent, baseBorder);
        return tb;
    }

    /**
     * Return the font specified by the given string. The format of the string must be:<br>
     * <br>
     * [name][sep ch][style string][sep ch][size]<br>
     * <br>
     * name: the name of the font<br>
     * sep ch: the separator character passed in<br>
     * style string: the style to use, either "bold", "italic", "plain", or "bold-italic". Plain is the default if the
     * string is invalid.<br>
     * size: the font size (this will do no checking to see if size is valid for that font). 12 is the default if the
     * integer is invalid.<br>
     * <br>
     * The possible return values are:<br>
     * <br>
     * valid Font: this means the font String was processed, for better or worse.<br>
     * null: this means the font string was improperly formatted.<br>
     * <br>
     * THIS IS NOT USED ANYMORE, but may be useful in the future.
     * 
     * @param fontstr
     * @param separator
     * @return
     */
    @Deprecated
    public static Font processFontString(final String fontstr, final char separator)
    {
        //I'm going to segment the line using separator:
        final SegmentedLine segline = new SegmentedLine(fontstr, "" + separator, SegmentedLine.MODE_ALLOW_EMPTY_SEGS);

        //Make sure there are three segments, even if they are empty.
        if((segline.getNumberOfSegments() != 3) && (segline.getNumberOfSegments() != 4))
        {
            return null;
        }

        //Get the font Style integer
        int style = Font.PLAIN;
        final String stylestr = segline.getSegment(1);
        if(stylestr.equalsIgnoreCase("Plain"))
        {
            style = Font.PLAIN;
        }
        else if(stylestr.equalsIgnoreCase("Bold"))
        {
            style = Font.BOLD;
        }
        else if(stylestr.equalsIgnoreCase("ITALIC"))
        {
            style = Font.ITALIC;
        }
        else if(stylestr.equalsIgnoreCase("Bold-italic"))
        {
            style = Font.BOLD + Font.ITALIC;
        }

        //Get the size integer
        int size;
        try
        {
            size = Integer.parseInt(segline.getSegment(2));
        }
        catch(final NumberFormatException e)
        {
            size = 12;
        }

        //Return the font.
        return new Font(segline.getSegment(0), style, size);
    }

    /**
     * This generates a String legible by the processFontString command above for the passed in font. THIS IS NOT USED
     * ANYMORE, but may be useful in the future.
     * 
     * @param font
     * @param separator
     * @return
     */
    @Deprecated
    public static String generateFontString(final Font font, final char separator)
    {
        //Get the font Style integer
        String style = "Plain";
        if(font.getStyle() == Font.PLAIN)
        {
            style = "Plain";
        }
        else if(font.getStyle() == Font.BOLD)
        {
            style = "Bold";
        }
        else if(font.getStyle() == Font.ITALIC)
        {
            style = "ITALIC";
        }
        else if(font.getStyle() == Font.BOLD + Font.ITALIC)
        {
            style = "Bold-italic";
        }

        //Return the font.
        return "" + font.getName() + separator + style + separator + font.getSize();
    }

    /**
     * Generates a ChartDirector ready font string. "null" is returned if the passed in Font is null. THIS IS NOT USED
     * ANYMORE, but may be useful in the future.
     */
    @Deprecated
    public static String generateChartDirectorFontString(final Font font)
    {
        if(font == null)
        {
            return "null";
        }
        String result = "" + font.getName();
        if(font.isBold())
        {
            result += " Bold";
        }
        if(font.isItalic())
        {
            result += " Italic";
        }

        return result;
    }

    /**
     * Returns the coordinates extracted from a ChartDirector HTML Image Map string, as generated by, for example,
     * LegendBox getImageCoor(...) method. This looks for 'coords=' in the imageMapString and then picks up the string
     * following it.
     * 
     * @param coordString The properly formatted (according to ChartDirector):<br>
     * <br>
     *            area shape="rect" coords="[x1],[y1],[x2],[y2]"<br>
     * <br>
     * @return int[]... x1, y1, x2, y2. Opposite corners of the box. Or null if not a valid image map string.
     */
    public static int[] parseChartDirectorImageMapCoordinateString(final String imageMapString)
    {
        //Breakdown the string based on '='.  The last segment will be the coordinate
        //portion.
        final int index = imageMapString.indexOf("coords=");
        if(index < 0)
        {
            return null;
        }
        final int firstQuoteIndex = index + 7;
        final int lastQuoteIndex = imageMapString.indexOf('"', firstQuoteIndex + 1);
        if((firstQuoteIndex < 0) || (lastQuoteIndex < 0))
        {
            return null;
        }

        //Get the coordinate string.
        String theString = imageMapString.substring(firstQuoteIndex + 1, lastQuoteIndex);

        //Remove any quotes.
        theString = theString.replaceAll("\"", "");

        //Segment the coordinate portion.
        final SegmentedLine coords = new SegmentedLine(theString, ",", SegmentedLine.MODE_NO_EMPTY_SEGS);
        if(coords.getNumberOfSegments() != 4)
        {
            return null; //Something weird happened.
        }

        //Acquire the results.
        final int[] results = new int[4];
        try
        {
            results[0] = Integer.parseInt(coords.getSegment(0));
            results[1] = Integer.parseInt(coords.getSegment(1));
            results[2] = Integer.parseInt(coords.getSegment(2));
            results[3] = Integer.parseInt(coords.getSegment(3));
        }
        catch(final NumberFormatException nfe)
        {
            return null;
        }
        return results;
    }

    /**
     * Return true if the passed in Point is within the image map string passed in.
     * 
     * @param pt The Point to check.
     * @param mapString The image map string, of the format ChartDirector uses. This is processed herein using
     *            parseChartDirectorImageMapCoordinateString.
     */
    public static boolean isPointWithinChartDirectorImageCoordinates(final Point pt, final String mapString)
    {
        final SegmentedLine segLine = new SegmentedLine(mapString, "\n", SegmentedLine.MODE_NO_EMPTY_SEGS);
        int i;
        int[] coords;
        for(i = 0; i < segLine.getNumberOfSegments(); i++)
        {
            coords = HSwingFactory.parseChartDirectorImageMapCoordinateString(segLine.getSegment(i));
            if((pt.getX() > Math.min(coords[0], coords[2])) && (pt.getX() < Math.max(coords[0], coords[2]))
                && (pt.getY() > Math.min(coords[1], coords[3])) && (pt.getY() < Math.max(coords[1], coords[3])))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * This method calls a JFileChooser to load a file.<br>
     * <br>
     * This may belong somewhere else.
     * 
     * @param basedir
     * @return returns the file name if one is selected, "" if nothing is chosen, or null if an error occurs.
     */
    public static String chooseFileName(final String basedir)
    {
        //First, open up a file dialog and get the file name.
        final JFileChooser chooser = new JFileChooser(basedir);
        final int result = chooser.showOpenDialog(null);

        //Do nothing if cancel was clicked
        if(result == JFileChooser.CANCEL_OPTION)
        {
            return "";
        }

        //Was there an error?
        if(result == JFileChooser.ERROR_OPTION)
        {
            JOptionPane.showMessageDialog(null, "Unable to open dialog.", "ERROR", JOptionPane.ERROR_MESSAGE);
            return null;
        }

        //Get the file.
        final File atfile = chooser.getSelectedFile();

        return atfile.getPath();
    }
}
