package ohd.hseb.hefs.utils.gui.tools;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;

import javax.swing.GrayFilter;

/**
 * Code is from http://mindmeat.blogspot.com/2008/07/java-image-comparison.html and cleaned/refactored.<br>
 * <br>
 * Tool is used to compare images examining the "brightness" in regions within the image. Brightness is determined by
 * making the image gray-scale and then calling {@link Raster#getSample(int, int, int)}, averaging the samples across a
 * region, and computing the difference. The caller can control the number of rectangular regions (specifying x and y
 * sizes) a sensitivity threshold and a stabilization value that should be proportional to the number of regions (same
 * order of magnitude). The larger the stabilization value the smaller the computed average brightness levels will be.
 * 
 * @author Hank.Herr
 */
public class ImageComparator
{

    protected BufferedImage img1 = null;
    protected BufferedImage img2 = null;
    protected BufferedImage imgc = null;
    protected int xRegions = 0;
    protected int yRegions = 0;
    protected int sensitivityThreshold = 0;
    protected int stabilizationFactor = 10;
    protected boolean match = false;
    protected int debugMode = 0; // 1: textual indication of change, 2: difference of factors

    /**
     * Pass in two files to compare.
     * 
     * @throws MalformedURLException If call to {@link File#toURI()} followed by {@link URI#toURL()} yields a malformed
     *             URL.
     */
    public ImageComparator(final File file1, final File file2) throws MalformedURLException
    {
        if(!file1.exists())
        {
            throw new IllegalArgumentException("File not found: " + file1.getAbsolutePath());
        }
        if(!file2.exists())
        {
            throw new IllegalArgumentException("File not found: " + file2.getAbsolutePath());
        }
        initialize(SwingTools.loadImage(file1.toURI().toURL()), SwingTools.loadImage(file2.toURI().toURL()));
    }

    /**
     * Pass in two images for comparison.
     */
    public ImageComparator(final Image img1, final Image img2)
    {
        this(imageToBufferedImage(img1), imageToBufferedImage(img2));
    }

    /**
     * Pass in two buffere images.
     */
    public ImageComparator(final BufferedImage img1, final BufferedImage img2)
    {
        initialize(img1, img2);
    }

    /**
     * Sets the two image attributes and calls {@link #autoSetParameters()}.
     */
    private void initialize(final BufferedImage img1, final BufferedImage img2)
    {
        if((img1 == null) || (img2 == null))
        {
            throw new IllegalArgumentException("A BufferedImage provided is null.");
        }
        this.img1 = img1;
        this.img2 = img2;
        autoSetParameters();
    }

    /**
     * Creates default parameters.
     */
    protected void autoSetParameters()
    {
        xRegions = 10;
        yRegions = 10;
        sensitivityThreshold = 10;
        stabilizationFactor = 10;
    }

    /**
     * @param x The number of columns along x-axis.
     * @param y The number of rows along the y-axis.
     * @param factorA A threshold value. If the difference in brightness exceeds this, then the region is considered
     *            different.
     * @param factorD A stabilization factor. This should be proportional to x and y.
     */
    public void setParameters(final int x, final int y, final int factorA, final int factorD)
    {
        this.xRegions = x;
        this.yRegions = y;
        this.sensitivityThreshold = factorA;
        this.stabilizationFactor = factorD;
    }

    /**
     * @param m Set to 0 for not excess output; 1 for some; 2 for region by region average brightness differences.
     */
    public void setDebugMode(final int m)
    {
        this.debugMode = m;
    }

    /**
     * Performs the comparison and updates both {@link #imgc} and {@link #match}.
     */
    public void compare()
    {
        // Setup for dissimilarity image.
        imgc = imageToBufferedImage(img2);
        final Graphics2D gc = imgc.createGraphics();
        gc.setColor(Color.RED);

        // Converts to gray scale.
        img1 = imageToBufferedImage(GrayFilter.createDisabledImage(img1));
        img2 = imageToBufferedImage(GrayFilter.createDisabledImage(img2));

        // Sizes the regions of comparison.
        final int blocksx = img1.getWidth() / xRegions;
        final int blocksy = img1.getHeight() / yRegions;

        // Set to a match by default.
        this.match = true;

        //Loop over the y-regions.
        for(int y = 0; y < yRegions; y++)
        {
            //Output results region by region if debug turned on.
            if(debugMode > 0)
            {
                System.out.print("|");
            }

            for(int x = 0; x < xRegions; x++)
            {
                final int b1 = computeAverageBrightness(img1.getSubimage(x * blocksx,
                                                                         y * blocksy,
                                                                         blocksx - 1,
                                                                         blocksy - 1));
                final int b2 = computeAverageBrightness(img2.getSubimage(x * blocksx,
                                                                         y * blocksy,
                                                                         blocksx - 1,
                                                                         blocksy - 1));
                final int diff = Math.abs(b1 - b2);

                if(diff > sensitivityThreshold)
                { // the difference in a certain region has passed the threshold value of factorA
                      // draw an indicator on the change image to show where change was detected.
                    gc.drawRect(x * blocksx, y * blocksy, blocksx - 1, blocksy - 1);
                    this.match = false;
                }

                if(debugMode == 1)
                {
                    System.out.print((diff > sensitivityThreshold ? "X" : " "));
                }
                if(debugMode == 2)
                {
                    System.out.print(diff + (x < xRegions - 1 ? "," : ""));
                }
            }
            if(debugMode > 0)
            {
                System.out.println("|");
            }
        }
    }

    /**
     * @return Image recording where differences were found.
     */
    public BufferedImage getChangeIndicator()
    {
        return imgc;
    }

    /**
     * @return The average brightness across the image.
     */
    protected int computeAverageBrightness(final BufferedImage img)
    {
        final Raster r = img.getData();
        int total = 0;
        for(int y = 0; y < r.getHeight(); y++)
        {
            for(int x = 0; x < r.getWidth(); x++)
            {
                total += r.getSample(r.getMinX() + x, r.getMinY() + y, 0);
            }
        }
        return total / ((r.getWidth() / stabilizationFactor) * (r.getHeight() / stabilizationFactor));
    }

    public boolean match()
    {
        return this.match;
    }

    /**
     * Converts the generic {@link Image} to a {@link BufferedImage} by redrawing the image.
     */
    protected static BufferedImage imageToBufferedImage(final Image img)
    {
        if(img == null)
        {
            throw new IllegalArgumentException("Null image provided; cannot convert to buffered image.");
        }
        final BufferedImage bi = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
        final Graphics2D g2 = bi.createGraphics();
        g2.drawImage(img, null, null);
        return bi;
    }

}
