package ohd.hseb.hefs.utils.gui.tools;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.awt.image.MemoryImageSource;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

import sun.awt.image.ToolkitImage;

public abstract class SwingTools
{

    /**
     * Gets the nearest parent of a component of the specified class, or null if there is none. Never returns the
     * component itself.
     * 
     * @param <C> the class to search for
     * @param component the component to look through the parents of
     * @param klass the class you wish to retrieve
     * @return the first parent of the component with the specified class
     */
    @SuppressWarnings("unchecked")
    public static <C> C getParentOfClass(final Component component, final Class<C> klass)
    {
        Container container = component.getParent();
        while(container != null)
        {
            if(klass.isInstance(container))
            {
                return (C)container;
            }

            container = container.getParent();
        }
        return null;
    }

    /**
     * Gets the nearest parent of a component of the specified class, or null if there is none. Can return the component
     * itself.
     * 
     * @param <C> the class to search for
     * @param component the component to look through the parents of
     * @param klass the class you wish to retrieve
     * @return the first parent of the component with the specified class
     */
    @SuppressWarnings("unchecked")
    public static <C> C getParentOfClassIncludingSelf(final Component component, final Class<C> klass)
    {
        if(klass.isInstance(component))
        {
            return (C)component;
        }
        return getParentOfClass(component, klass);
    }

    /**
     * Gets the nearest parent of a component of a class with the specified annotation, or null if there is none. Will
     * return the component itself if applicable.
     * 
     * @param component the component to look through the parents of
     * @param annotationClass the class of annotation to look for
     * @return the first parent of the component with the specified annotation
     */
    public static Component getParentWithAnnotation(Component component,
                                                    final Class<? extends Annotation> annotationClass)
    {
        if(component == null)
        {
            return null;
        }
        Annotation annotation = component.getClass().getAnnotation(annotationClass);
        while(annotation == null && component != null)
        {
            component = component.getParent();
            if(component != null)
            {
                annotation = component.getClass().getAnnotation(annotationClass);
            }
        }
        return component;
    }

    /**
     * @param component
     * @return Parent component that should serve as the dialog parent for the given component. The component is
     *         identified by searching the parents via {@link #getParentWithAnnotation(Component, Class)} where the
     *         annotation is {@link GlobalDialogParent}. If no such parent is found, the original component is returned;
     *         null is never returned.
     */
    public static Component getGlobalDialogParent(final Component component)
    {
        final Component c = getParentWithAnnotation(component, GlobalDialogParent.class);
        if(c != null)
        {
            return c;
        }
        return component;
    }

    /**
     * Calls {@link SwingUtilities#invokeLater(Runnable)} to call {@link Component#setVisible(boolean)}. This may avoid
     * some thread lock issues.
     * 
     * @param c
     * @param visible
     */
    public static void setVisibleInvokeLater(final Component c, final boolean visible)
    {
        SwingUtilities.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                c.setVisible(visible);
            }
        });

    }

    /**
     * @param item
     * @param combobox
     * @return true if the item is found; false otherwise.
     */
    public static boolean searchJComboBoxForString(final String item, final JComboBox<String> combobox)
    {
        int i;
        for(i = 0; i < combobox.getItemCount(); i++)
        {
            //Break out if I've found the plottype.
            if(item.equalsIgnoreCase(combobox.getItemAt(i)))
            {
                break;
            }
        }

        if(i == combobox.getItemCount())
        {
            return false;
        }
        return true;
    }

    /**
     * @param resourceName Used in {@link ClassLoader#getSystemResource(String)}.
     * @return An {@link ImageIcon} loaded from the given resource or null if the resource is not found.
     */
    public static ImageIcon loadImageIcon(final String resourceName)
    {
        final URL url = ClassLoader.getSystemResource(resourceName);
        ImageIcon icon = null;
        if(url != null)
        {
            icon = new ImageIcon(url);
        }
        return icon;
    }

    /**
     * @param imageSrc {@link URL} specifying an image file.
     * @return {@link BufferedImage} loaded from the specifed URL. Null is returned if an exception occurs.
     */
    public static BufferedImage loadImage(final URL imageSrc)
    {
        BufferedImage bi = null;
        int width, height;
        try
        {
            bi = ImageIO.read(imageSrc);
            if(bi == null)
            {
                throw new IllegalArgumentException("Provided image src, " + imageSrc
                    + ", does not correspond to an image.");
            }
            width = bi.getWidth(null);
            height = bi.getHeight(null);
            if(bi.getType() != BufferedImage.TYPE_INT_RGB)
            {
                final BufferedImage bi2 = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
                final Graphics big = bi2.getGraphics();
                big.drawImage(bi, 0, 0, null);
                bi = bi2;
            }
        }
        catch(final IOException e)
        {
            return null;
        }

        return bi;
    }

    /**
     * @param baseImage The image to draw into a new image in the upper left corner.
     * @param targetDimensions The desired dimensions of the new image.
     * @return A new image with the provided target dimensions that contains the provided image drawn without change in
     *         size in the upper left corner. Any space in the new image that is not drawn over will be left as empty
     *         (transparent).
     */
    public static Image expandImageWithEmptySpace(final Image baseImage, final Dimension targetDimensions)
    {
        final BufferedImage bufImage =
                                     new BufferedImage((int)targetDimensions.getWidth(),
                                                       (int)targetDimensions.getHeight(),
                                                       BufferedImage.TYPE_INT_ARGB);

        final Graphics2D graphics = bufImage.createGraphics();
        graphics.setBackground(ColorTools.TRANSPARENT_BLACK);
        graphics.clearRect(0, 0, (int)targetDimensions.getWidth(), (int)targetDimensions.getHeight());
        graphics.drawImage(baseImage, 0, 0, null);
        return bufImage;
    }

    /**
     * Used to populate a pixel array from the given image within the provided pixels. Called by
     * {@link #createMemoryImageSource(Image, Dimension)}.
     */
    private static void populatePixelArrayFromImage(final int[] pixels,
                                                    final BufferedImage baseImage,
                                                    final int rowSize)
    {
        for(int row = 0; row < baseImage.getHeight(null); row++)
        {
            for(int col = 0; col < baseImage.getWidth(null); col++)
            {
//                System.err.println("####>> " + row + ", " + col + ", " + baseImage.getRGB(col, row) + " -- "
//                    + new Color(baseImage.getRGB(col, row), true) + " -- "
//                    + new Color(baseImage.getRGB(col, row), true).getAlpha());
                pixels[row * rowSize + col] = baseImage.getRGB(col, row);
            }
        }
    }

    /**
     * Creates a memory image source with the given base image drawn in the upper corner. Coding is not complete for
     * this method: it assumes a 32x32 size image. DO NOT USE THIS YET!
     * 
     * @param baseImage
     * @param desiredSize
     * @return
     */
    public static MemoryImageSource createMemoryImageSource(final Image baseImage, final Dimension desiredSize)
    {
        final int[] pixels = new int[(int)desiredSize.getWidth() * (int)desiredSize.getHeight()];

        final BufferedImage bufImage = new BufferedImage(baseImage.getWidth(null),
                                                         baseImage.getHeight(null),
                                                         BufferedImage.TYPE_INT_ARGB);
        ((ToolkitImage)baseImage).getBufferedImage();
        final Graphics g = bufImage.getGraphics();
        g.drawImage(baseImage, 0, 0, null);
        populatePixelArrayFromImage(pixels, bufImage, 32);
        return new MemoryImageSource(32, 32, pixels, 0, 32);
    }

//
//    public static Image createCursorImage(final BufferedImage baseImage)
//    {
//        
//        int[] pixels = new int[baseImage.getWidth(null) * baseImage.getHeight(null)];
//        for (int row = 0; row < baseImage.getWidth(null); row ++)
//        {
//            for (int col = 0; col < baseImage.getHeight(null); col ++)
//            {
//                pixels[row * baseImage.getHeight(null) + col ] = baseImage.getRGB(row, col);
//            }
//        }
//        
//        final Image bgImage = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(baseImage.getWidth(null),
//                                                                                            baseImage.getHeight(null),
//                                                                                            pixels,
//                                                                                            0,
//                                                                                            baseImage.getWidth(null)));
//        final Graphics g = bgImage.getGraphics();
//        g.drawImage(baseImage, 0, 0, null);
//        return bgImage;
//    }

    /**
     * Return the index that the String found in the {@link JComboBox} has. This does a search that is NOT
     * case-sensitive!
     * 
     * @param item
     * @param combobox
     * @return Index of the item in the combobox or -1 if not found.
     */
    public static int findStringInJComboBox(final String item, final JComboBox<String> combobox)
    {
        int i;
        for(i = 0; i < combobox.getItemCount(); i++)
        {
            //Break out if I've found the plottype.
            if(item.equalsIgnoreCase(combobox.getItemAt(i)))
            {
                break;
            }
        }

        if(i == combobox.getItemCount())
        {
            return -1;
        }
        return i;
    }

    /**
     * This is useful for any component that makes use of a {@link ListModel}, including a {@link JComboBox}.
     * 
     * @return The list implied by the {@link ListModel} provided.
     */
    public static <T> List<T> getListModelList(final ListModel<T> listModel)
    {
        final ArrayList<T> choices = new ArrayList<>();
        final int num = listModel.getSize();
        for(int i = 0; i < num; i++)
        {
            choices.add(listModel.getElementAt(i));
        }
        return choices;
    }

    /**
     * This creates a JOptionPane message dialog that is NOT modal! Note that the displayed object MUST be a JComponent.
     * A String will NOT work!!! Also, the parent MUST be a Frame subclass!
     * 
     * @param parent
     * @param message
     * @param title
     * @param messagetype
     * @param height
     * @param width
     */
    public static void showNonModalMessageDialog(final Frame parent,
                                                 final JComponent message,
                                                 final String title,
                                                 final int messagetype,
                                                 final int height,
                                                 final int width)
    {
        final JOptionPane pane = new JOptionPane(message, messagetype);

        //Turn on the okay/cancel property
        pane.setOptionType(JOptionPane.DEFAULT_OPTION);

        //Create the dialog to contain the pane.
        //JDialog dialog = pane.createDialog(parent, title);
        final JDialog dialog = new JDialog(parent, title);
        dialog.getContentPane().add(pane);
        dialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        dialog.setSize(height, width);
        dialog.setVisible(true);
        dialog.setModal(false);
    }

    /**
     * Return a grid bag constrainsts object. See the parameters for how they are used. <br>
     * Sometimes... sometimes... it may be useful to use createJFillerPanel-s to create extra rows or columns to
     * establish the number of grid cells for others to use. This is particularly useful if the components are to be of
     * inconsistent size taking up inconsistent numbers of cells horizontally or vertically. By create filler panels,
     * you are telling the grid bag layout precisely how many vertical or horizontal cells there are... rather than
     * letting it figure it out on its own (which can be chaotic). See IVPStatisticChooserManager, createDisplay, for a
     * coding example.<br>
     * <br>
     * 
     * @param gridx Grid column
     * @param gridy Grid row
     * @param gridwidth Number of grid cells in width
     * @param gridheight Number of grid cells in height
     * @param weightx Relative width of component cells compared with other cells; useful when amount of space available
     *            exceeds the minimum space required.
     * @param weighty Relative height
     * @param anchor How to display component when it is smaller than the grid cell (GridBagConstraints.CENTER, EAST,
     *            NORTH, etc.)
     * @param fill Which dimensions of component should grow when the component is smaller than the grid cell
     *            (GridBagConstraints.NONE, BOTH, HORIZONTAL, VERTICAL)
     * @param insets Insets to use within the grid cell
     * @param ipadx Padding to place on either side of the component within the cell. This will increase the size of a
     *            component beyond its default minimum size.
     * @param ipady Same, but top and bottom.
     * @return
     */
    public static GridBagConstraints returnGridBagConstraints(final int gridx,
                                                              final int gridy,
                                                              final int gridwidth,
                                                              final int gridheight,
                                                              final double weightx,
                                                              final double weighty,
                                                              final int anchor,
                                                              final int fill,
                                                              final Insets insets,
                                                              final int ipadx,
                                                              final int ipady)
    {
        final GridBagConstraints gbc = new GridBagConstraints();

        gbc.gridx = gridx;
        gbc.gridy = gridy;
        gbc.gridwidth = gridwidth;
        gbc.gridheight = gridheight;
        gbc.weightx = weightx;
        gbc.weighty = weighty;
        gbc.anchor = anchor;
        gbc.fill = fill;
        gbc.insets = insets;
        gbc.ipadx = ipadx;
        gbc.ipady = ipady;

        return gbc;
    }

    /**
     * @return A {@link JPanel} in {@link FlowLayout} using the passed in layout option, such as
     *         {@link FlowLayout#CENTER}.
     */
    public static JPanel wrapInFlowLayout(final Component c, final int layoutOption)
    {
        final JPanel panel = new JPanel(new FlowLayout(layoutOption));
        panel.add(c);
        return panel;
    }

    /**
     * Wraps the given component in a {@link JPanel}, set to position {@link BorderLayout#CENTER}.
     * 
     * @param c the component to wrap
     * @return the wrapping JPanel
     */
    public static JPanel wrap(final Component c)
    {
        return wrap(c, false);
    }

    /**
     * Wraps the given component in a {@link JPanel}, set to position {@link BorderLayout#CENTER}.
     * 
     * @param c the component to wrap
     * @param transferSizes if true, set the wrapping container's minimum, preferred, and maximum sizes appropriately.
     * @return the wrapping JPanel
     */
    public static JPanel wrap(final Component c, final boolean transferSizes)
    {
        final JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());
        panel.add(c, BorderLayout.CENTER);
        if(transferSizes)
        {
            panel.setMinimumSize(c.getMinimumSize());
            panel.setPreferredSize(c.getPreferredSize());
            panel.setMaximumSize(c.getMaximumSize());
        }
        return panel;
    }

    /**
     * This is useful if the popup trigger is not compatible with how software is written. For example, for a FEWS
     * chart, right button press triggers the popup menu on Linux and Mac. However, it also starts the panning. So, if
     * the chart is panned, the popup menu will be displayed on top of the chart as it is panned. This is used before
     * the event is processed by JFreeChart code to turn off the popup for mouse pressed.
     * 
     * @param baseEvent The starting mouse event.
     * @param trigger The trigger to use.
     * @return A new {@link MouseEvent} with the popup trigger (i.e., if true, then the popup menu will be triggered)
     *         set to the given value.
     */
    public static MouseEvent setPopupTrigger(final MouseEvent baseEvent, final boolean trigger)
    {

        return new MouseEvent((Component)baseEvent.getSource(),
                              baseEvent.getID(),
                              baseEvent.getWhen(),
                              baseEvent.getModifiersEx(),
                              baseEvent.getX(),
                              baseEvent.getY(),
                              baseEvent.getXOnScreen(),
                              baseEvent.getYOnScreen(),
                              baseEvent.getClickCount(),
                              trigger, //This will never be considered the popup trigger.
                              baseEvent.getButton());
    }

    /**
     * For whatever reason, sometimes it is necessary to set a component's visibility to false then true in order to
     * force a full repaint. This performs that action, but only if the component's visible flag is true.
     * 
     * @param c Component to forcibly refresh.
     */
    public static void forceComponentRedraw(final Component c)
    {
        if(c.isVisible())
        {
            c.setVisible(false);
            c.setVisible(true);
        }
    }

    /**
     * Adds a {@link HierarchyListener} to the provided component so that if it is displayed in a {@link JOptionPane}
     * generated dialog, the dialog will become resizeable.
     * 
     * @param editorComponent The component displayed in a dialog.
     */
    public static void makeJOptionPaneResizeable(final Component editorComponent)
    {
        editorComponent.addHierarchyListener(new HierarchyListener()
        {
            @Override
            public void hierarchyChanged(final HierarchyEvent e)
            {
                final Window window = SwingUtilities.getWindowAncestor(editorComponent);
                if(window instanceof Dialog)
                {
                    final Dialog dialog = (Dialog)window;
                    if(!dialog.isResizable())
                    {
                        dialog.setResizable(true);
                    }
                }
            }
        });
    }
}
