package ohd.hseb.hefs.utils.jobs;

import java.awt.Component;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.log4j.LoggingTools;

/**
 * Superclass for all jobs that are processed as part of the ohd.hseb.hefs.utils.jobs package. Any job should extend
 * this, override the {@link #processJob()} method, and then override or set other methods and attributes as needed.<br>
 * <br>
 * The following is required of any job:<br>
 * <br>
 * 1. When a failure occurs, call {@link #fireProcessJobFailure(Exception, boolean)} passing in appropriate arguments.
 * <br>
 * <br>
 * 2. When done successfully, call {@link #endTask()}. By default, endTask calls
 * {@link #fireProcessSuccessfulJobCompletion()}. Optionally, you can override {@link #endTask()} to do other stuff.<br>
 * <br>
 * 3. Periodically (whenever reasonable) check the returns of {@link #isCanceled()} ({@link GenericJob} method) and
 * {@link #isJobInterrupted()}. If either is true, call {@link #fireProcessJobFailure(Exception, boolean)} indicating
 * user cancellation or unexpected interruption and exit the {@link #processJob()} method. Note that if a user clicks
 * the Cancel button, the {@link #_canceled} flag will be set via {@link #setCanceledDoNotUpdateProgress(boolean)}, so
 * that any {@link #_jobMonitor} remains unaffected. <br>
 * <br>
 * Note that the {@link #_canceled} flag indicates to the job that it needs to stop, whereas the {@link #_done} flag
 * indicates that the job is already stopped. Hence, when {@link #setDone(boolean)} is called, any {@link #_jobMonitor}
 * will close, whereas there are two set canceled methods that can be called,
 * {@link #setCanceledAndUpdateProgress(boolean)} and {@link #setCanceledDoNotUpdateProgress(boolean)} which will update
 * the progress immediately or not (respectively). If the job does not shut down immediately, it is generally best to
 * call {@link #setCanceledDoNotUpdateProgress(boolean)} so that any monitoring dialogs remain open until the job stops.
 * <br>
 * <br>
 * When the job is started, the {@link GenericJobInternalThread} instance {@link #_runningThread} is started. That
 * thread will call this job's {@link #setDone(boolean)} method upon completion (success or failure), meaning any
 * monitoring dialogs will close. Bottom line: any implementation of this job need not call
 * {@link GenericJob#setDone(boolean)} and, in most cases, {@link GenericJob#setCanceledAndUpdateProgress(boolean)},
 * unless it wants to force monitoring dialogs to close before the job is fully complete, perhaps giving this job time
 * to react to a success after the job is done.
 * 
 * @author hank.herr
 */
public abstract class GenericJob
{
    private static final Logger LOG = LogManager.getLogger(GenericJob.class);

    /**
     * Indicates if the job was canceled, probably because of a user clicking on a cancel button somewhere.
     */
    private boolean _canceled = false;

    /**
     * Indicates if the job is done. When a job's {@link GenericJobInternalThread} completes, this flag will be set to
     * true indicating that the job is completed and any progress displays can be made invisible.
     */
    private boolean _done = false;

    /**
     * The {@link JobMonitor} associated with this job.
     */
    private JobMonitor _jobMonitor = null;

    /**
     * Current {@link JobMonitorAttr} indicating job progress. Note that this is a linked list (internally to
     * {@link JobMonitorAttr}), with each instance in the list corresponding to a subjob noted as such in an job monitor
     * (e.g., multilevel progress bars).
     */
    private JobMonitorAttr _jobMonitorAttr = null;

    /**
     * Stores {@link JobListener} instances.
     */
    private final List<JobListener> _jobListeners = new ArrayList<JobListener>();

    /**
     * {@link Component} relative to which messages should be displayed pertaining to this {@link GenericJob}.
     */
    private Component _parentComponent;

    /**
     * Defaults to 'GenericJob'. This name is used within the name of the {@link GenericJobInternalThread} that runs the
     * job.
     */
    private String _name = "GenericJob";

    /**
     * {@link Thread} used to run this job when {@link #startJob()} is called. A wrapper is used so that if any
     * exception occurs while trying to run the {@link Thread}, {@link #fireProcessJobFailure(Exception, boolean)} is
     * called indicating a failure for any listeners.
     */
    private GenericJobInternalThread _runningThread = null;

    /**
     * Loops over {@link #_jobListeners} and calls each
     * {@link JobListener#processJobFailure(Exception, GenericJob, boolean)} passing in the provided display message
     * flag and this as the {@link GenericJob} instance.
     * 
     * @param exc Exception associated with the failure.
     * @param displayMessage If a message should be display through the interface explaining the failure.
     */
    protected void fireProcessJobFailure(final Exception exc, final boolean displayMessage)
    {
        for(int i = 0; i < this._jobListeners.size(); i++)
        {
            _jobListeners.get(i).processJobFailure(exc, this, displayMessage);
        }
    }

    /**
     * Loops over {@link #_jobListeners} and calls each {@link JobListener#processSuccessfulJobCompletion(GenericJob)}.
     */
    protected void fireProcessSuccessfulJobCompletion()
    {
        for(int i = 0; i < this._jobListeners.size(); i++)
        {
            _jobListeners.get(i).processSuccessfulJobCompletion(this);
        }
    }

    public void addListener(final JobListener listener)
    {
        this._jobListeners.add(listener);
    }

    public void removeListener(final JobListener listener)
    {
        this._jobListeners.remove(listener);
    }

    public void clearListeners()
    {
        this._jobListeners.clear();
    }

    /**
     * Override this method as needed to perform special stuff when a task is completed, successfully or not. It should
     * be called within the subclass {@link #processJob()} when a job ends without any issues. Note that this does not
     * call {@link #setDone(boolean)}. Rather, by default, it just calls {@link #fireProcessSuccessfulJobCompletion()}.
     */
    public void endTask()
    {
        this.fireProcessSuccessfulJobCompletion();
    }

    /**
     * See {@link #_canceled} flag description.
     */
    public boolean isCanceled()
    {
        return _canceled;
    }

    /**
     * See {@link #_done} flag description.
     * 
     * @return
     */
    public boolean isDone()
    {
        return _done;
    }

    /**
     * Calls {@link JobMonitor#updateProgress(JobMonitorAttr)} for the current {@link #_jobMonitor}, so that a change in
     * progress is shown.
     */
    private void progressStatusChanged()
    {
        if(getJobMonitor() != null)
        {
            getJobMonitor().updateProgress(getTopLevelJobMonitorAttr());
        }
    }

    /**
     * Increment the progress and update the progress display.
     */
    public void madeProgress()
    {
        madeProgress(null);
    }

    /**
     * Set the note, increment the progess, and update the progress display.
     * 
     * @param note
     */
    public void madeProgress(final String note)
    {
        final JobMonitorAttr attr = getCurrentAttributes();
        if(note != null)
        {
            attr.setNote(note);
        }
        attr.incrementProgress();
        progressStatusChanged();
    }

    /**
     * Set the note and update the progress display.
     * 
     * @param note
     */
    public void updateNote(final String note)
    {
        final JobMonitorAttr attr = getCurrentAttributes();
        if(note != null)
        {
            attr.setNote(note);
        }
        progressStatusChanged();
    }

    /**
     * Removes the job monitor attribute, allowing for one job to be run multiple times. The next time a job monitor
     * attribute is needed, it will be created from scratch.
     */
    public void clearJobMonitorAttr()
    {
        setJobMonitorAttr(null);
    }

    /**
     * Resets the progress made to 0, reseting the progress display.
     */
    public void resetMonitorProgress()
    {
        getCurrentAttributes().setProgress(0);
    }

    /**
     * Creates a new subjob and updates the progress display.
     */
    public void newMonitorSubJob()
    {
        getCurrentAttributes().newChild();
        progressStatusChanged();
    }

    /**
     * Clears the deepest subjob and updates the progress display.
     */
    public void clearMonitorSubJob()
    {
        getTopLevelJobMonitorAttr().getDeepestChild().clearMyself();
        progressStatusChanged();
    }

    /**
     * If the {@link #getJobMonitor()} returns a {@link Component}, this will call
     * {@link SwingTools#setVisibleInvokeLater(Component, boolean)} for it. This allows for a job to make its monitor
     * temporarily invisible if it needs to open another dialog while running (on some systems, like Linux via cygwin,
     * when the second dialog is displayed, the monitor will be displayed on top of it making it hard to interact with).
     * 
     * @param b The visibility flag to set.
     */
    protected void setJobMonitorComponentVisibility(final boolean b)
    {
        if(getJobMonitor() instanceof Component)
        {
            //Calling setVisible directly can cause thread lock!
            SwingTools.setVisibleInvokeLater((Component)getJobMonitor(), b);
        }
    }

    public void setName(final String name)
    {
        _name = name;
    }

    public String getName()
    {
        return _name;
    }

    /**
     * The parent frame is needed in case any JOptionPanes or other dialogs (including the progress panel) need to open
     * up a dialog.
     * 
     * @return The parent frame passed into the initialize method.
     */
    public Component getParentComponent()
    {
        return _parentComponent;
    }

    /**
     * The parent component is needed in case any JOptionPanes or other dialogs (including the progress panel) need to
     * open up a dialog.
     * 
     * @param c The parent component passed into the initialize method.
     */
    public void setParentComponent(final Component c)
    {
        _parentComponent = c;
    }

    /**
     * @return An associated job monitor, typically a {@link HJobMonitorDialog}.
     */
    public JobMonitor getJobMonitor()
    {
        return _jobMonitor;
    }

    /**
     * @return The first attributes associated with this job, which tracks the starting point.
     */
    private JobMonitorAttr getTopLevelJobMonitorAttr()
    {
        if(_jobMonitorAttr == null)
        {
            _jobMonitorAttr = new JobMonitorAttr();
        }

        return _jobMonitorAttr;
    }

    /**
     * Set the canceled flag, but do NOT update progress (leave it up to the job to react). This is called if a cancel
     * button is clicked, given the job time to process the cancel before closing the dialog.
     * 
     * @param canceled
     */
    public void setCanceledDoNotUpdateProgress(final boolean canceled)
    {
        //XXX Useful for debug puroses:  GeneralTools.dumpStackTrace();
        _canceled = canceled;
    }

    /**
     * Set the canceled flag as specified and call {@link #progressStatusChanged()}.
     * 
     * @param canceled
     */
    public void setCanceledAndUpdateProgress(final boolean canceled)
    {
        //XXX Useful for debug puroses:  GeneralTools.dumpStackTrace();
        this._canceled = canceled;
        progressStatusChanged();
    }

    /**
     * @return The attributes associated with the lowest level subjob that is being performed.
     */
    private JobMonitorAttr getCurrentAttributes()
    {
        return getTopLevelJobMonitorAttr().getDeepestChild();
    }

    /**
     * Set the status of {@link #getCurrentAttributes()} to match the attributes passed in.
     * 
     * @param attributes Specifies the progress integer, min, max, and note.
     */
    public void setCurrentJobStatus(final JobMonitorAttr attributes)
    {
        getCurrentAttributes().setMinimum(attributes.getMinimum());
        getCurrentAttributes().setMaximum(attributes.getMaximum());
        getCurrentAttributes().setNote(attributes.getNote());
        getCurrentAttributes().setProgress(attributes.getProgress());
        progressStatusChanged();
    }

    /**
     * Make the deepest subjob indeterminate (not progress counter) and update the progress display.
     * 
     * @param indeterminateFlag
     */
    public void setIndeterminate(final boolean indeterminateFlag)
    {
        getCurrentAttributes().setIndeterminate(indeterminateFlag);
        progressStatusChanged();
    }

    /**
     * Set the maximum number of steps for the deepest subjob and update the progress display.
     * 
     * @param max
     */
    public void setMaximumNumberOfSteps(final int max)
    {
        getCurrentAttributes().setMaximum(max);
        progressStatusChanged();
    }

    /**
     * Set the done flag as specified and call {@link #progressStatusChanged()}.
     * 
     * @param done
     */
    public void setDone(final boolean done)
    {
        this._done = done;
        progressStatusChanged();
    }

    public void setJobMonitor(final JobMonitor jobMonitor)
    {
        this._jobMonitor = jobMonitor;
    }

    protected void setJobMonitorAttr(final JobMonitorAttr progressEntity)
    {
        this._jobMonitorAttr = progressEntity;
    }

    /**
     * @return {@link Thread#isAlive()} for the running wrapper, unless that is not defined, in which case it returns
     *         false.
     */
    public boolean isJobRunning()
    {
        if(_runningThread == null)
        {
            return false;
        }
        return _runningThread.isAlive();
    }

    /**
     * Calls {@link Thread#interrupt()} for the wrapped thread, {@link #_runningThread}.
     */
    public void interruptJob()
    {
        _runningThread.interrupt();
    }

    /**
     * @return The return of {@link Thread#isInterrupted()} for the wrapped thread.
     */
    public boolean isJobInterrupted()
    {
        return _runningThread.isInterrupted();
    }

    /**
     * Creates {@link #_runningThread}, if it does not yet exist, and then calls the thread's {@link Thread#start()}
     * method.
     */
    public synchronized void startJob()
    {
        //Construct a wrapper around this GenericJob that catches any unhandled exceptions/throwables from the job's run()
        //method and dumps a stacktrace as well as signaling any listeners that the job failed and 
        _runningThread = new GenericJobInternalThread();
        _runningThread.start();
    }

    /**
     * Runs the task that the job must perform. This must be overridden by any instance of {@link GenericJob}.
     */
    public abstract void processJob();

    /**
     * This wraps a {@link GenericJob} and ensures that, when run, ANY {@link Throwable} that is thrown will force a
     * call to the job's {@link GenericJob#fireProcessJobFailure(Exception, boolean)} with an exception describing the
     * failure and true. It will also dumpe a stack trace; in general its best for GenericJob's to catch their own
     * exceptions. This is just a fail safe to ensure that any monitor closes if a job fails unpredictably.<br>
     * <br>
     * It also calls the job's {@link GenericJob#setDone(boolean)} method with true upon completion of the job (either
     * success or failure). This means {@link GenericJob} instances need not call {@link GenericJob#setDone(boolean)}
     * when done; just return from the run method.
     * 
     * @author hankherr
     */
    protected class GenericJobInternalThread extends Thread
    {
        public GenericJobInternalThread()
        {
            setName(GenericJob.this.getName());
        }

        public GenericJob getWrappedJob()
        {
            return GenericJob.this;

        }

        @Override
        public void interrupt()
        {
            //XXX Useful for debug purposes: GeneralTools.dumpStackTrace();
            super.interrupt();
        }

        @Override
        public void run()
        {
            try
            {
                GenericJob.this.processJob();
            }
            catch(final Throwable t)
            {
                LoggingTools.outputStackTraceAsDebug(LOG, t);

                Exception exc;
                if(t instanceof Exception)
                {
                    exc = (Exception)t;
                }
                else
                {
                    exc = new Exception("Failed to complete job; cause: " + t.getMessage());
                    exc.setStackTrace(t.getStackTrace());
                }
                GenericJob.this.fireProcessJobFailure(exc, true);
            }
            finally
            {
                //Make sure any listeners know that the job is done.
                //In addition to other reactions, this will ensure an job monitors are closed.
                GenericJob.this.setDone(true);
            }
        }
    }

}
