package ohd.hseb.hefs.utils.jobs;

import java.awt.Component;
import java.util.LinkedList;
import java.util.Queue;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

/**
 * Manages a list of jobs to perform. Each is an instance of {@link GenericJob}. It is assumed only one job can run at a
 * time, so that any job added while another is running will be placed on the list of jobs. However, there is
 * functionality for clearing other waiting jobs before adding the job to the list.<br>
 * <br>
 * The {@link HJobManager} is also a {@link JobListener}. When a job is added, this is set as a listener. It will
 * display a {@link JOptionPane} message when a job fails with the displayMessage flag of true. It will trigger the next
 * waiting job whenever the current job fails or succeeds.
 * 
 * @author hank.herr
 */
public class HJobManager implements JobListener
{
    private GenericJob _currentJob = null;

    private final Queue<GenericJob> _jobQueue = new LinkedList<GenericJob>();

    private Component _parentComponent = null;

    public HJobManager(final Component parentComponentForMessages)
    {
        this._parentComponent = parentComponentForMessages;
    }

    /**
     * Add the job so that it is executed at the next opportunity.
     * 
     * @param job
     */
    public synchronized void addJob(final GenericJob job)
    {
        job.addListener(this);
        _jobQueue.add(job);

        job.setIndeterminate(true);
        job.updateNote("Waiting for previous jobs to complete... ");

        startNextJob();
    }

    /**
     * Clear the waiting list before adding.
     * 
     * @param job
     */
    public synchronized void addJobAndClearOtherWaitingJobs(final GenericJob job)
    {
        clearWaitingJobs();
        addJob(job);
    }

    public synchronized void clearWaitingJobs()
    {
        _jobQueue.clear();
    }

    public synchronized void cancelRunningJob()
    {
        if(_currentJob != null)
        {
            _currentJob.setCanceledAndUpdateProgress(true);
        }
    }

    public boolean isJobRunning()
    {
        return _currentJob != null && !_currentJob.isDone();
    }

    public boolean isJobWaiting()
    {
        return !_jobQueue.isEmpty();
    }

    @Override
    public void processJobFailure(final Exception exc, final GenericJob theJob, final boolean displayMessage)
    {
        if(displayMessage)
        {
            //This displays the message independently of other things that may be happening.  For example, 
            //if the job indicated it has ended and has an associated dialog, that should be closed before
            //this message opens.  This will allow that.  Also, this will allow the next job to start processing
            //without waiting for this job to end.
            SwingUtilities.invokeLater(new Runnable()
            {
                @Override
                public void run()
                {
                    JOptionPane.showMessageDialog(_parentComponent,
                                                  exc.getMessage(),
                                                  "Error Executing Job",
                                                  JOptionPane.ERROR_MESSAGE);
                }
            });
        }
        _currentJob = null;
        startNextJob();
    }

    @Override
    public void processSuccessfulJobCompletion(final GenericJob theJob)
    {
        _currentJob = null;
        startNextJob();
    }

    /**
     * Start the next non-cancelled job in the queue.
     */
    private void startNextJob()
    {
        if((_currentJob == null) && (_jobQueue.size() > 0))
        {
            _currentJob = _jobQueue.poll();
            while(_currentJob.isCanceled())
            {
                if(_jobQueue.size() == 0)
                {
                    return;
                }
                _currentJob = _jobQueue.poll();
            }
            _currentJob.startJob();
        }
    }

}
