package ohd.hseb.hefs.utils.jobs;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.StringTools;

/**
 * Dialog designed to display the status of {@link GenericJob} implementers. Messages and progress are displayed in this
 * dialog, which monitors the status of the {@link GenericJob} in order to determine progress. Alternatively, the
 * {@link JobMessenger} static methods can be used; see {@link JobMessenger} for more info. <br>
 * Recommended usage is to create an instance of {@link GenericJob} and add a {@link JobListener} to it. Then create an
 * instance of this providing a parent, title, the job being monitored, and flag indicating if it can be canceled or
 * closed. Then start the job and make the dialog visible (see caps note below), in that order.<br>
 * <br>
 * Default behavior is for the dialog to always be one top but NOT be modal; i.e., methods
 * {@link #setAlwaysOnTop(boolean)} called with true and {@link #setModal(boolean)} called with false. You must decide
 * if the user should be able to multitask while the job is being performed: if not, make this dialog modal and call
 * {@link #setAlwaysOnTop(boolean)} with false (it becomes redundant and I'm not sure it will work properly). If
 * multitasking is allowed, then you need to decide if you want this dialog to always be on top and set the flag
 * appropriately.<br>
 * <br>
 * IF THE DIALOG WILL BE MODAL, BE SURE TO START THE JOB BEFORE MAKING THIS DIALOG VISIBLE, otherwise the thread will
 * lock waiting for this dialog to close before the job even runs. <br>
 * <br>
 * Finally, set this to be visible when you are ready to display the status.<br>
 * <br>
 * Examples can be found by searching using the constructors.
 * 
 * @author hank.herr
 */
public class HJobMonitorDialog extends JDialog implements ActionListener, JobMonitor
{
    private static final long serialVersionUID = 2699127301487982477L;

    final static String CLASSNAME = "HJobMonitorDialog";

    private Component _parentUsedForLocation = null;

    private JButton _cancelButton = null;
    private JButton _closeButton = null;
    private List<JProgressBar> _progressBars = null;
    private List<JLabel> _noteLabels = null;
    private JPanel _centerPanel = null;

    /**
     * Records the constraints to use for the next label and bar.
     */
    private GridBagConstraints _nextConstraints = null;

    private GenericJob _monitoredJob = null;

    private boolean _canCancel = false;
    private boolean _canClose = false;
    private int _numBars = 0;

    public HJobMonitorDialog(final Component parent, final String title, final boolean canCancel)
    {
        this(parent, title, null, canCancel, false);
    }

    public HJobMonitorDialog(final Component parent, final String title, final GenericJob job, final boolean canCancel)
    {
        this(parent, title, job, canCancel, false);
    }

    /**
     * @param parent Parent {@link Component} associated with locating this dialog.
     * @param title The title.
     * @param job The job being monitored.
     * @param canCancel If the dialog should include a cancel.
     * @param canClose If the dialog should include a close (i.e., close the dialog by settings its visibility to false
     *            but leave the job running and this dialog still in existence).
     */
    public HJobMonitorDialog(final Component parent,
                             final String title,
                             final GenericJob job,
                             final boolean canCancel,
                             final boolean canClose)
    {
        _parentUsedForLocation = parent;
        setTitle(title);
        setJob(job);
        setCanCancel(canCancel);
        setCanClose(canClose);
        job.setJobMonitor(this);

        createDisplay();
    }

    @Override
    public void actionPerformed(final ActionEvent evt)
    {
        if(evt.getSource() == _closeButton)
        {
            SwingTools.setVisibleInvokeLater(this, false);
        }
        else if(evt.getSource() == _cancelButton)
        {
            setTitle(getTitle() + " (CANCELED!)");
            _cancelButton.setEnabled(false);
            if(_monitoredJob != null)
            {
                _monitoredJob.setCanceledDoNotUpdateProgress(true);
            }
        }
    }

    public boolean canCancel()
    {
        return _canCancel;
    }

    public boolean canClose()
    {
        return _canClose;
    }

    /**
     * Create the display.
     */
    private void createDisplay()
    {
        _noteLabels = new ArrayList<JLabel>();
        _progressBars = new ArrayList<JProgressBar>();

        final JPanel buttonPanel = new JPanel(new FlowLayout());
        if(canClose())
        {
            _closeButton = HSwingFactory.createJButton("Close", null, this);
            buttonPanel.add(_closeButton);
            this.setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
        }
        else
        {
            this.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        }

        if(canCancel())
        {
            _cancelButton = HSwingFactory.createJButton("Cancel", null, this);
            buttonPanel.add(_cancelButton);
        }

        //Create the inner panel.
        _centerPanel = new JPanel();
        _centerPanel.setLayout(new BoxLayout(_centerPanel, BoxLayout.Y_AXIS));
        _centerPanel.setLayout(new GridBagLayout());

        //Create the outer panel.
        final JPanel progressPanel = new JPanel();
        progressPanel.setLayout(new GridBagLayout());
        Insets theinsets = new Insets(10, 10, 5, 10);

        //The center panel takes up the top space.
        theinsets = new Insets(5, 10, 5, 10);
        GridBagConstraints constraints = SwingTools.returnGridBagConstraints(0,
                                                                             0,
                                                                             1,
                                                                             1,
                                                                             1,
                                                                             0,
                                                                             GridBagConstraints.WEST,
                                                                             GridBagConstraints.BOTH,
                                                                             theinsets,
                                                                             0,
                                                                             0);
        progressPanel.add(_centerPanel, constraints);

        //This filler panel will swallow up the extra space created when bars are removed.
        //If all bars are used, this should be as small as possible; hence 0'd insets.
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          1,
                                                          GridBagConstraints.CENTER,
                                                          GridBagConstraints.NONE,
                                                          new Insets(0, 0, 0, 0),
                                                          0,
                                                          0);
        progressPanel.add(new JPanel(), constraints);

        //Button panel at the bottom.
        theinsets = new Insets(5, 10, 5, 10);
        constraints = SwingTools.returnGridBagConstraints(0,
                                                          2,
                                                          1,
                                                          1,
                                                          1,
                                                          0,
                                                          GridBagConstraints.SOUTH,
                                                          GridBagConstraints.HORIZONTAL,
                                                          theinsets,
                                                          0,
                                                          0);
        progressPanel.add(buttonPanel, constraints);

        getContentPane().add(progressPanel);
        pack();
        setLocationRelativeTo(_parentUsedForLocation);
        setAlwaysOnTop(true);
    }

    public GenericJob getJob()
    {
        return _monitoredJob;
    }

    /**
     * @return True if {@link #_monitoredJob} is null or for {@link #_monitoredJob}, the value of
     *         {@link GenericJob#isCanceled()} or {@link GenericJob#isDone()} is true.
     */
    public boolean isTaskDone()
    {
        if(_monitoredJob == null)
        {
            return true;
        }
        else
        {
            return _monitoredJob.isCanceled() || _monitoredJob.isDone();
        }
    }

    protected void setCanCancel(final boolean _canCancel)
    {
        this._canCancel = _canCancel;
    }

    protected void setCanClose(final boolean canClose)
    {
        this._canClose = canClose;
    }

    protected void setJob(final GenericJob job)
    {
        this._monitoredJob = job;
    }

    @Override
    public void setTitle(final String title)
    {
        super.setTitle(title);
    }

    /**
     * Wraps {@link GenericJob#startJob()} for {@link #_monitoredJob} and shows a message dialog indicating the job
     * failed to start if an exception occurs.
     */
    public void start()
    {
        try
        {
            if(_monitoredJob != null)
            {
                _monitoredJob.startJob();
            }
        }
        catch(final Exception e)
        {
            JOptionPane.showMessageDialog(this,
                                          "Failed to process task:\n" + e.getMessage(),
                                          "Error Processing Task",
                                          JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     * Force this dialog to be invisible if {@link #_monitoredJob} is done or canceled. It will then call
     * {@link #dispose()} for this. Otherwise, it will set the visible flag accordingly to the boolean provided.
     */
    @Override
    public void setVisible(final boolean b)
    {
        if(_monitoredJob.isDone() || _monitoredJob.isCanceled())
        {
            super.setVisible(false);
            HJobMonitorDialog.this.dispose();
        }
        else
        {
            super.setVisible(b);
        }
    }

    @Override
    public synchronized void updateProgress(final JobMonitorAttr entity)
    {
        //Invoke later must be done to avoid threadlock.  Threadlock can occur with each setVisible and even
        //with calls to HSwingFactory.createJLabel, particularly when running through Cygwin.  Here, I've wrapped
        //the entire content in a Runnable (which handles the HSwingFactory threadlock) and then individually
        //wrapped calls the setVisible (which handles most cases... those that are triggered with setVisible).  
        SwingUtilities.invokeLater(new Runnable()
        {
            @Override
            public void run()
            {
                if(_monitoredJob == null)
                {
                    return;
                }

                if(_monitoredJob.isDone() || _monitoredJob.isCanceled())
                {
                    SwingTools.setVisibleInvokeLater(HJobMonitorDialog.this, false); //Invoking setVisible later!
                    HJobMonitorDialog.this.dispose();
                    return;
                }

                //Initialize the _nextConstraints if not defined.
                if(_nextConstraints == null)
                {
                    _nextConstraints = SwingTools.returnGridBagConstraints(0,
                                                                           0,
                                                                           1,
                                                                           1,
                                                                           1,
                                                                           0,
                                                                           GridBagConstraints.WEST,
                                                                           GridBagConstraints.BOTH,
                                                                           new Insets(5, 5, 0, 5),
                                                                           0,
                                                                           0);
                }

                // Add placeholder bars and labels as necessary.  These are added but never removed, allowing the panel to grow
                // as necessary.  Hence the reason it must remember the next constraints to use.
                while(entity.getDepth() > _numBars)
                {
                    final JLabel label = HSwingFactory.createJLabel("Waiting for job initialization ...", null, null);
                    final JPanel labelPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
                    labelPanel.add(label);
                    _noteLabels.add(label);
                    _centerPanel.add(labelPanel, _nextConstraints);
                    _nextConstraints.gridy++;

                    final JProgressBar bar = new JProgressBar(0, 100);
                    bar.setValue(0);
                    bar.setStringPainted(true);
                    bar.setBorder(BorderFactory.createLoweredBevelBorder());
                    _progressBars.add(bar);
                    _centerPanel.add(bar, _nextConstraints);
                    _nextConstraints.gridy++;
                    _numBars++;

                    pack();
                }

                // Loop through lots of things at the same time.
                final Iterator<JLabel> labelIter = _noteLabels.iterator();
                final Iterator<JProgressBar> barIter = _progressBars.iterator();
                JobMonitorAttr currentEntity = entity;
                while(currentEntity != null)
                {
                    //Set the label messages for display later.
                    if(labelIter.hasNext())
                    {
                        final JLabel label = labelIter.next();
                        label.setText(StringTools.htmlizeString(currentEntity.getNote()));
                        SwingTools.setVisibleInvokeLater(label, true); //Invoking setVisible later!
                    }

                    //Setup the bar
                    if(barIter.hasNext())
                    {
                        final JProgressBar bar = barIter.next();
                        bar.setIndeterminate(currentEntity.isIndeterminate());
                        bar.setMaximum(currentEntity.getMaximum());
                        bar.setMinimum(currentEntity.getMinimum());
                        bar.setValue(currentEntity.getProgress());
                        bar.setStringPainted(!bar.isIndeterminate());
                        SwingTools.setVisibleInvokeLater(bar, true); //Invoking setVisible later!
                    }

                    currentEntity = currentEntity.getChild();
                }

                // Make invisible rest of the bars.
                while(labelIter.hasNext())
                {
                    SwingTools.setVisibleInvokeLater(labelIter.next(), false); //Invoking setVisible later!
                    SwingTools.setVisibleInvokeLater(barIter.next(), false); //Invoking setVisible later!
                }

                //pack();
            }
        });

    }

    /*
     * Visual testing of dialog.
     */
    public static void main(final String[] args)
    {
        // Job that does nothing but take time.
        final class TestJob extends GenericJob
        {
            TestJob()
            {
                final JobMonitorAttr attr = new JobMonitorAttr();
                attr.setMaximum(14);
                attr.setNote("Doing nothing...");
                setJobMonitorAttr(attr);
            }

            @Override
            public void processJob()
            {
                try
                {
                    for(int i = 0; i < 14; i++)
                    {
                        //Top level job
                        if(Math.random() < 0.4)
                        {
                            Thread.sleep(500);
                        }
                        else
                        {
                            newMonitorSubJob();
                            setMaximumNumberOfSteps(6);
                            updateNote("Sub Job # " + i);
                            for(int j = 0; j < 6; j++)
                            {
                                //First subjob
                                Thread.sleep(250);

                                if(Math.random() < 0.4)
                                {
                                    Thread.sleep(500);
                                }
                                else
                                {
                                    //Second subjob
                                    newMonitorSubJob();
                                    setMaximumNumberOfSteps(6);
                                    updateNote("Sub Job # " + i + " - " + j);
                                    for(int k = 0; k < 6; k++)
                                    {
                                        Thread.sleep(250);
                                        madeProgress();
                                    }
                                    clearMonitorSubJob();
                                }

                                madeProgress();
                            }
                            clearMonitorSubJob();
                        }

                        madeProgress();
                    }
                }
                catch(final InterruptedException e)
                {
                    e.printStackTrace();
                    setCanceledAndUpdateProgress(true);
                    endTask();
                    fireProcessJobFailure(new Exception("Error occurred while executing step: " + e.getMessage()), true);
                }
                setDone(true);
                endTask();
                fireProcessSuccessfulJobCompletion();
                System.exit(0);
            }
        }

        final TestJob job = new TestJob();
        final HJobMonitorDialog dialog = new HJobMonitorDialog(null, "Test", job, true, true);
        dialog.setMinimumSize(new Dimension(700, 10));
        dialog.setModal(true);
        job.startJob();
        dialog.setVisible(true);
    }
}
