package ohd.hseb.hefs.utils.jobs;

import ohd.hseb.hefs.utils.jobs.GenericJob.GenericJobInternalThread;

/**
 * This class only works if the {@link GenericJob} is started directly (e.g., job.start()) and not via an external
 * thread (e.g., new Thread(job).start()). These methods call {@link Thread#currentThread()} and, if it is an instance
 * of {@link GenericJob}, updates its progress and displayed messages. These tools are useful if the caller does not
 * have access to the GenericJob that is running it or may not necessarily be getting called via a GenericJob (in which
 * case, these method do nothing).
 * 
 * @author hank.herr
 */
public abstract class JobMessenger
{

    /**
     * @return For the current thread (i.e., {@link Thread#currentThread()}), if the thread is an instance of
     *         {@link GenericJobInternalThread}, this returns the value of
     *         {@link GenericJobInternalThread#getWrappedJob()}. If the thread is a {@link GenericJob}, it returns the
     *         thread. Otherwise null is returned.
     */
    private static GenericJob getParentJob()
    {
        final Thread thread = Thread.currentThread();
        if(thread instanceof GenericJobInternalThread)
        {
            return ((GenericJobInternalThread)thread).getWrappedJob();
        }
//        if(thread instanceof GenericJob)
//        {
//            return (GenericJob)thread;
//        }
        return null;
    }

    /**
     * Increment the progress of the job without a new message.
     */
    public static void madeProgress()
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.madeProgress();
        }
    }

    /**
     * Increment the progress of the job and change the message.
     * 
     * @param note
     */
    public static void madeProgress(final String note)
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.madeProgress(note);
        }
    }

    /**
     * Sets the status from scratch based on the passed in attributes.
     */
    public static void setJobStatus(final JobMonitorAttr statusAttributes)
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.setCurrentJobStatus(statusAttributes);
        }
    }

    /**
     * Update the message without incrementing the progress. Useful for indeterminate jobs.
     * 
     * @param note
     */
    public static void updateNote(final String note)
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.updateNote(note);
        }
    }

    /**
     * Create a new sub-task within the overall task. It will have its own progress bar.
     */
    public static void newMonitorSubJob()
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.newMonitorSubJob();
        }
    }

    /**
     * Clear the current sub-task. It will result in removing the progress bar.
     */
    public static void clearMonitorSubJob()
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.clearMonitorSubJob();
        }
    }

    /**
     * Makes the current task/job indeterminate or not based on the flag.
     * 
     * @param indeterminateFlag
     */
    public static void setIndeterminate(final boolean indeterminateFlag)
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.setIndeterminate(indeterminateFlag);
        }
    }

    /**
     * Sets the maximum number of steps for the current task/job.
     * 
     * @param max The maximum number of steps.
     */
    public static void setMaximumNumberOfSteps(final int max)
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            job.setMaximumNumberOfSteps(max);
        }
    }

    /**
     * @return If the current task has been canceled.
     */
    public static boolean isCanceled()
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            return job.isCanceled();
        }
        return false;
    }

    /**
     * @return If the current task is done.
     */
    public static boolean isDone()
    {
        final GenericJob job = getParentJob();
        if(job != null)
        {
            return job.isDone();
        }
        return false;
    }
}
