package ohd.hseb.hefs.utils.jtree;

import java.awt.Cursor;
import java.awt.Point;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetContext;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JTree;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

/**
 * Used internally in DragAndDropJTree. It receives the drop actions and reacts accordingly by calling the attached
 * TreeDropTargetListeners. This assumes that leafs can be dragged into nodes, but not other leaves.
 * 
 * @author hank.herr
 */
public class TreeDropTarget implements DropTargetListener
{

    @SuppressWarnings("unused")
    private final DropTarget _target;

    private final JTree _targetTree;

    private final List<TreeDropTargetListener> _listeners = new ArrayList<TreeDropTargetListener>();

    public TreeDropTarget(JTree targetTree)
    {
        _targetTree = targetTree;
        _target = new DropTarget(targetTree, this);
    }

    public void addListener(TreeDropTargetListener listener)
    {
        this._listeners.add(listener);
    }

    public void removeListener(TreeDropTargetListener listener)
    {
        this._listeners.remove(listener);
    }

    public void fireDropRequested(TreeNode[] nodesBeingDropped, TreeNode nodeDroppedInto, int dropAction)
    {
        for(int i = 0; i < _listeners.size(); i++)
        {
            _listeners.get(i).dropRequested(nodesBeingDropped, nodeDroppedInto, dropAction);
        }
    }

    /**
     * @param dtde The DropTargetDragEvent.
     * @return The TreeNode to which the event points. If the user releases the mouse button, this is where the drop
     *         woudl occur.
     */
    private TreeNode getNodeForEvent(DropTargetDragEvent dtde)
    {
        Point p = dtde.getLocation();
        DropTargetContext dtc = dtde.getDropTargetContext();
        JTree tree = (JTree)dtc.getComponent();
        TreePath path = tree.getClosestPathForLocation(p.x, p.y);
        return (TreeNode)path.getLastPathComponent();
    }

    @Override
    public void dragEnter(DropTargetDragEvent dropEvent)
    {
        TreeNode node = getNodeForEvent(dropEvent);
        if(node.isLeaf())
        {
            dropEvent.rejectDrag();
        }
        else
        {
            dropEvent.acceptDrag(DnDConstants.ACTION_MOVE);
        }
    }

    @Override
    public void dragOver(DropTargetDragEvent dropEvent)
    {
        TreeNode node = getNodeForEvent(dropEvent);
        if(node.isLeaf())
        {
            dropEvent.rejectDrag();
        }
        else
        {
            dropEvent.acceptDrag(DnDConstants.ACTION_MOVE);
        }
    }

    @Override
    public void dragExit(DropTargetEvent dte)
    {
    }

    @Override
    public void dropActionChanged(DropTargetDragEvent dtde)
    {
    }

    @Override
    public void drop(DropTargetDropEvent dropEvent)
    {
        this._targetTree.setCursor(Cursor.getDefaultCursor());

        //Get the path for the drop
        Point pt = dropEvent.getLocation();
        DropTargetContext dtc = dropEvent.getDropTargetContext();
        JTree tree = (JTree)dtc.getComponent();
        TreePath parentPath = tree.getClosestPathForLocation(pt.x, pt.y);
        DefaultMutableTreeNode nodeDroppedInto = (DefaultMutableTreeNode)parentPath.getLastPathComponent();
        if(nodeDroppedInto.isLeaf())
        {
            dropEvent.rejectDrop();
            return;
        }

        try
        {
            Transferable transferable = dropEvent.getTransferable();
            DataFlavor[] flavors = transferable.getTransferDataFlavors();
            for(int i = 0; i < flavors.length; i++)
            {
                if(transferable.isDataFlavorSupported(flavors[i]))
                {
                    dropEvent.acceptDrop(dropEvent.getDropAction());
                    TreePath[] treePathsToMove = (TreePath[])transferable.getTransferData(flavors[i]);

                    TreeNode[] nodesBeingDropped = new TreeNode[treePathsToMove.length];
                    for(int j = 0; j < nodesBeingDropped.length; j++)
                    {
                        nodesBeingDropped[j] = (TreeNode)treePathsToMove[j].getLastPathComponent();
                    }

                    fireDropRequested(nodesBeingDropped, nodeDroppedInto, dropEvent.getDropAction());
                    dropEvent.dropComplete(true);
                    return;
                }
            }
            dropEvent.rejectDrop();
        }
        catch(Exception e)
        {
            e.printStackTrace();
            dropEvent.rejectDrop();
        }
    }
}
