package ohd.hseb.hefs.utils.junit;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.FormattedMessageFactory;

import junit.framework.TestCase;
import nl.wldelft.util.LogUtils;
import ohd.hseb.hefs.utils.adapter.HEFSModelAdapter;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.util.misc.HString;
import ohd.hseb.util.misc.MiscTools;


/**
 * General high-level framework class for testing {@link HEFSModelAdapter} implementations.
 * 
 * @author hankherr
 */
public abstract class AdapterTestFramework extends TestCase
{
	
	private static  Logger LOG ;
	
    public AdapterTestFramework()
    {
        super();
    }
    
    public void setUp() throws Exception
    {
        super.setUp();
        LogUtils.initConsole();
        LOG = LogManager.getLogger(AdapterTestFramework.class);
    } 

    public AdapterTestFramework(final String testName)
    {
        super(testName);
    }
    

    /**
     * Removes the output directory and recreates it, essentially blanking out previous test results. The work directory
     * is created if it does not already exist. This does not remove anything from work.
     * 
     * @param The testing base directory for all scenarios.
     * @param scenarioName The subdirectory name within the base directory for this scenario.
     * @return The scenario specific base directory.
     */
    protected File prepScenario(final String scenarioName)
    {
        //Clear out the contents under output.
        File dir = new File(buildScenarioTestBaseDir(scenarioName) + "/output");
        if(dir.exists())
        {
            try
            {
                FileUtils.deleteDirectory(dir);
            }
            catch(final IOException e)
            {
            }
        }
        dir.mkdir();

        //Ensure work exists.  Its up to the individual test to decide if it must be cleared;
        dir = new File(buildScenarioTestBaseDir(scenarioName) + "/work");
        if(clearWorkDir(scenarioName))
        {
            if(dir.exists())
            {
                try
                {
                    FileUtils.deleteDirectory(dir);
                }
                catch(final IOException e)
                {
                }
            }
        }
        if(!dir.exists())
        {
            dir.mkdir();
        }

        return new File(buildScenarioTestBaseDir(scenarioName));
    }

    /**
     * If the test requires that the work directory be removed before each scenario run, have this return true.
     * 
     * @return True if the work directory is to be cleared before each test scenario run, false otherwise (e.g., if
     *         there is data in the directory that must be available and cannot be removed). The default return is true.
     */
    protected boolean clearWorkDir(final String scenarioName)
    {
        return true;
    }

    /**
     * Checks all files within the scenario's output dir to see if it matches a benchmark within the scenario dir. A
     * benchmark must have the same name as the output file but with a 'benchmark_' prefix. The files/directories ., ..,
     * and diag.xml are ignored.
     * 
     * @param scenarioName
     */
    protected void checkOutputBenchmarks(final String scenarioName)
    {
        final String outputDirName = buildScenarioTestBaseDir(scenarioName) + "/output/";
        final String benchDirName = buildScenarioTestBaseDir(scenarioName) + "/";

        //Listing of output files.
        final File outputDir = new File(outputDirName);
        final String[] files = outputDir.list();

        //Listing of benchmark files.  Should produce one output per bench.
        final File benchDir = new File(benchDirName);
        final String[] benchFiles = benchDir.list(new FilenameFilter()
        {
            @Override
            public boolean accept(final File dir, final String name)
            {
                return name.startsWith("benchmark_");
            }

        });
        @SuppressWarnings("unchecked")
        final List<Object> benchFileList = MiscTools.listFromArray(benchFiles);
        LOG.info("For " + scenarioName + ", benchmarks file: " + HString.buildStringFromList(benchFileList, "; ") + ".");

        //Check for no output dir.
        if(files == null)
        {
            fail("Output directory " + outputDirName + " was not created.");
        }

        //For each file...
        for(int i = 0; i < files.length; i++)
        {
            if((!files[i].equals("diag.xml")) && (!files[i].equals(".")) && (!files[i].equals("..")))
            {
                final File outputFile = new File(outputDirName + files[i]);
                final File benchFile = new File(benchDirName + "benchmark_" + files[i]);

                if(files[i].endsWith(".png"))
                {
                    LOG.info("File " + files[i] + " appears to be an image.  Skipping benchmark check.");
                    benchFileList.remove("benchmark_" + files[i]);
                }
                else
                {
                    try
                    {
                        assertTrue("For " + files[i] + ", output file does not match benchmark.",
                                   FileComparisonUtilities.areTwoFileContentsTheSame(outputFile, benchFile));
                        benchFileList.remove("benchmark_" + files[i]);
                        LOG.info("For " + scenarioName + " output files " + outputFile + " and " + benchFile
                            + " match.");
                    }
                    catch(final IOException e)
                    {
                        e.printStackTrace();
                        fail("Unexpected exception " + e.getClass().getName() + ": " + e.getMessage());
                    }
                }
            }
        }

        //Check bencfile list for any remaining files.  If found, fail out...
        if(!benchFileList.isEmpty())
        {
            fail("The following benchmark files were not found to have any output corresponding to them: "
                + benchFileList.toString() + ".");
        }
    }

    /**
     * Calls {@link #runTestScenario(String, boolean, Class, String[])} with a null expectedDiagnosticsText array.
     */
    protected void runTestScenario(final String scenarioName,
                                   final boolean expectFailure,
                                   final Class<? extends HEFSModelAdapter> adapterClass)
    {
        runTestScenario(scenarioName, expectFailure, adapterClass, (String[])null);
    }

    /**
     * Runs the test scenario.
     * 
     * @param scenarioName The scenario name.
     * @param expectFailure If false, then {@link #checkOutputBenchmarks(String)} will be called to compare all files in
     *            the output directory, except for diag.xml, with benchmark files in the scenario main directory. If
     *            true, then the text 'Error executing model:' will be looked for in the diag.xml file, since that
     *            message is generated by the top level {@link HEFSModelAdapter} code.
     * @param adapterClass The class of the {@link HEFSModelAdapter} subclass to run.
     * @param expectedDiagnosticsText Array of Strings to checkfor in the diag.xml file. The test will fail for the
     *            first string that is found to not be present in the diagnostics. Pass in a null or empty array for no
     *            checking.
     */
    protected void runTestScenario(final String scenarioName,
                                   final boolean expectFailure,
                                   final Class<? extends HEFSModelAdapter> adapterClass,
                                   final String... expectedDiagnosticsText)
    {
    	
    	LogUtils.initConsole();
        final File scenarioBaseDir = prepScenario(scenarioName);
        if(!validateReady())
        {
            return;
        }

        final String[] args = {scenarioBaseDir.getAbsolutePath() + "/run_info.xml"};

        try {
			HEFSModelAdapter.runAdapter(args, adapterClass.getName());
		} catch (final IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

        //No failure expected... look at benchmarks.
        if(!expectFailure)
        {
            checkOutputBenchmarks(scenarioName);
        }
        //Error expected... look for this message which is part of the top level HEFSModelAdapter.
        else
        {
            checkExpectedTextInDiagnosticsFile(scenarioName, "Error executing model:");
        }
        if(expectedDiagnosticsText != null)
        {
            for(final String text: expectedDiagnosticsText)
            {
                checkExpectedTextInDiagnosticsFile(scenarioName, text);
            }
        }
    }

    /**
     * @param scenarioName Name of scenario to check.
     * @param expectedText Text to look for.
     */
    private void checkExpectedTextInDiagnosticsFile(final String scenarioName, final String expectedText)
    {
        try
        {
            assertTrue("For scenario " + scenarioName + " expected text, \"" + expectedText
                           + "\", not found in diag.xml file.",
                       FileTools.doesFileContainText(new File(buildScenarioTestBaseDir(scenarioName)
                           + "/output/diag.xml"), expectedText));
        }
        catch(final IOException e)
        {
            fail("Unable to access diag.xml file for scenario '" + scenarioName + "': " + e.getMessage());
        }
    }

    /**
     * @param scenarioName
     * @return The full scenario directory based on the {@link #getUnitTestBaseDir()} and the given scenarioName.
     */
    private String buildScenarioTestBaseDir(final String scenarioName)
    {
        return getUnitTestBaseDir() + "/" + scenarioName;
    }

    /**
     * Must be overridden! This method is responsible for any messaging that must be done to inform the user why the
     * test was not performed. Also, if the reason for not performing constitutes a test case failure, then this method
     * should call {@link TestCase#fail()} with an appropriate message.
     * 
     * @return True if the test can be performed, false if not for any reason. For example, this may check for FEWS
     *         PI-service connectivity before performing a test.
     */
    protected abstract boolean validateReady();

    /**
     * Must be overridden!
     * 
     * @return The base directory to store the test information for all scenarios.
     */
    protected abstract String getUnitTestBaseDir();

}
