package ohd.hseb.hefs.utils.junit;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.newLinkedList;
import static com.google.common.collect.Maps.newHashMap;

import java.awt.image.BufferedImage;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.net.MalformedURLException;
import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;

import org.apache.commons.io.FileUtils;

import junit.framework.TestCase;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.utils.gui.tools.ImageComparator;
import ohd.hseb.hefs.utils.gui.tools.SwingTools;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.StreamTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

/**
 * General file comparison utilities intended for use in unit testing. This extends {@link TestCase} so that it can
 * 
 * @author Hank.Herr
 */
public class FileComparisonUtilities extends TestCase
{
    public void test1()
    {
        //DoNothing -- this gets rid of a warning!!!
    }

    /**
     * ( Wraps {@link #areTwoFileContentsTheSame(File, File)}, calling {@link TestCase#assertTrue(String, boolean)}
     * based on the results with the passed in message. If an exception occurs, it calls {@link TestCase#fail(String)}.
     * 
     * @param failureMessage Message to display on failure, unless an exception occurs.
     * @param file1 The first file.
     * @param file2 The second file.
     */
    public static void assertTwoFileContentsToSame(final String failureMessage, final File file1, final File file2)
    {
        try
        {
            assertTrue(failureMessage, areTwoFileContentsTheSame(file1, file2));
        }
        catch(final IOException e)
        {
            e.printStackTrace();
            fail("Unexpected exception assert files are equal: " + e.getMessage());
        }
    }

    /**
     * @param file1
     * @param file2
     * @return True if the files are the same, where the files are compared line by line where a trim is performed on
     *         the compared lines (avoiding the window/linux new-line carriage return issue.
     * @throws IOException
     */
    public static boolean areTwoFileContentsTheSame(final File file1, final File file2) throws IOException
    {
        //TODO Rename this method to areToFilesEqualLineByLineIgnoreWhitespace... or something like that.
        boolean filesSame = true;

        //Check number of lines first.
        final LineNumberReader lnr = new LineNumberReader(new FileReader(file1));
        final LineNumberReader lnr2 = new LineNumberReader(new FileReader(file2));
        try
        {
            if(lnr.getLineNumber() != lnr2.getLineNumber())
            {
                filesSame = false;
                return filesSame;
            }
        }
        finally
        {
            StreamTools.closeStream(lnr);
            StreamTools.closeStream(lnr2);
        }

        //Now read line by line and ignore whitespace at ends.
        final BufferedReader reader1 = new BufferedReader(new FileReader(file1));
        final BufferedReader reader2 = new BufferedReader(new FileReader(file2));
        String line1 = reader1.readLine();
        String line2 = reader2.readLine();
        try
        {
            while(line1 != null)
            {
                if(!line1.trim().equals(line2.trim()))
                {
                    filesSame = false;
                    break;
                }
                line1 = reader1.readLine();
                line2 = reader2.readLine();
            }

            if(line2 != null)
            {
                filesSame = false;
            }

            return filesSame;
        }
        finally
        {
            StreamTools.closeStream(reader1);
            StreamTools.closeStream(reader2);
        }
    }

    /**
     * Calls {@link FileUtils#contentEquals(File, File)}. This is an exact comparator. It does not call the other
     * version.
     * 
     * @param fileName1 The first file.
     * @param fileName2 The second file.
     * @return True if the files are the same; false if they are different.
     */
    public static boolean areTwoFileContentsTheSame(final String fileName1, final String fileName2) throws IOException
    {
        final File file1 = new File(fileName1);
        final File file2 = new File(fileName2);
        if(!file1.exists())
        {
            System.out.println("Error comparing files: " + file1.getCanonicalPath() + " does not exist");
            return false;
        }
        if(!file2.exists())
        {
            System.out.println("Error comparing files: " + file2.getCanonicalPath() + " does not exist");
            return false;
        }
        return FileUtils.contentEquals(new File(fileName1), new File(fileName2));
    }

    /**
     * Returns the names of the os specific benchmark file for a given filename for all supported os's. The first
     * returned name is the one for this os.
     * 
     * @param filename the benchmark filename to make os-specific
     * @return the list of os-specific filenames, with the first one as this os's
     */
    public static List<String> makeOsSpecificFilenames(final String filename)
    {
        final List<String> filenameList = newArrayList();
        final List<String> osInsertList = newArrayList();

        // Find this os.
        final String os = System.getProperty("os.name").toLowerCase();
        if(os.contains("windows"))
        {
            osInsertList.add("windows");
            osInsertList.add("linux");
        }
        else
        {
            osInsertList.add("linux");
            osInsertList.add("windows");
        }

        // Add filenames.
        for(final String osInsert: osInsertList)
        {
            final int dotIndex = filename.lastIndexOf(".");
            final String newFilename = filename.substring(0, dotIndex + 1) + osInsert
                + filename.substring(dotIndex, filename.length());
            filenameList.add(new File(newFilename).getPath());
        }

        return filenameList;
    }

    /**
     * Make a map to be used by checkAllFilesUnderDirectoriesForEquality for the given benchmark filename
     * 
     * @param filename
     * @return
     */
    public static Map<String, String> makeBenchmarkOsRenameMap(final String filename)
    {
        final Map<String, String> renameMap = newHashMap();
        final List<String> filenameList = makeOsSpecificFilenames(filename);
        renameMap.put(filenameList.get(0), filename);
        filenameList.remove(0);
        for(final String benchmarkFilename: filenameList)
        {
            renameMap.put(benchmarkFilename, null);
        }
        return renameMap;
    }

    /**
     * To be run by a TestCase, it asserts that all files under outputDir and benchmarkDir are equal. It will call fail
     * if one is not equal. This skips any files whose relative path name contain the string .svn + File.separator.
     * 
     * @param outputDir Directory containing files created by testing.
     * @param benchmarkDir Directory containing benchmarks.
     */
    public static void assertAllFilesUnderDirectoriesEqual(final File outputDir, final File benchmarkDir)
    {
        final Map<String, String> emptyMap = Collections.emptyMap();
        assertAllFilesUnderDirectoriesEqual(outputDir, benchmarkDir, emptyMap);
    }

    /**
     * To be run by a {@link TestCase}, it asserts that all files under outputDir and benchmarkDir are equal. It will
     * call fail if one is not equal. This skips any files whose relative path name contain the string .svn +
     * File.separator.
     * 
     * @param outputDir Directory containing files created by testing
     * @param benchmarkDir Directory containing benchmarks
     * @param benchToOutRenameMap check the given benchmark files against the specified output files instead of the
     *            default ones
     */
    @SuppressWarnings("unchecked")
    public static void assertAllFilesUnderDirectoriesEqual(final File outputDir,
                                                           final File benchmarkDir,
                                                           final Map<String, String> benchToOutRenameMap)
    {
        final List<File> benchmarkFileList = newLinkedList();
        final List<File> outputFileList = newLinkedList();
        final Map<File, File> fileComparisonMap = newHashMap();

        // Grab all files in the two directories.
        benchmarkFileList.addAll(FileUtils.listFiles(benchmarkDir, null, true));
        outputFileList.addAll(FileUtils.listFiles(outputDir, null, true));

        // Strip out .svn files.
        final List<File> toRemove = newArrayList();
        for(final File file: benchmarkFileList)
        {
            if(file.getAbsolutePath().contains(".svn" + File.separator))
            {
                toRemove.add(file);
            }
        }
        benchmarkFileList.removeAll(toRemove);
        toRemove.clear();
        for(final File file: outputFileList)
        {
            if(file.getAbsolutePath().contains(".svn" + File.separator))
            {
                toRemove.add(file);
            }
        }
        outputFileList.removeAll(toRemove);
        toRemove.clear();

        // Pair up benchmark files with their corresponding output files.
        for(final File benchmarkFile: benchmarkFileList)
        {
            final String benchmarkRelativeFilename = FileTools.determineFileNameRelativeToBaseDirectory(benchmarkDir,
                                                                                                        benchmarkFile);

            // Match up the benchmark file to the output file.
            String outputRelativeFilename;
            if(benchToOutRenameMap.containsKey(benchmarkRelativeFilename))
            {
                outputRelativeFilename = benchToOutRenameMap.get(benchmarkRelativeFilename);

                // If null, it means we ignore this benchmark file.
                if(outputRelativeFilename == null)
                {
                    toRemove.add(benchmarkFile);
                    continue;
                }
            }
            else
            {
                outputRelativeFilename = benchmarkRelativeFilename;
            }
            final File outputFile = new File(outputDir + File.separator + outputRelativeFilename);

            // Add to map to be checked later.
            fileComparisonMap.put(benchmarkFile, outputFile);
        }

        // Remove ignored benchmark files.
        benchmarkFileList.removeAll(toRemove);

        // Remove matched files from lists.
        for(final Map.Entry<File, File> entry: fileComparisonMap.entrySet())
        {
            benchmarkFileList.remove(entry.getKey());
            outputFileList.remove(entry.getValue());
        }

        // Check for unmatched files.
        if(benchmarkFileList.size() + outputFileList.size() > 0)
        {
            String msg = "";
            if(benchmarkFileList.size() > 0)
            {
                msg = "Benchmark files: ["
                    + FileTools.determineFileNameRelativeToBaseDirectory(benchmarkDir, benchmarkFileList.get(0));
                benchmarkFileList.remove(0);
                for(final File benchmarkFile: benchmarkFileList)
                {
                    msg += ", " + FileTools.determineFileNameRelativeToBaseDirectory(benchmarkDir, benchmarkFile);
                }
                msg += "] were not found in created output dir " + outputDir + ".";
            }
            if(outputFileList.size() > 0)
            {
                if(msg != "")
                {
                    msg += "\n";
                }
                msg = "Output files: ["
                    + FileTools.determineFileNameRelativeToBaseDirectory(outputDir, outputFileList.get(0));
                outputFileList.remove(0);
                for(final File outputFile: outputFileList)
                {
                    msg += ", " + FileTools.determineFileNameRelativeToBaseDirectory(outputDir, outputFile);
                }
                msg += "] were not found in benchmark dir " + benchmarkDir + ".";
            }
            fail(msg);
        }

        // Compare matched files.
        try
        {
            for(final Map.Entry<File, File> entry: fileComparisonMap.entrySet())
            {
                assertTrue("Files do not equal: " + entry.getKey().getAbsolutePath() + ", "
                    + entry.getValue().getAbsolutePath(),
                           FileComparisonUtilities.areTwoFileContentsTheSame(entry.getKey(), entry.getValue()));
            }
        }
        catch(final IOException e)
        {
            e.printStackTrace();
            fail("Unexpected exception comparing output under " + outputDir.getAbsolutePath()
                + " with benchmarks under " + benchmarkDir.getAbsolutePath() + ".");
        }
    }

    public static void assertTwoPITimeSeriesFilesTheSameOrderMatters(final File outputFile,
                                                                     final File benchmarkFile) throws IOException,
                                                                                               InterruptedException
    {
        final TimeSeriesArrays outputTS = TimeSeriesArraysTools.readFromFile(outputFile);
        final TimeSeriesArrays benchTS = TimeSeriesArraysTools.readFromFile(benchmarkFile);
        if(outputTS.size() != benchTS.size())
        {
            fail("The number of output time series, " + outputTS.size() + ", does not match the benchmark, "
                + benchTS.size());
        }
        for(int i = 0; i < outputTS.size(); i++)
        {
            try
            {
                TimeSeriesArrayTools.checkTimeSeriesEqual(outputTS.get(i), benchTS.get(i));
            }
            catch(final Exception e)
            {
                e.printStackTrace();
                fail("Time series " + i + " does not match benchmark (output = "
                    + TimeSeriesArrayTools.createHeaderString(outputTS.get(i)) + "; bench = "
                    + TimeSeriesArrayTools.createHeaderString(benchTS.get(i)) + "): " + e.getMessage());
            }
        }
    }

    /**
     * Calls {@link #assertImageFileSimilarToBenchmark(File, File, int, int, int, boolean, boolean)} with the number of
     * columns being 32 and number of row regions being 24.
     */
    public static void assertImageFileSimilarToBenchmark(final File outputFile,
                                                         final File benchmarkFile,
                                                         final int sensitivity,
                                                         final boolean generateDissimilarityFileIfNotSimilar,
                                                         final boolean debugMode) throws MalformedURLException
    {
        assertImageFileSimilarToBenchmark(outputFile,
                                          benchmarkFile,
                                          32,
                                          24,
                                          sensitivity,
                                          generateDissimilarityFileIfNotSimilar,
                                          debugMode);
    }

    /**
     * Uses {@link ImageComparator}. There are 32 regions along the x-axis, 24 along the y-axis, and a stabilization
     * factor of 30. With those values, typical differences between linux and windows should peak around 5.
     * 
     * @param numberOfColumns The number of region to use along the x-dimension.
     * @param numberOfRows The number of regions to use along the y-dimension.
     * @param sensitivity An integer factor that serves as a measure of sensitivity for the test; a threshold above
     *            which regions of the image are considered different. Typically, with the others settings of
     *            {@link ImageComparator} left as is, differences between linux and windows will yield region values up
     *            to 5. True differences will be larger. A value of 8 may be reasonable, but it will require
     *            test-by-test experimentation to select an appropriate value for each test.
     * @param generateDissimilarityFileIfNotSimilar If true, then, if the file does not match, a dissimilarity file will
     *            be generated next to the output file (same name, prefixed by "dissimilarity.".)
     * @param debugMode Set to true to receive full output; this should only be done in testing, because the output goes
     *            to stdout, not through a logging tool.
     * @throws MalformedURLException If either of the two files yields a bad URL after calling {@link File#toURI()} and
     *             {@link URI#toURL()}.
     */
    public static void assertImageFileSimilarToBenchmark(final File outputFile,
                                                         final File benchmarkFile,
                                                         final int numberOfColumns,
                                                         final int numberOfRows,
                                                         final int sensitivity,
                                                         final boolean generateDissimilarityFileIfNotSimilar,
                                                         final boolean debugMode) throws MalformedURLException
    {
        // Create a compare object specifying the 2 images for comparison.
        final ImageComparator ic = new ImageComparator(benchmarkFile, outputFile);
        // Set the comparison parameters.
        // (num vertical regions, num horizontal regions, sensitivity, stabilizer)
        ic.setParameters(numberOfColumns, numberOfRows, sensitivity, 30);

        // Display some indication of the differences in the image.
        if(debugMode)
        {
            ic.setDebugMode(2);
        }
        else
        {
            ic.setDebugMode(0);
        }

        // Compare.
        ic.compare();

        // If its not a match then write a file to show changed regions.
        if(!ic.match())
        {
            if(generateDissimilarityFileIfNotSimilar)
            {
                try
                {
                    final File dissimilarityFile = FileTools.newFile(outputFile.getParent(),
                                                                     "dissimilarities." + outputFile.getName());
                    ImageIO.write(ic.getChangeIndicator(), "png", dissimilarityFile);
                    System.out.println("Differences found in image files for test; see dissimilarity file, "
                        + dissimilarityFile.getAbsolutePath());
                }
                catch(final IOException e)
                {
                    e.printStackTrace();
                    System.err.println("Unable to create image file showing different regions: " + e.getMessage());
                }
            }
            fail("Output file " + outputFile.getAbsolutePath() + " does not match benchmark file "
                + benchmarkFile.getAbsolutePath());
        }
    }

    /**
     * This must only be called during unit testing! It calls the {@link TestCase#fail(String)} method. Looks for an
     * EXACT match pixel-by-pixel.
     */
    @SuppressWarnings("null") //The use of outputImage and benchmarkImage below is flagged for possibly being null.  That's because Java doesn't know how fail will work.
    public static void assertImageFileIdenticalToBenchmark(final File outputFile, final File benchmarkFile)
    {
        BufferedImage outputImage = null;
        BufferedImage benchmarkImage = null;
        try
        {
            outputImage = SwingTools.loadImage(outputFile.toURI().toURL());
            if(outputImage == null)
            {
                fail("Unable to load output image file; the acquired image is null.");
            }
        }
        catch(final Throwable t)
        {
            fail("Unable to load output image: " + t.getMessage());
        }
        try
        {
            benchmarkImage = SwingTools.loadImage(benchmarkFile.toURI().toURL());
            if(benchmarkImage == null)
            {
                fail("Unable to load output image file; the acquired image is null.");
            }
        }
        catch(final Throwable t)
        {
            fail("Unable to load benchmark image: " + t.getMessage());
        }

        //Compare
        if((outputImage.getHeight() != benchmarkImage.getHeight())
            || (outputImage.getWidth() != benchmarkImage.getWidth()))
        {
            fail("Image dimensions do not match; output dimensions (" + outputImage.getHeight() + ", "
                + outputImage.getWidth() + ") while benchmark is (" + benchmarkImage.getHeight() + ", "
                + benchmarkImage.getWidth() + ").");
        }
        for(int i = 0; i < outputImage.getWidth(); i++)
        {
            for(int j = 0; j < outputImage.getHeight(); j++)
            {
                if(outputImage.getRGB(i, j) != benchmarkImage.getRGB(i, j))
                {
                    fail("Output image differs from benchmark at pixel (" + i + ", " + j + ").");
                }
            }
        }
    }
}
