package ohd.hseb.hefs.utils.junit;

import java.io.File;
import java.util.List;

import junit.framework.TestCase;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

/**
 * Utilities for comparing {@link TimeSeriesArray} and {@link TimeSeriesArrays} instances. It makes use of
 * {@link TimeSeriesArrayTools#checkTimeSeriesEqual(TimeSeriesArray, TimeSeriesArray, double)}.
 * 
 * @author Hank.Herr
 */
public class TimeSeriesComparisonUtilities extends TestCase
{

    /**
     * Checks for the equality of two time series using
     * {@link TimeSeriesArrayTools#checkTimeSeriesEqual(TimeSeriesArray, TimeSeriesArray, double)}. The exception is
     * caught and its message is added to the provided message after a ':' is added.
     * 
     * @param message Message to print.
     * @param ts1 First time series.
     * @param ts2 Second time series.
     * @param tolerance Tolerance allowed.
     */
    public static void assertTimeSeriesEqual(final String message,
                                             final TimeSeriesArray ts1,
                                             final TimeSeriesArray ts2,
                                             final double tolerance)
    {
        try
        {
            TimeSeriesArrayTools.checkTimeSeriesEqual(ts1, ts2, tolerance);
        }
        catch(final Exception e)
        {
            fail(message + ": " + e.getMessage());
        }
    }

    /**
     * Checks for the equality of two time series using
     * {@link TimeSeriesArraysTools#checkForEquality(List, List, double)}. The exception is caught and its message is
     * added to the provided message after a ':' is added.
     * 
     * @param message Message to print.
     * @param ts1 First time series.
     * @param ts2 Second time series.
     * @param tolerance Tolerance allowed.
     */
    public static void assertTimeSeriesEqual(final String message,
                                             final List<TimeSeriesArray> ts1,
                                             final List<TimeSeriesArray> ts2,
                                             final double tolerance)
    {
        try
        {
            checkForEquality(ts1, ts2, tolerance);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            fail(message + ": " + e.getMessage());
        }
    }

    /**
     * This will call fail if the time series in the two files do not match within the given tolerance. It calls
     * {@link #assertTimeSeriesEqual(String, TimeSeriesArrays, TimeSeriesArrays, double)} after reading the files. It
     * will also fail if either file fails to read.
     * 
     * @param message Message to output if unequal.
     * @param piTSFile1 {@link File} specifying the first set of time series.
     * @param piTSFile2 {@link File} specifying the second set of time series.
     * @param tolerance Tolerance allowed for comparison.
     */
    public static void assertTimeSeriesEqual(final String message,
                                             final File piTSFile1,
                                             final File piTSFile2,
                                             final double tolerance)
    {
        try
        {
            final TimeSeriesArrays ts1 = TimeSeriesArraysTools.readFromFile(piTSFile1);
            final TimeSeriesArrays ts2 = TimeSeriesArraysTools.readFromFile(piTSFile2);
            TimeSeriesComparisonUtilities.assertTimeSeriesEqual("Time series out does not match time series in.",
                                                                ts1,
                                                                ts2,
                                                                tolerance);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            fail("Exception occurred reading in one of the two time series to compare: " + e.getMessage());
        }
    }

    /**
     * This will call fail if the time series in the file do not match the provided teim series within the given
     * tolerance. It calls {@link #assertTimeSeriesEqual(String, TimeSeriesArrays, TimeSeriesArrays, double)} after
     * reading the file. It will also fail if the file fails to read.
     * 
     * @param message Message to output if unequal.
     * @param ts Time series to compare against contents of a file.
     * @param benchmark {@link File} specifying a benchmark set of time series.
     * @param tolerance Tolerance allowed for comparison.
     */
    public static void assertTimeSeriesEqual(final String message,
                                             final TimeSeriesArrays ts,
                                             final File benchmark,
                                             final double tolerance)
    {
        try
        {
            final TimeSeriesArrays benchmarkTS = TimeSeriesArraysTools.readFromFile(benchmark);
            TimeSeriesComparisonUtilities.assertTimeSeriesEqual(message, ts, benchmarkTS, tolerance);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            fail("Exception occurred reading in one of the two time series to compare: " + e.getMessage());
        }
    }

    /**
     * This will call fail if the time series in the file do not match the provided time series within the given
     * tolerance. It calls {@link #assertTimeSeriesEqual(String, TimeSeriesArrays, TimeSeriesArrays, double)} after
     * reading the file. It will also fail if the file fails to read.
     * 
     * @param message Message to output if unequal.
     * @param ts Time series to compare against contents of a file.
     * @param benchmark {@link File} specifying a benchmark set of time series.
     * @param tolerance Tolerance allowed for comparison.
     */
    public static void assertTimeSeriesEqual(final String message,
                                             final List<TimeSeriesArray> ts,
                                             final File benchmark,
                                             final double tolerance)
    {
        try
        {
            final TimeSeriesArrays benchmarkTS = TimeSeriesArraysTools.readFromFile(benchmark);
            TimeSeriesComparisonUtilities.assertTimeSeriesEqual("Time series out does not match time series in.",
                                                                ts,
                                                                TimeSeriesArraysTools.convertTimeSeriesArraysToList(benchmarkTS),
                                                                tolerance);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            fail("Exception occurred reading in one of the two time series to compare: " + e.getMessage());
        }
    }

    /**
     * Checks for the equality of two time series using
     * {@link TimeSeriesArraysTools#checkForEquality(List, List, double)}. The exception is caught and its message is
     * added to the provided message after a ':' is added.
     * 
     * @param message Message to print.
     * @param ts1 First time series.
     * @param ts2 Second time series.
     * @param tolerance Tolerance allowed.
     */
    public static void assertTimeSeriesEqual(final String message,
                                             final TimeSeriesArrays ts1,
                                             final TimeSeriesArrays ts2,
                                             final double tolerance)
    {
        try
        {
            checkForEquality(ts1, ts2, tolerance);
        }
        catch(final Exception e)
        {
            fail(message + ": " + e.getMessage());
        }
    }

    /**
     * Basic method checks for equality of two lists of time series, calling the {@link TimeSeriesArraysTools} method.
     * It is recommended you call {@link #assertTimeSeriesEqual(String, List, List, double)}.
     * 
     * @param ts1
     * @param ts2
     * @param tolerance
     * @throws Exception
     */
    public static void checkForEquality(final List<TimeSeriesArray> ts1,
                                        final List<TimeSeriesArray> ts2,
                                        final double tolerance) throws Exception
    {
        if(ts1.size() != ts2.size())
        {
            throw new Exception("The number of time series in two lists differ.");
        }
        for(int i = 0; i < ts1.size(); i++)
        {
            TimeSeriesArrayTools.checkTimeSeriesEqual(ts1.get(i), ts2.get(i), tolerance);
        }
    }

    /**
     * Basic method checks for equality of two lists of time series, calling the {@link TimeSeriesArraysTools} method.
     * It is recommended you call {@link #assertTimeSeriesEqual(String, TimeSeriesArrays, TimeSeriesArrays, double)}.
     * 
     * @param ts1
     * @param ts2
     * @param tolerance
     * @throws Exception
     */
    public static void checkForEquality(final TimeSeriesArrays ts1, final TimeSeriesArrays ts2, final double tolerance) throws Exception
    {
        if(ts1.size() != ts2.size())
        {
            throw new Exception("The number of time series in two lists differ.");
        }
        for(int i = 0; i < ts1.size(); i++)
        {
            try
            {
                TimeSeriesArrayTools.checkTimeSeriesEqual(ts1.get(i), ts2.get(i), tolerance);
            }
            catch(final Exception e)
            {
                throw new Exception("Match failure for time series index " + i + ": " + e.getMessage());
            }
        }
    }

}
