package ohd.hseb.hefs.utils.log4j;

import java.io.File;
import java.io.IOException;

import org.apache.commons.lang.ArrayUtils;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.LoggerConfig;
import org.apache.logging.log4j.core.layout.PatternLayout;

import nl.wldelft.util.logging.LoggerContextWrapper;
import ohd.hseb.util.io.ExceptionParser;
import ohd.hseb.util.misc.SegmentedLine;


public abstract class LoggingTools
{

    /**
     * Creates a log-file appending logger. Assumes the additivity is false.
     * 
     * @param logFile The file in which to store messages.
     * @param packageName The name of the package to which to tie this appender.
     * @param level The level of messaging to put in the file.
     * @return The name of the appender, which can be used to identify it later.
     * @throws IOException If a problem occurs opening the file.
     */
    public static String initializeLogFileLogger(final File logFile, final String packageName, final Level level) throws IOException
    {
        return initializeLogFileLogger(logFile, packageName, level, false);
    }


    /**
     * Assumes pattern "%d{HH:mm:ss.SSS} %level %msg%n" for logging, calling {@link #initializeLogFileLogger(File, String, Level, boolean, String)}.
     * 
     * @param logFile The file in which to store messages.
     * @param packageName The name of the package to which to tie this appender.
     * @param level The level of messaging to put in the file.
     * @param additivity Message additivity to use. When true, some extra messaging appears to be generated to stdout
     *            and I'm not sure why.
     * @return The name of the appender, which can be used to identify it later.
     * @throws IOException If a problem occurs opening the file.
     */
    public static String initializeLogFileLogger(final File logFile,
                                                 final String packageName,
                                                 final Level level,
                                                 final boolean additivity) throws IOException
    {
        return initializeLogFileLogger(logFile, packageName, level, additivity, "%d{HH:mm:ss.SSS} %level %msg%n");
    }


    /**
     * Creates a log-file appending logger.
     * 
     * @param logFile The file in which to store messages.
     * @param packageName The name of the package to which to tie this appender.
     * @param level The level of messaging to put in the file.
     * @param additivity Message additivity to use. When true, some extra messaging appears to be generated to stdout
     *            and I'm not sure why.
     * @param logPattern The pattern for log messages in the log file.
     * @return The name of the appender, which can be used to identify it later.
     * @throws IOException If a problem occurs opening the file.
     */
    public static String initializeLogFileLogger(final File logFile,
                                                 final String packageName,
                                                 final Level level,
                                                 final boolean additivity, 
                                                 final String logPattern) throws IOException
    {
        //Get the configuration and logger context.
        Configuration config;
        LoggerContext ctx;
        final Object contextReturn = LogManager.getContext(false);
        if (contextReturn instanceof LoggerContextWrapper)
        {
            config = ((LoggerContext)((LoggerContextWrapper)contextReturn).getContext()).getConfiguration();
            ctx = (LoggerContext)((LoggerContextWrapper)contextReturn).getContext();
        }
        else if (contextReturn instanceof LoggerContext)
        {
            config = ((LoggerContext)contextReturn).getConfiguration();
            ctx = (LoggerContext)contextReturn;
        }
        else
        {
            throw new IOException("Log4j tool not initialized for loggin!!!");
        }
        
        //Create a new logger config.  This is equivalent to defining a Logger XML element to be placed within the 
        //Loggers element of a log4j2 XML file.
        final LoggerConfig loggerConfig = new LoggerConfig(packageName, Level.DEBUG, additivity);
        
        //Construct an OHDFileAppender with a specific pattern and start it.
        final PatternLayout layout = PatternLayout.newBuilder()        
                .withPattern(logPattern)
                .build();    
        final OHDFileAppender appender = new OHDFileAppender("OHDLogFileAppender_" + packageName, config, layout, logFile.getAbsolutePath());
        appender.start();
            
        //Add the appender to the logger and the logger to the configuration, again as specified in that XML file.
        loggerConfig.addAppender(appender, level, null);
        config.addLogger(packageName, loggerConfig);
        
        //Update the loggers. 
        ctx.updateLoggers();
        
        //Return the name of the logger, which I think will match the package name.  
        return appender.getName();
        
    }
    
    public static void removeLoggingAppender(final String appenderName, final String packageName) throws IOException
    {       	
//        LoggerContext ctx = (LoggerContext) LogManager.getContext(false);
//        Configuration config = ctx.getConfiguration();
        
    	Configuration config = null;
    	LoggerContext ctx = null;
    	final Object contextReturn = LogManager.getContext(false);
    	if (contextReturn instanceof LoggerContextWrapper)
    	{
    	    config = ((LoggerContext)((LoggerContextWrapper)contextReturn).getContext()).getConfiguration();
    	    ctx = (LoggerContext)((LoggerContextWrapper)contextReturn).getContext();
    	}
    	else if (contextReturn instanceof LoggerContext)
    	{
    	    config = ((LoggerContext)contextReturn).getConfiguration();
    	    ctx = (LoggerContext)contextReturn;
    	}
    	else
    	{
    		throw new IOException("Log4j tool not initialized for loggin!!!");
    	}
        
        final LoggerConfig loggerConfig = config.getLoggerConfig(packageName); 
        final Appender appender = loggerConfig.getAppenders().get(appenderName);
        if (appender != null)
        {
            appender.stop();
        }
        loggerConfig.removeAppender(appenderName);
        config.removeLogger(packageName);
        ctx.updateLoggers();

    }

    /**
     * Dumps the stack trace for the given {@link Throwable} as debug level messages in the provided {@link Logger}.
     */
    public static void outputStackTraceAsDebug(final Logger log, final Throwable t)
    {
        LoggingTools.outputDebugLines(log, ExceptionParser.multiLineStackTrace(t));
    }

    /**
     * Dumps the stack trace for the given {@link Throwable} as debug level messages in the provided {@link Logger}.
     * Only lineCount many lines are output.
     */
    public static void outputStackTraceAsDebug(final Logger log, final Throwable t, final int lineCount)
    {
        final Object[] outputItems = ArrayUtils.subarray(t.getStackTrace(), 0, lineCount);
        LoggingTools.outputDebugLines(log, outputItems);
        if(outputItems.length < t.getStackTrace().length)
        {
            outputDebugLines(log, "... [additional lines not shown]");
        }
    }

    /**
     * Dumps the stack trace for the given {@link Throwable} as debug level messages in the provided {@link Logger}.
     */
    public static void outputStackTraceAsInfo(final Logger log, final Throwable t)
    {
        LoggingTools.outputInfoLines(log, ExceptionParser.multiLineStackTrace(t));
    }

    /**
     * Dumps the stack trace for the given {@link Throwable} as debug level messages in the provided {@link Logger}.
     * Only lineCount many lines are output.
     */
    public static void outputStackTraceAsInfo(final Logger log, final Throwable t, final int lineCount)
    {
        final Object[] outputItems = ArrayUtils.subarray(t.getStackTrace(), 0, lineCount);
        LoggingTools.outputInfoLines(log, outputItems);
        if(outputItems.length < t.getStackTrace().length)
        {
            outputInfoLines(log, "... [additional lines not shown]");
        }
    }

    /**
     * Calls {@link #outputDebugLines(Logger, Object[])}.
     */
    public static void outputDebugLines(final Logger log, final String fullString)
    {
        final SegmentedLine segLine = new SegmentedLine(fullString, "\n", SegmentedLine.MODE_NO_EMPTY_SEGS);
        outputDebugLines(log, segLine.getSegments().toArray());
    }

    /**
     * Calls {@link #outputInfoLines(Logger, Object[])}.
     */
    public static void outputInfoLines(final Logger log, final String fullString)
    {
        final SegmentedLine segLine = new SegmentedLine(fullString, "\n", SegmentedLine.MODE_NO_EMPTY_SEGS);
        outputInfoLines(log, segLine.getSegments().toArray());
    }

    /**
     * Calls {@link #outputWarnLines(Logger, Object[])}.
     */
    public static void outputWarnLines(final Logger log, final String fullString)
    {
        final SegmentedLine segLine = new SegmentedLine(fullString, "\n", SegmentedLine.MODE_NO_EMPTY_SEGS);
        outputWarnLines(log, segLine.getSegments().toArray());
    }

    /**
     * Calls {@link #outputErrorLines(Logger, Object[])}.
     */
    public static void outputErrorLines(final Logger log, final String fullString)
    {
        final SegmentedLine segLine = new SegmentedLine(fullString, "\n", SegmentedLine.MODE_NO_EMPTY_SEGS);
        outputErrorLines(log, segLine.getSegments().toArray());
    }

    /**
     * Calls {@link #outputFatalLines(Logger, Object[])}.
     */
    public static void outputFatalLines(final Logger log, final String fullString)
    {
        final SegmentedLine segLine = new SegmentedLine(fullString, "\n", SegmentedLine.MODE_NO_EMPTY_SEGS);
        outputFatalLines(log, segLine.getSegments().toArray());
    }

    /**
     * Output multiple lines calling {@link Logger#debug(Object)}.
     * 
     * @param log The {@link Logger} to use.
     * @param lines The lines to output.
     */
    public static void outputDebugLines(final Logger log, final Object[] lines)
    {
        for(final Object line: lines)
        {
            log.debug(line.toString().replaceAll("\\s+$", ""));
        }
    }

    /**
     * Output multiple lines calling {@link Logger#info(Object)}.
     * 
     * @param log The {@link Logger} to use.
     * @param lines The lines to output.
     */
    public static void outputInfoLines(final Logger log, final Object[] lines)
    {
        for(final Object line: lines)
        {
            log.info(line.toString());
        }
    }

    /**
     * Output multiple lines calling {@link Logger#warn(Object)}.
     * 
     * @param log The {@link Logger} to use.
     * @param lines The lines to output.
     */
    public static void outputWarnLines(final Logger log, final Object[] lines)
    {
        for(final Object line: lines)
        {
            log.warn(line.toString());
        }
    }

    /**
     * Output multiple lines calling {@link Logger#error(Object)}.
     * 
     * @param log The {@link Logger} to use.
     * @param lines The lines to output.
     */
    public static void outputErrorLines(final Logger log, final Object[] lines)
    {
        for(final Object line: lines)
        {
            log.error(line.toString());
        }
    }

    /**
     * Output multiple lines calling {@link Logger#fatal(Object)}.
     * 
     * @param log The {@link Logger} to use.
     * @param lines The lines to output.
     */
    public static void outputFatalLines(final Logger log, final Object[] lines)
    {
        for(final Object line: lines)
        {
            log.fatal(line.toString());
        }
    }
}
