package ohd.hseb.hefs.utils.notify;

import com.google.common.eventbus.EventBus;

/**
 * Default implementation of {@link Notifier}. Note that if the notifications are stopped (via {@link #stopNotifying()}
 * then started via {@link #startNotifying()}, one {@link CompositeModificationNotice} notice specifying all held events
 * is fired. The {@link #_bus} is not initialized until something is registered to listen to it via
 * {@link #register(Object)} or {@link #registerAll(Iterable)}.
 * 
 * @author hank.herr
 */
public class NotifierBase implements Notifier
{
    //Initialized when something is registered to it.
    private EventBus _bus = null;

    private NotifyState _state = NotifyState.SEND;
    private CompositeModificationNotice _composite = new CompositeModificationNotice(this);

    protected void post(final Object notice)
    {
        if(_bus != null)
        {
            switch(_state)
            {
                case SEND:
                    if(notice instanceof Notice)
                    {
                        ((Notice)notice).addToNotifierHistory(this);
                    }
                    _bus.post(notice);
                    break;
                case HOLD:
                    _composite.addEvent(notice);
                    break;
                case IGNORE:
                    break;
            }
        }
    }

    public void registerAll(final Iterable<? extends Object> objects)
    {
        for(final Object object: objects)
        {
            register(object);
        }
    }

    @Override
    public void register(final Object object)
    {
        if(_bus == null)
        {
            _bus = new EventBus();
        }
        //Make sure it does not register the same object multiple times
        try
        {
            _bus.unregister(object);
        }
        catch(final Exception e)
        {
            //Illegal argument exception is caught here an ignored if the object is not registered!
        }

        _bus.register(object);
    }

    @Override
    public void unregister(final Object object)
    {
        if(_bus != null)
        {
            _bus.unregister(object);
        }
    }

    @Override
    public void startNotifying()
    {
        if(_state == NotifyState.IGNORE)
        {
            _state = NotifyState.SEND;
        }
    }

    @Override
    public void stopNotifying()
    {
        _state = NotifyState.IGNORE;
    }

    @Override
    public void holdNotifications()
    {
        _state = NotifyState.HOLD;
    }

    @Override
    public void releaseNotifications()
    {
        _state = NotifyState.SEND;
        post(_composite);
        _composite = new CompositeModificationNotice(this); //Clear it
    }
}
