package ohd.hseb.hefs.utils.notify.collect;

import java.util.Collection;
import java.util.Iterator;

import ohd.hseb.hefs.utils.Decorator;
import ohd.hseb.hefs.utils.notify.NoticeForwarder;
import ohd.hseb.hefs.utils.notify.Notifier;

import com.google.common.collect.ForwardingIterator;

/**
 * Wraps a {@link Collection} to make it generate notices whenever it is modified.<br/>
 * As a decorator, this also forwards any notices received from its delegate.
 * 
 * @see Notifier
 * @author alexander.garbarino
 */
@SuppressWarnings({"rawtypes", "unchecked"})
public class NotifyingCollectionWrapper<E, C extends Collection<E>> extends NoticeForwarder implements Decorator<C>,
NotifyingCollection<E>
{
    protected final C _delegate;

    private boolean _notifyEnabled = true;

    protected NotifyingCollectionWrapper(final C delegate)
    {
        _delegate = delegate;
        Notifier.Tools.tryRegister(_delegate, this);
    }

    public void setNotifyEnabled(final boolean b)
    {
        _notifyEnabled = b;
    }

    /**
     * Ensures that the passed collection implements {@link NotifyingCollection}.
     */
    public static <E> NotifyingCollection<E> wrap(final Collection<E> collection)
    {
        if(collection instanceof NotifyingCollection<?>)
        {
            return (NotifyingCollection<E>)collection;
        }
        else
        {
            return new NotifyingCollectionWrapper<E, Collection<E>>(collection);
        }
    }

    @Override
    public C delegate()
    {
        return _delegate;
    }

    @Override
    public boolean add(final E e)
    {
        return add(e, this);
    }

    @Override
    public boolean add(final E e, final Object source)
    {
        final boolean result = _delegate.add(e);
        if(result && _notifyEnabled)
        {
            post(new CollectionAddNotice(source, this, e));
        }
        return result;
    }

    @Override
    public boolean remove(final Object o)
    {
        return remove(o, this);
    }

    @Override
    public boolean remove(final Object o, final Object source)
    {
        final boolean result = _delegate.remove(o);
        if(result && _notifyEnabled)
        {
            post(new CollectionRemoveNotice(source, this, o));
        }
        return result;
    }

    @Override
    public boolean addAll(final Collection<? extends E> c)
    {
        return addAll(c, this);
    }

    @Override
    public boolean addAll(final Collection<? extends E> c, final Object source)
    {
        final boolean result = _delegate.addAll(c);
        if(result && _notifyEnabled)
        {
            post(new CollectionAddAllNotice(source, this, c));
        }
        return result;
    }

    @Override
    public boolean removeAll(final Collection<?> c)
    {
        return removeAll(c, this);
    }

    @Override
    public boolean removeAll(final Collection<?> c, final Object source)
    {
        final boolean result = _delegate.removeAll(c);
        if(result && _notifyEnabled)
        {
            post(new CollectionRemoveAllNotice(source, this, c));
        }
        return result;
    }

    @Override
    public boolean retainAll(final Collection<?> c)
    {
        return retainAll(c, this);
    }

    @Override
    public boolean retainAll(final Collection<?> c, final Object source)
    {
        final boolean result = _delegate.retainAll(c);
        if(result && _notifyEnabled)
        {
            post(new CollectionRetainAllNotice(source, this, c));
        }
        return result;
    }

    @Override
    public void clear()
    {
        clear(this);
    }

    @Override
    public void clear(final Object source)
    {
        _delegate.clear();
        if(_notifyEnabled)
        {
            post(new CollectionClearNotice(source, this));
        }
    }

    @Override
    public NotifyingIterator<E> iterator()
    {
        return new NotifyingCollectionWrapperIterator();
    }

    // Straight-forward delegate methods.

    @Override
    public int size()
    {
        return _delegate.size();
    }

    @Override
    public boolean isEmpty()
    {
        return _delegate.isEmpty();
    }

    @Override
    public boolean contains(final Object o)
    {
        return _delegate.contains(o);
    }

    @Override
    public Object[] toArray()
    {
        return _delegate.toArray();
    }

    @Override
    public <T> T[] toArray(final T[] a)
    {
        return _delegate.toArray(a);
    }

    @Override
    public boolean containsAll(final Collection<?> c)
    {
        return _delegate.containsAll(c);
    }

    private class NotifyingCollectionWrapperIterator extends ForwardingIterator<E> implements NotifyingIterator<E>
    {
        private final Iterator<E> _delegateIterator = _delegate.iterator();
        private E _lastReturned;

        @Override
        protected Iterator<E> delegate()
        {
            return _delegateIterator;
        }

        @Override
        public E next()
        {
            _lastReturned = super.next();
            return _lastReturned;
        }

        @Override
        public void remove()
        {
            remove(NotifyingCollectionWrapper.this);
        }

        @Override
        public void remove(final Object source)
        {
            super.remove();
            post(new CollectionRemoveNotice(source, NotifyingCollectionWrapper.this, _lastReturned));
        }
    }
}
