package ohd.hseb.hefs.utils.notify.collect;

import java.util.Collection;
import java.util.List;
import java.util.ListIterator;

import ohd.hseb.hefs.utils.notify.Notifier;

import com.google.common.collect.ForwardingListIterator;

/**
 * Wraps a {@link List} to make it generate notices whenever it is modified.<br/>
 * As a decorator, this also forwards any notices received from its delegate.
 * 
 * @see Notifier
 * @author alexander.garbarino
 */
public class NotifyingListWrapper<E, C extends List<E>> extends NotifyingCollectionWrapper<E, C> implements
NotifyingList<E>
{
    protected NotifyingListWrapper(final C delegate)
    {
        super(delegate);
    }

    /**
     * Ensures that {@code list} implements {@link NotifyingList}.
     */
    public static <E> NotifyingList<E> wrap(final List<E> list)
    {
        if(list instanceof NotifyingList<?>)
        {
            return (NotifyingList<E>)list;
        }
        else
        {
            return new NotifyingListWrapper<E, List<E>>(list);
        }
    }

    @Override
    public boolean addAll(final int index, final Collection<? extends E> c)
    {
        return addAll(index, c, this);
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    @Override
    public boolean addAll(final int index, final Collection<? extends E> c, final Object source)
    {
        final boolean result = delegate().addAll(index, c);
        if(result)
        {
            post(new ListAddAllNotice(source, this, index, c));
        }
        return result;
    }

    @Override
    public E set(final int index, final E element)
    {
        return set(index, element, this);
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    @Override
    public E set(final int index, final E element, final Object source)
    {
        final E result = delegate().set(index, element);
        post(new ListSetNotice(source, this, index, element, result));
        return result;
    }

    @Override
    public void add(final int index, final E element)
    {
        add(index, element, this);
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    @Override
    public void add(final int index, final E element, final Object source)
    {
        delegate().add(index, element);
        post(new ListAddNotice(source, this, index, element));
    }

    @Override
    public E remove(final int index)
    {
        return remove(index, this);
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    @Override
    public E remove(final int index, final Object source)
    {
        final E element = delegate().remove(index);
        post(new ListRemoveNotice(source, this, index, element));
        return element;
    }

    @Override
    public NotifyingList<E> subList(final int fromIndex, final int toIndex)
    {
        return wrap(delegate().subList(fromIndex, toIndex));
    }

    @Override
    public E get(final int index)
    {
        return delegate().get(index);
    }

    @Override
    public int indexOf(final Object o)
    {
        return delegate().indexOf(o);
    }

    @Override
    public int lastIndexOf(final Object o)
    {
        return delegate().lastIndexOf(o);
    }

    @Override
    public NotifyingListIterator<E> listIterator()
    {
        return new NotifyingListWrapperIterator();
    }

    @Override
    public NotifyingListIterator<E> listIterator(final int index)
    {
        return new NotifyingListWrapperIterator(index);
    }

    private class NotifyingListWrapperIterator extends ForwardingListIterator<E> implements NotifyingListIterator<E>
    {
        private final ListIterator<E> _delegateIterator;
        private E _lastReturned;
        private int _lastReturnedIndex;

        private NotifyingListWrapperIterator()
        {
            _delegateIterator = NotifyingListWrapper.this.delegate().listIterator();
        }

        private NotifyingListWrapperIterator(final int index)
        {
            _delegateIterator = NotifyingListWrapper.this.delegate().listIterator(index);
        }

        @Override
        protected ListIterator<E> delegate()
        {
            return _delegateIterator;
        }

        @Override
        public E next()
        {
            _lastReturnedIndex = super.nextIndex();
            _lastReturned = super.next();
            return _lastReturned;
        }

        @Override
        public E previous()
        {
            _lastReturnedIndex = super.previousIndex();
            _lastReturned = super.previous();
            return _lastReturned;
        }

        @Override
        public void remove()
        {
            remove(NotifyingListWrapper.this);
        }

        @SuppressWarnings({"rawtypes", "unchecked"})
        @Override
        public void remove(final Object source)
        {
            super.remove();
            post(new CollectionRemoveNotice(source, NotifyingListWrapper.this, _lastReturned));
        }

        @Override
        public void add(final E o)
        {
            add(o, NotifyingListWrapper.this);
        }

        @SuppressWarnings({"unchecked", "rawtypes"})
        @Override
        public void add(final E o, final Object source)
        {
            super.add(o);
            post(new ListAddNotice(source, NotifyingListWrapper.this, _lastReturnedIndex, o));
        }

        @Override
        public void set(final E o)
        {
            set(o, NotifyingListWrapper.this);
        }

        @SuppressWarnings({"unchecked", "rawtypes"})
        @Override
        public void set(final E o, final Object source)
        {
            super.set(o);
            post(new ListSetNotice(source, NotifyingListWrapper.this, _lastReturnedIndex, o, _lastReturned));
        }
    }
}
