package ohd.hseb.hefs.utils.piservice;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import nl.wldelft.fews.system.pi.FewsPiService;

/**
 * Currently used only by GraphGen (2/25/15), an object of this class monitors the FEWS PI-service through calls to
 * {@link FewsPiService#getLastUpdateTime()}. Whenever the last update time changes, it calls the method
 * {@link FewsPiServiceMonitorListener#reactToPossiblyChangedSegmentOrSystemTime()} for all
 * {@link FewsPiServiceMonitorListener} instances added via {@link #addListener(FewsPiServiceMonitorListener)}. If any
 * any time the call to method {@link FewsPiService#getLastUpdateTime()} throws an exception, this will call
 * {@link FewsPiServiceProvider#testConnectionAndLogAMessage()} forcing it to reconfirm that the connection still
 * exists. Note that this will only make the calls to {@link FewsPiService#getLastUpdateTime()} if
 * {@link FewsPiServiceProvider#checkLastAttemptToConnectSuccess()} returns true, and that is set via the test method.
 * 
 * @author hank.herr
 */
public class FewsPiServiceMonitor extends Thread
{
    private static final Logger LOG = LogManager.getLogger(FewsPiServiceMonitor.class);

    private final List<FewsPiServiceMonitorListener> _listeners = new ArrayList<FewsPiServiceMonitorListener>();

    private long _lastUpdateTime = Long.MIN_VALUE;
    private boolean _ignoreLastUpdateTimeUpdate = false;

    private long _lastModificationTimeDataSets = Long.MIN_VALUE;

    /**
     * Resets the FewsPiServiceProvider recorded segment id and system time. Records the update time and last
     * modificatin time. Readies the thread for starting.
     */
    public FewsPiServiceMonitor()
    {
        super("FewsPiServiceMonitor");
        FewsPiServiceProvider.resetActiveSegmentId();
        FewsPiServiceProvider.resetSystemTime();
        recordLastUpdateTime();
        recordLastModificationTimeDataSets();
        setIgnoreNextLastUpdateTimeUpdate(true);
    }

    public void addListener(final FewsPiServiceMonitorListener listener)
    {
        this._listeners.add(listener);
    }

    public void removeListener(final FewsPiServiceMonitorListener listener)
    {
        this._listeners.remove(listener);
    }

    private void fireReactToPossiblyChangedSegmentOrSystemTime()
    {
        for(final FewsPiServiceMonitorListener listener: _listeners)
        {
            listener.reactToPossiblyChangedSegmentOrSystemTime();
        }
    }

    private synchronized void recordLastUpdateTime()
    {
        if(FewsPiServiceProvider.checkLastAttemptToConnectSuccess())
        {
            try
            {
                _lastUpdateTime = FewsPiServiceProvider.getDefaultService().getLastUpdateTime();
            }
            catch(final Exception e)
            {

            }
        }
    }

    private synchronized void recordLastModificationTimeDataSets()
    {
        if(FewsPiServiceProvider.checkLastAttemptToConnectSuccess())
        {
            try
            {
                _lastModificationTimeDataSets = FewsPiServiceProvider.getDefaultService()
                                                                     .getLastModificationTimeDataSets();
            }
            catch(final Exception e)
            {
            }
        }
    }

    private synchronized void recordLastUpdateTime(final long updateTime)
    {
        _lastUpdateTime = updateTime;
    }

    public void setIgnoreNextLastUpdateTimeUpdate(final boolean b)
    {
        this._ignoreLastUpdateTimeUpdate = b;
    }

    public synchronized void recordLastModificationTimeDataSets(final long updateTime)
    {
        _lastModificationTimeDataSets = updateTime;
    }

    @Override
    public void interrupt()
    {
        //TODO This is only here so that I can see if FEWS calls the interrupt method!
        super.interrupt();
    }

    @Override
    public void run()
    {
        LOG.info("Starting FEWS pi-service monitor (last update time is " + new Date(_lastUpdateTime)
            + "; last modification time for data sets " + new Date(this._lastModificationTimeDataSets) + ")...");
        try
        {
            while(true)
            {
                sleep(100);
                if(FewsPiServiceProvider.checkLastAttemptToConnectSuccess())
                {
                    try
                    {
                        //Last update time check.
                        final long lastUpdateTime = FewsPiServiceProvider.getDefaultService().getLastUpdateTime();
                        if(_lastUpdateTime != lastUpdateTime)
                        {
                            if(!_ignoreLastUpdateTimeUpdate)
                            {
                                FewsPiServiceProvider.resetActiveSegmentId();
                                FewsPiServiceProvider.resetSystemTime();
                                fireReactToPossiblyChangedSegmentOrSystemTime();
                            }
                            else
                            {
                                _ignoreLastUpdateTimeUpdate = false;
                            }
                            recordLastUpdateTime(lastUpdateTime);
                        }
                    }
                    catch(final Exception e)
                    {
                        LOG.warn("Problem encountered trying to process system time or active segment change; confirming PI-service connection is still established: "
                            + e.getMessage());
                        FewsPiServiceProvider.testConnectionAndLogAMessage();
                    }
                }
            }
        }
        catch(final InterruptedException e)
        {
        }
    }
}
