package ohd.hseb.hefs.utils.piservice;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import nl.wldelft.fews.pi.PiTimeSeriesParser;
import nl.wldelft.fews.common.config.GlobalProperties;
import nl.wldelft.fews.system.pi.FewsPiService;
import nl.wldelft.util.FileUtils;
import nl.wldelft.util.IOUtils;
import nl.wldelft.util.XmlUtils;
import nl.wldelft.util.io.LineReader;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.SimpleTimeSeriesContentHandler;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesHeaderInfoList;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandlerException;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HStopWatch;
import ohd.hseb.util.misc.HString;

import org.apache.commons.codec.binary.Base64;
//import org.apache.log4j.lf5.util.StreamUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.codehaus.xfire.client.XFireProxyFactory;
import org.codehaus.xfire.service.Service;
import org.codehaus.xfire.service.binding.ObjectServiceFactory;

import com.google.common.base.Strings;

/**
 * This class provides wrapper methods on the FewsPiService methods. It also provides a means to acquire a
 * "default service", or instance of FewsPiService, if it is possible. This can be used to call the core methods, most
 * of which are not wrapped herein.
 * 
 * @author herrhd
 */
public class FewsPiServiceProvider
{
    private static final Logger LOG = LogManager.getLogger(FewsPiServiceProvider.class);

    public static String ACTIVE_SEGMENT_NONE = "none";
    static Map<Thread, FewsPiServiceAttributes> _threadToServiceInfoMap = new HashMap<Thread, FewsPiServiceAttributes>();

    /**
     * @return Whether or not the service for this thread is currently connected. This will NOT try to connect to a
     *         default service! Rather, it calls {@link #getStorageAttributes(Thread, boolean)} with a second argument
     *         of false purposely making it only check for a connection.
     */
    public static boolean isDefaultServiceConnected()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread(), false);
        return (threadInfo != null) && (threadInfo.getDefaultService() != null); //Prevent possibly null ptr exception
    }

    /**
     * Calls {@link #getStorageAttributes(Thread, boolean)} passing in true for the second argument so that the default
     * service will be initialized if nothing is found for the provided thread and the default has not be initialized
     * yet.
     */
    private static FewsPiServiceAttributes getStorageAttributes(final Thread thread)
    {
        return getStorageAttributes(thread, true);
    }

    /**
     * FB 1728: Every method that uses any variable in the {@link FewsPiServiceAttributes} class must call the below
     * code to identify the {@link FewsPiServiceAttributes} class instance to use when accessing those variables.
     */
    private static FewsPiServiceAttributes getStorageAttributes(final Thread thread,
                                                                final boolean initializeDefaultIfNotFound)
    {
        FewsPiServiceAttributes threadInfo = _threadToServiceInfoMap.get(thread);

        //None found, so look for the default.
        if(threadInfo == null)
        {
            threadInfo = _threadToServiceInfoMap.get(null);

            //Default not found, so create the default and return it.
            if((threadInfo == null) && (initializeDefaultIfNotFound))
            {
                threadInfo = new FewsPiServiceAttributes(null);
                _threadToServiceInfoMap.put(null, threadInfo);
                initializeDefaultService(false, null);
                threadInfo = _threadToServiceInfoMap.get(null);
            }
        }
        return threadInfo;
    }

    /**
     * Initialize the default service. Calls {@link #initializeDefaultService(boolean, String, String)}.
     * 
     * @param attemptReconnect If true, attempt a reconnect if the last attempt to connect failed.
     * @param portNumberOverride If not null, provides an override for the port number acquired via the properties.
     */
    public synchronized static void initializeDefaultService(final boolean attemptReconnect,
                                                             final String portNumberOverride)
    {
        initializeDefaultService(null, attemptReconnect, portNumberOverride, null, null);
    }

    /**
     * Initialize the default service.
     * 
     * @param attemptReconnect If true, attempt a reconnect if the last attempt to connect failed.
     * @param portNumberOverride If not null, provides an override for the port number acquired via the properties.
     * @param hostNameOverride If not null, provides an override for the host name of the pi-service.
     * @param urlBackendRFCIdentifierOverride The portion of the url string used in the backend PI-service that
     *            identifies the rfc. For example, for nerfc, it would be "nerfc_pi". This string is inserted after the
     *            host-name and portnumber and before the "FewsPiService" suffix (slashes, '/', will be inserted around
     *            it). This overrides what is in the global properties.
     */
    public synchronized static void initializeDefaultService(final Thread thread, //Add this as an argument.  Make sure everything set in here is stored in the map.  Adapter should pass in Thread.currentThread().
                                                             final boolean attemptReconnect,
                                                             final String portNumberOverride,
                                                             final String hostNameOverride,
                                                             final String urlBackendRFCIdentifierOverride)
    {
        //I need to grab the existing ThreadInfo if one exists.  If it does, and it failed ot connect, then reference
        //the attempt reconnect argument.  If we are not to attempt to reconnect, just return false.
        FewsPiServiceAttributes threadInfo = _threadToServiceInfoMap.get(thread);
        if(threadInfo != null)
        {
            threadInfo.setCurrentConnectionPortNumber("");
            threadInfo.setCurrentConnectionURL("");
            if((threadInfo.getLastAttemptToConnectFailed()) && (!attemptReconnect))
            {
                threadInfo.setCurrentConnectingToPiService(false);
                return;
            }

            //If you get to this point without returning, then we are about to reconnect, so remove the existing Threadinfo.
            _threadToServiceInfoMap.remove(thread);
        }

        //At this point, I *know* I need to connect from scratch.  First, performa general cleanup on the map.
        removeInactiveThreadsFromServiceInfoMap();

        //Now, construct a new threadInfo and put it into the map.
        threadInfo = new FewsPiServiceAttributes(threadInfo);
        _threadToServiceInfoMap.put(thread, threadInfo);

        //We are currently trying to connect. 
        threadInfo.setCurrentConnectingToPiService(true);

        //String portNumber = GlobalProperties.get("portNumber");
        //String hostName = GlobalProperties.get("hostName");
        String portNumber = "8100";
        if(GlobalProperties.get("graphGenHEFSPIServicePortNumber") != null)
        {
            portNumber = GlobalProperties.get("graphGenHEFSPIServicePortNumber");
        }
        if(portNumberOverride != null)
        {
            portNumber = portNumberOverride;
        }

        //Defaults to localHost.
        String hostName = "localHost";
        if(GlobalProperties.get("graphGenHEFSPIServiceHostName") != null)
        {
            hostName = GlobalProperties.get("graphGenHEFSPIServiceHostName");
        }
        if(hostNameOverride != null)
        {
            hostName = hostNameOverride;
        }

        //Defaults to none.
        String urlBackendRFCIdentifier = null;
        if(GlobalProperties.get("graphGenHEFSBackendPIServiceRFCIdentifierForURL") != null)
        {
            urlBackendRFCIdentifier = GlobalProperties.get("graphGenHEFSBackendPIServiceRFCIdentifierForURL");
        }
        if(urlBackendRFCIdentifierOverride != null)
        {
            urlBackendRFCIdentifier = urlBackendRFCIdentifierOverride;
        }

        LOG.info("OHD FEWS explorer plug-in software establishing connection to CHPS FewsPiServiceImpl on " + hostName
            + " : " + portNumber + "...");

        String serverUrl = "http://" + hostName + ':' + String.valueOf(portNumber) + "/FewsPiService";
        if(!Strings.isNullOrEmpty(urlBackendRFCIdentifier)) //Only use if the RFC identifier is non-empty
        {
            serverUrl = "http://" + hostName + ':' + String.valueOf(portNumber) + "/" + urlBackendRFCIdentifier
                + "/FewsPiService";
        }
        LOG.info("The PI-service URL used is: " + serverUrl);
        threadInfo.setDefaultService(getWebService(serverUrl));
        testConnectionAndLogAMessage();
        threadInfo.setCurrentConnectingToPiService(false);

        //If we make it this far, then record the current port number and full URL, and fire the react event.
        threadInfo.setCurrentConnectionPortNumber(portNumber);
        threadInfo.setCurrentConnectionURL(serverUrl);
        fireReactToReinitializedConnection();
    }

    /**
     * Reinitializes the PI-service connection. It is intended to mimick
     * {@link #initializeDefaultService(boolean, String, String, String)}, but uses the existing stored
     * {@link #_currentConnectionURL}. It will blank out the {@link #_currentConnectionPortNumber} and
     * {@link #_currentConnectionURL} if a failure occurs, just as
     * {@link #initializeDefaultService(boolean, String, String, String)} will do.
     */
    public synchronized static void reinitializeDefaultService()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());

        //TODO This should not be needed.  Somehow _currentConnectionURL is being blanked out after the
        //previous connection attempt and before clicking this button.  Why?
        if(Strings.isNullOrEmpty(threadInfo.getCurrentConnectionURL()))
        {
            LOG.warn("The current connection was not successful, so a reconnect cannot be done.");
            return;
        }
        //Record the url and connection port number strings.
        final String url = threadInfo.getCurrentConnectionURL();
        final String currentConnectionPortNumber = threadInfo.getCurrentConnectionPortNumber();
        threadInfo.setCurrentConnectionPortNumber("");
        threadInfo.setCurrentConnectionURL("");

        //Reconnect...
        threadInfo.setCurrentConnectingToPiService(true);
        LOG.info("The PI-service URL used is: " + url);
        threadInfo.setDefaultService(getWebService(url));
        testConnectionAndLogAMessage();
        threadInfo.setCurrentConnectingToPiService(false);

        //Store the URL and port number strings
        threadInfo.setCurrentConnectionURL(url);
        threadInfo.setCurrentConnectionPortNumber(currentConnectionPortNumber);

        //Fire the event.
        fireReactToReinitializedConnection();
    }

    /**
     * Convenience call to {@link #getStorageAttributes(Thread)} and
     * {@link FewsPiServiceAttributes#getCurrentConnectingToPiService()}.
     */
    private static boolean isCurrentlyConnectingToPIService()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread(), false);
        return threadInfo.getCurrentConnectingToPiService();
    }

    /**
     * Convenience call to {@link #getStorageAttributes(Thread)} and
     * {@link FewsPiServiceAttributes#getCurrentConnectionPortNumber()}.
     */
    public static String getCurrentConnectionPortNumber()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        return threadInfo.getCurrentConnectionPortNumber();
    }

    /**
     * Will do nothing if {@link #_defaultService} is null. Otherwise, it will attempt to get the system time. If the
     * test fails, {@link #_defaultService} will be set to null, {@link #_lastAttemptToConnectFailed} will be true, and
     * log warnings will be output.
     */
    protected static void testConnectionAndLogAMessage() //XXX FREDDY -- You may want to put this method in the inner class. 
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        //Default service is not present... do nothing.
        if(threadInfo.getDefaultService() == null)
        {
            return;
        }

        try
        {
            final Date systemTime = threadInfo.getDefaultService().getSystemTime(null);
            LOG.info("Connection established successfully.  System time: " + systemTime);
            threadInfo.setLastAttemptToConnectFailed(false);
        }
        catch(final Throwable e)
        {
            LOG.warn("Failed to connect to database with this message: " + e.getMessage());
            LOG.warn("Future attempts to use connection will fail.");
            threadInfo.setDefaultService(null);
            threadInfo.setLastAttemptToConnectFailed(true);
        }
    }

    /**
     * This method catches any {@link Throwable}s generated in an attempt to acquire the PI-service. If any throwable
     * occurs, the returned {@link FewsPiService} will be null, {@link #_lastAttemptToConnectFailed} will be true, an
     * error message output to log (these exceptions/errors should NEVER occur and likely indicate a bad class path),
     * and stacktrace dumped.
     */
    private static FewsPiService getWebService(final String serviceIdent)
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        try
        {
            final XFireProxyFactory proxyFactory = new XFireProxyFactory();
            Object proxy = null;
            final ObjectServiceFactory serviceFactory = new ObjectServiceFactory();
            final Service service = serviceFactory.create(FewsPiService.class);
            proxy = proxyFactory.create(service, serviceIdent);
            if(proxy != null && proxy instanceof FewsPiService)
            {
                return (FewsPiService)proxy;
            }
            else
            {
                threadInfo.setLastAttemptToConnectFailed(true);
                return null;
            }
        }
        catch(final Throwable t)
        {
            t.printStackTrace();
            LOG.error("Unexpected exception/error getting web service: " + t.getMessage());
            threadInfo.setLastAttemptToConnectFailed(true);
            return null;
        }
    }

    /**
     * @return True if the last attempt to connect was successful, false if not.
     */
    public static boolean checkLastAttemptToConnectSuccess()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        return !threadInfo.getLastAttemptToConnectFailed();
    }

    /**
     * @return The _defaultService, reinitializing if needed and if a connection can be established.
     */
    public static FewsPiService getDefaultService()
    {
        //Pass in false for the re-initialize argument at this point.  It will be handled directly below.
        FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread(), false);

        //This class will need to return the service corresponding to the active thread and, if it is not found,
        //then call the DEFAULT initialization (passing in null for the Thread) and return that default.
        //
        //If a threadInfo is found, then wait until any current reconnect is done.
        if(threadInfo != null)
        {
            while(isCurrentlyConnectingToPIService())
            {
                try
                {
                    Thread.sleep(10);
                }
                catch(final InterruptedException e)
                {
                }
            }
        }

        //Now, if there was no thread info or the one that was foudn did not successfully connect, try to reconnect and return
        //the resulting service blindly.
        if((threadInfo == null) || (!isDefaultServiceConnected()))
        {
            initializeDefaultService(false, null);
            threadInfo = getStorageAttributes(Thread.currentThread(), false);
        }
        return threadInfo.getDefaultService();
    }

    /**
     * Calls {@link #resetSystemTime()} if the system time stored in the {@link #getStorageAttributes(Thread)} object is
     * null. Otherwise, it just returns that time.
     * 
     * @return The current system time according to the connected PI-service.
     */
    public static Date getSystemTime()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        if(threadInfo.getCurrentSystemTime() == null)
        {
            resetSystemTime();
        }
        return threadInfo.getCurrentSystemTime();
    }

    /**
     * Resets the internal static system time based on the pi-service getSystemTime(null) return.
     */
    public synchronized static void resetSystemTime()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        if(getDefaultService() == null)
        {
            threadInfo.setCurrentSystemTime(null);
        }
        else
        {
            threadInfo.setCurrentSystemTime(getDefaultService().getSystemTime(null));
        }
    }

    /**
     * @return The internal static active segment id, which is updated by the {@link FewsPiServiceMonitor} as needed.
     *         This method does not access the pi-service directly, but reflects such an access by the monitor.
     */
    public static String getActiveSegmentId()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        return threadInfo.getActiveSegmentId();
    }

    /**
     * Resets the internal static active segment id based on the pi-service getActiveSegmentId() return.
     */
    public synchronized static void resetActiveSegmentId()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        threadInfo.setActiveSegmentId(null);
        if(getDefaultService() != null)
        {
            threadInfo.setActiveSegmentId(FewsPiServiceProvider.getDefaultService().getActiveSegmentId());
            LOG.debug("Active segment in PI-service found to be " + threadInfo.getActiveSegmentId());
        }
    }

    /**
     * return the time zone ID from the fews pi-service.
     * 
     * @return
     */
    public static String getTimeZoneId()
    {
        if(FewsPiServiceProvider.getDefaultService() != null)
        {
            //I assume the argument is a client id, so pass in the standard GraphGen.

            //XXX If this is null, an ERROR is reported by the pi-service, despite the fact that 
            //documentation implies passing null is ok.
            return getDefaultService().getTimeZoneId("GraphGen");
        }
        else
        {
            return null;
        }
    }

    /**
     * Applies a rating curve to a time series, with the location id in the time series identifying the curve. The
     * resulting time series will have a new parameter id returned by obtainRatingCurveConvertedParameterId.
     * 
     * @param ts Time series to convert.
     * @param stageToFlow True if stage to flow, false if flow to stage.
     * @return Time series that is a clone of the original but with new values and a new parameter id.
     * @throws Exception
     */
    public static TimeSeriesArray applyRatingCurveToTimeSeries(final TimeSeriesArray ts, final boolean stageToFlow) throws Exception
    {
        final FewsPiService defaultService = FewsPiServiceProvider.getDefaultService();
        if(defaultService == null)
        {
            throw new Exception("CHPS web-service cannot be established.  " + "Cannot load rating curve.");
        }

        final float[] values = new float[ts.size()];
        for(int i = 0; i < values.length; i++)
        {
            values[i] = ts.getValue(i);
        }

        float[] results;
        final String newParameterId = obtainRatingCurveConvertedParameterId(ts.getHeader().getParameterId(),
                                                                            stageToFlow);
        final String newParameterName = ts.getHeader().getParameterName() + " RC converted";

        if(stageToFlow)
        {
            results = defaultService.convertStageToDischarge(values,
                                                             ts.getHeader().getLocationId(),
                                                             ts.getForecastTime());
        }
        else
        {
            results = defaultService.convertDischargeToStage(values,
                                                             ts.getHeader().getLocationId(),
                                                             ts.getForecastTime());
        }
        // replace the clone() method by duplicate() 
        final TimeSeriesArray newTS = ts.duplicate();
        for(int i = 0; i < results.length; i++)
        {
            newTS.setValue(i, results[i]);
        }

        ((DefaultTimeSeriesHeader)newTS.getHeader()).setParameterId(newParameterId);
        ((DefaultTimeSeriesHeader)newTS.getHeader()).setParameterName(newParameterName);

        return newTS;
    }

    /**
     * Applies a rating curve to the given set of values.
     * 
     * @param values Values to convert.
     * @param stageToFlow True if stage to flow conversion, false if flow to stage.
     * @param locationId Location id identifying the rating curve.
     * @param unitConversionRequired If true, then the values will be inverse converted to storage units (metric) before
     *            applying the rating curve and converted back to display units after. The rating curve is applied in
     *            storage/metric units.
     * @param T0 The forecast time corresponding to the values to convert.
     * @return Array of converted values.
     * @throws Exception If the service cannot be established or the called pi-service methods throws one.
     */
    public static double[] applyRatingCurveToValues(final double[] values,
                                                    final boolean stageToFlow,
                                                    final String locationId,
                                                    final boolean unitConversionRequired,
                                                    final int signigicantDigits,
                                                    Long T0) throws Exception
    {
        final FewsPiService defaultService = FewsPiServiceProvider.getDefaultService();
        if(defaultService == null)
        {
            throw new Exception("CHPS web-service cannot be established.  " + "Cannot load rating curve.");
        }

        //Setup an assumed parameter id for the original data and converted data based on flag.
        final String originalParameterId = getRatingCurveAssumedOriginalParameterId(stageToFlow);
        final String targetParameterId = getRatingCurveAssumedTargetParameterId(stageToFlow);

        if(T0 == null)
        {
            T0 = getSystemTime().getTime();
        }

        //Convert to floats
        final float[] floatValues = new float[values.length];
        for(int i = 0; i < values.length; i++)
        {
            floatValues[i] = (float)values[i];
        }

        float[] floatResults = floatValues;

        //Convert to original (storage) units if necessary.  The PI-service rating curve method works in storage units only.
        if(unitConversionRequired)
        {
            floatResults = defaultService.inverseConvertToDisplayUnitValue(originalParameterId, floatResults);
        }

        //Obtain results
        if(stageToFlow)
        {
            floatResults = defaultService.convertStageToDischarge(floatResults, locationId, T0);
        }
        else
        {
            floatResults = defaultService.convertDischargeToStage(floatResults, locationId, T0);
        }

        //Convert back to display units, if necessary.
        if(unitConversionRequired)
        {
            floatResults = defaultService.convertToDisplayUnitValue(targetParameterId, floatResults);
        }

        //TODO: START FB 1835 - Need to apply the number of significant digits here.
        //Convert back to double
        final double[] results = new double[floatResults.length];
        for(int i = 0; i < results.length; i++)
        {
            //results[i] = floatResults[i];
            results[i] = roundToSignificantFigures(floatResults[i], signigicantDigits);
        }

        return results;
    }

    /**
     * Round number to a giving significant figures number.
     * 
     * @param num the number to round
     * @param n the number of significant digits to round the given number
     * @return the rounded number if it is valid operation, otherwise the same value.
     */
    public static double roundToSignificantFigures(final double numberToRound, final int numberOfSignificantDigits)
    {
        if(numberToRound == 0)
        {
            return 0;
        }
        if(numberOfSignificantDigits == 0)
        {
            return numberToRound;
        }

        final double d = Math.ceil(Math.log10(numberToRound < 0 ? -numberToRound : numberToRound));
        final int power = numberOfSignificantDigits - (int)d;

        final double magnitude = Math.pow(10, power);
        final long shifted = Math.round(numberToRound * magnitude);
        return shifted / magnitude;
    }

    // TODO: END FB 1835

    /**
     * @param stageToFlow True if stage to flow; false if flow to stage.
     * @return A parameter id that is assumed to be representative of the targetted data type in terms of its units.
     *         Since the units for flow can be QIN, QINE, SQIN, etc, we assume QIN is representative of the whole group.
     *         Stg is returned for stage.
     */
    public static String getRatingCurveAssumedTargetParameterId(final boolean stageToFlow)
    {
        if(stageToFlow)
        {
            return "QIN";
        }
        return "STG";
    }

    /**
     * @param stageToFlow True if stage to flow; false if flow to stage.
     * @return A parameter id that is assumed to be representative of the original data type in terms of its units.
     *         Since the units for flow can be QIN, QINE, SQIN, etc, we assume QIN is representative of the whole group.
     *         Stg is returned for stage.
     */
    public static String getRatingCurveAssumedOriginalParameterId(final boolean stageToFlow)
    {
        if(stageToFlow)
        {
            return "STG";
        }
        return "QIN";
    }

    /**
     * Returns the parameter id to use after applying a rating curve. This does NOT use the pi-service.
     * 
     * @param originalId
     * @param stageToFlow True if the conversion is stage to flow, false if flow to stage.
     * @return New parameter id resulting from rating curve.
     */
    public static String obtainRatingCurveConvertedParameterId(final String originalId, final boolean stageToFlow)
    {
        if(stageToFlow)
        {
            if(originalId.equalsIgnoreCase("STG"))
            {
                return "QIN";
            }
            else if(originalId.equalsIgnoreCase("STG"))
            {
                return "QINE";
            }
            else
            {
                return originalId + " RC";
            }
        }
        else
        {
            if(originalId.equalsIgnoreCase("QIN"))
            {
                return "STG";
            }
            else if((originalId.equalsIgnoreCase("QINE")) || (originalId.equalsIgnoreCase("SQIN")))
            {
                return "SSTG";
            }
            else
            {
                return originalId + " RC";
            }
        }
    }

    /**
     * Call this instead of
     * {@link FewsPiService#getTimeSeriesBytes(String, String, String, Date, Date, Date, String[], String[], String, int)}
     * in order to acquire data, since this will not have the length limit that the Deltares method has.
     * 
     * @return The byte[] containing the data corresponding to the provided query information.
     * @throws Exception for any one of various reasons.
     */
    private static byte[] getTimeSeriesBytes(final String clientId,
                                             final String queryId,
                                             final Date startTime,
                                             final Date t0,
                                             final Date endTime,
                                             final String[] parameterIds,
                                             final String[] locationIds,
                                             final String ensembleId,
                                             final int ensembleMemberIndex) throws Exception
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());

        if(getDefaultService() == null)
        {
            throw new IOException("No PI-service connection is established, so time series binary data cannot be acquired.");
        }
        if(Strings.isNullOrEmpty(threadInfo.getCurrentConnectionURL()))
        {
            throw new IOException("Though a PI-service connection has been established, the URL was not recorded; cannot acquire binary ts data.");
        }

        //Get the connection.  This connection will be closed later.
        //final URL url = new URL(_currentConnectionURL);
        final URL url = new URL(threadInfo.getCurrentConnectionURL());
        final HttpURLConnection connection = (HttpURLConnection)url.openConnection();

        try
        {
            connection.setDoOutput(true);
            connection.setDoInput(true);
            connection.setRequestMethod("GET");
            connection.setRequestProperty("SOAPAction", "");
            connection.setRequestProperty("Content-type", "text/xml");
            final OutputStream outputStream = connection.getOutputStream();
            final OutputStreamWriter wout = new OutputStreamWriter(outputStream);

            //First part of the soap message...
            String toWrite = "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xmlns:ns1=\"http://pi.system.fews.wldelft.nl\">\n"
                + "   <soapenv:Body>\n" + "      <ns1:getTimeSeriesBytes>\n";
            toWrite += "         <ns1:in0>" + clientId + "</ns1:in0>\n";
            toWrite += "         <ns1:in1>" + queryId + "</ns1:in1>\n";
            toWrite += "         <ns1:in2 xsi:nil=\"true\"/>\n";

            //Dates should work even if MIN/MAX are provided.
            toWrite += "         <ns1:in3>"
                + HCalendar.buildDateStr(startTime.getTime(), "yyyy-MM-dd'T'HH:mm:ss.SSSXXX") + "</ns1:in3>\n";
            toWrite += "         <ns1:in4>" + HCalendar.buildDateStr(t0.getTime(), "yyyy-MM-dd'T'HH:mm:ss.SSSXXX")
                + "</ns1:in4>\n";
            toWrite += "         <ns1:in5>" + HCalendar.buildDateStr(endTime.getTime(), "yyyy-MM-dd'T'HH:mm:ss.SSSXXX")
                + "</ns1:in5>\n";

            //Parameter ids as an array
            if(parameterIds == null)
            {
                toWrite += "         <ns1:in6 xsi:nil=\"true\"/>\n";
            }
            else
            {
                toWrite += "         <ns1:in6>\n";
                for(final String parameterId: parameterIds)
                {
                    toWrite += "         //if(Strings.isNullOrEmpty(_currentConnectionURL))             <ns1:string>"
                        + parameterId + "</ns1:string>\n";
                }
                toWrite += "         </ns1:in6>\n";
            }

            //Location ids as an array.
            if(locationIds == null)
            {
                toWrite += "         <ns1:in7 xsi:nil=\"true\"/>\n";
            }
            else
            {
                toWrite += "         <ns1:in7>\n";
                for(final String locationId: locationIds)
                {
                    toWrite += "             <ns1:string>" + locationId + "</ns1:string>\n";
                }
                toWrite += "         </ns1:in7>\n";
            }

            //Ensemble id if given.
            if(ensembleId == null)
            {
                toWrite += "         <ns1:in8 xsi:nil=\"true\"/>\n";
            }
            else
            {
                toWrite += "         <ns1:in8>" + ensembleId + "</ns1:in8>\n";
            }

            //Close it out with the member index and remaining stuff.
            toWrite += "         <ns1:in9>" + ensembleMemberIndex + "</ns1:in9>\n";
            toWrite += "      </ns1:getTimeSeriesBytes>\n";
            toWrite += "   </soapenv:Body>\n";
            toWrite += "</soapenv:Envelope>";

            //Write the message and close the opening.
            wout.write(toWrite);
            wout.flush();
            wout.close();

            //Try to get the input stream with the return value.
            InputStream in;
            try
            {
                in = connection.getInputStream();               
            }
            catch(final IOException e)
            {
                in = connection.getErrorStream();
                
                
                //final byte[] errorBytes = StreamUtils.getBytes(in);

                final byte[] errorBytes = org.apache.commons.io.IOUtils.toByteArray(in);
                
                LOG.debug("When processing soap message to get binary ts data, an exception occurred wit this message: "
                    + e.getMessage());
                LOG.debug("The error stream XML content was as follows: " + new String(errorBytes));
                throw new IOException("Error processing soap message to get binary ts data: " + e.getMessage());
            }

            //Read the bytes.
            byte[] bytes = null;
            XMLStreamReader xmlByteReader = null;
            try
            {
                xmlByteReader = XmlUtils.createStreamReader(in, "timeseries.xml");

                String xmlValue = null;
                while(xmlByteReader.hasNext())
                {
                    xmlByteReader.nextTag();
                    if(!xmlByteReader.isStartElement())
                        continue;
                    if(!xmlByteReader.getLocalName().equals("out"))
                        continue;
                    xmlValue = xmlByteReader.getElementText();
                    break;
                }
                bytes = Base64.decodeBase64(xmlValue.getBytes()); //Convert to original byte array
            }
            catch(final Exception e)
            {
                throw new IOException("Failed to process binary read" + e.getMessage());
            }
            finally
            {
                try
                {
                    xmlByteReader.close();
                }
                catch(final XMLStreamException e)
                {
                    e.printStackTrace();
                    throw new IOException("GOTTA CHANGE THIS MOTHER FUCKER!!!");
                }
            }
            return bytes;
        }
        finally
        {
            connection.disconnect();
        }
    }

    /**
     * Wraps call to getTimeSeries within the PI-service and translates results to a TimeSeriesArrays object.
     * 
     * @param clientId Corresponds to the name of the file in the PIServiceConfigFiles directory.
     * @param queryId The id of the query defined with the file.
     * @param startTime Start time of time series to return.
     * @param t0 The system time, but I'm not sure how its used.
     * @param endTime End time of the time series to return.
     * @param parameterIds Restricts query results to time series with these parameterIds.
     * @param locationIds Restricts query results to time series with these locationIds.
     * @param ensembleId Restricts query results to time series with this ensemble id.
     * @param ensembleMemberIndex Restricts query results to time series with this member index.
     * @param includeThresholds If true, then returned time series will include threshold info.
     * @return TimeSeriesArrays specifying found time series, or null if no XML string was returned by PI-service,
     *         implying no time series.
     * @throws IOException If there is a problem parsing string returned by PI-service, which there better not be.
     */
    public static TimeSeriesArrays retrieveTimeSeries(final String clientId,
                                                      final String queryId,
                                                      final Date startTime,
                                                      final Date t0,
                                                      final Date endTime,
                                                      final String[] parameterIds,
                                                      final String[] locationIds,
                                                      final String ensembleId,
                                                      final int ensembleMemberIndex,
                                                      final boolean includeThresholds) throws Exception
    {
//        System.out.println("####>> getting ts -- '" + clientId + "' '" + queryId + "'.");
//        System.out.println("####>>        start - " + startTime);
//        System.out.println("####>>        t0 - " + t0);
//        System.out.println("####>>        end - " + endTime);
//        System.out.println("####>>        parameterIds -- " + Arrays.toString(parameterIds));
//        System.out.println("####>>        locations -- " + Arrays.toString(locationIds));
//        System.out.println("####>>        ensembleId -- " + ensembleId);
//        System.out.println("####>>        member index -- " + ensembleMemberIndex);
//        System.out.println("####>>        include thresholds -- " + includeThresholds);

        //Try to acquire the data using binary reading.  If any part of the process yields a problem, instead using standard PI-service 
        //very slow XML parsing (see catch).
        try
        {
            HStopWatch binaryTimer;
            HStopWatch headerTimer;
            HStopWatch creatingTimeSeriesTimer;

            //Get the binary data...
            binaryTimer = new HStopWatch();
            final byte[] bytes = getTimeSeriesBytes(clientId,
                                                    queryId,
                                                    startTime,
                                                    t0,
                                                    endTime,
                                                    parameterIds,
                                                    locationIds,
                                                    ensembleId,
                                                    ensembleMemberIndex);

            if(bytes == null)
            {
                throw new Exception("Returned bytes array was null.");
            }
            binaryTimer.stop();

            //Get the header...
            headerTimer = new HStopWatch();
            final String header = getDefaultService().getTimeSeriesHeaders(clientId,
                                                                           queryId,
                                                                           null,
                                                                           startTime,
                                                                           t0,
                                                                           endTime,
                                                                           parameterIds,
                                                                           locationIds,
                                                                           ensembleId,
                                                                           ensembleMemberIndex,
                                                                           includeThresholds);
            if(header == null)
            {
                throw new IOException("No time series header could be found for client '" + clientId + "', query '"
                    + queryId + "', locations " + HString.buildStringFromArray(locationIds, ",")
                    + ", and other provided parameters.");
            }
            headerTimer.stop();

            //Prepare for parsing the header and binary data together.
            creatingTimeSeriesTimer = new HStopWatch();
            final PiTimeSeriesParser piTimeSeriesParser = new PiTimeSeriesParser();
            final String virtualFileName = "example";
            final LineReader reader = new LineReader(new StringReader(header), virtualFileName);

            //Map the virtual file name above to the binary data and set the virtual input dir for the parser.
            final HashMap<String, byte[]> virtualFileNameToBytesMap = new HashMap<>();
            virtualFileNameToBytesMap.put(FileUtils.getPathWithOtherExtension(virtualFileName, "bin"), bytes);
            piTimeSeriesParser.setVirtualInputDir(IOUtils.createVirtualInputDir(virtualFileNameToBytesMap));

            //Parse the XML.
            final SimpleTimeSeriesContentHandler importContentHandler = new SimpleTimeSeriesContentHandler();
            XMLStreamReader xmlReader = null;
            try
            {
                xmlReader = XmlUtils.createStreamReader(reader, virtualFileName);
                piTimeSeriesParser.parse(xmlReader, virtualFileName, importContentHandler);
                final TimeSeriesArrays timeSeriesArrays = importContentHandler.getTimeSeriesArrays();
                creatingTimeSeriesTimer.stop();

                LOG.debug("PI-service retrieve time series breakdown: get binary data = "
                    + binaryTimer.getElapsedMillis() + " msec; get XML headers = " + headerTimer.getElapsedMillis()
                    + " msec; parse into time series = " + creatingTimeSeriesTimer.getElapsedMillis() + " msec.");
                return timeSeriesArrays;
            }
            catch(final Exception e)
            {
                //The exception will be caught below.
                throw new IOException(e.getMessage());
            }
            finally
            {
                try
                {
                    xmlReader.close();
                }
                catch(final Exception e)
                {
                    throw new IOException(e.getMessage());
                }
            }

        }

        //The catch method uses standard XML reading.  If a problem is encountered here, it is thrown to the caller.
        catch(final Throwable t)
        {
            LOG.warn("Unable to acquire time series data via binary reading (XML will be used instead); reason: "
                + t.getMessage());

            final String xmlString = getDefaultService().getTimeSeries(clientId,
                                                                       queryId,
                                                                       null,
                                                                       startTime,
                                                                       t0,
                                                                       endTime,
                                                                       parameterIds,
                                                                       locationIds,
                                                                       ensembleId,
                                                                       ensembleMemberIndex,
                                                                       includeThresholds);
            TimeSeriesArrays ts = null;
            if(xmlString != null)
            {
                ts = TimeSeriesArraysTools.createTimeSeriesArraysFromXml(xmlString);
            }
            return ts;
        }

    }

    /**
     * Calls
     * {@link FewsPiService#getTimeSeriesHeaders(String, String, String, Date, Date, Date, String[], String[], String, int, boolean)}
     * . It parses it via {@link TimeSeriesHeaderInfoList} and returns the resulting list.
     * 
     * @param clientId Corresponds to the name of the file in the PIServiceConfigFiles directory.
     * @param queryId The id of the query defined with the file.
     * @param startTime Start time of time series to return.
     * @param t0 The system time, but I'm not sure how its used.
     * @param endTime End time of the time series to return.
     * @param parameterIds Restricts query results to time series with these parameterIds.
     * @param locationIds Restricts query results to time series with these locationIds.
     * @param ensembleId Restricts query results to time series with this ensemble id.
     * @param ensembleMemberIndex Restricts query results to time series with this member index.
     * @param includeThresholds If true, then returned time series will include threshold info.
     * @return TimeSeriesHeaderInfoList containing information about the time series includiong TimeSeriesHeader
     *         objects.
     * @throws IOException If there is a problem parsing string returned by PI-service, which there better not be.
     */
    public static TimeSeriesHeaderInfoList retrieveTimeSeriesHeaders(final String clientId,
                                                                     final String queryId,
                                                                     final Date startTime,
                                                                     final Date t0,
                                                                     final Date endTime,
                                                                     final String[] parameterIds,
                                                                     final String[] locationIds,
                                                                     final String ensembleId,
                                                                     final int ensembleMemberIndex,
                                                                     final boolean includeThresholds) throws IOException,
                                                                                                     GenericXMLReadingHandlerException
    {
        final String xmlString = getDefaultService().getTimeSeriesHeaders(clientId,
                                                                          queryId,
                                                                          null,
                                                                          startTime,
                                                                          t0,
                                                                          endTime,
                                                                          parameterIds,
                                                                          locationIds,
                                                                          ensembleId,
                                                                          ensembleMemberIndex,
                                                                          includeThresholds);
        if(xmlString != null)
        {
            final TimeSeriesHeaderInfoList results = new TimeSeriesHeaderInfoList();
            XMLTools.readXMLFromString(xmlString, results);
            return results;
        }
        else
        {
            return null;
        }
    }

    /**
     * @param listener The listener to add.
     */
    public static void addFewsPiServiceProviderListener(final FewsPiServiceProviderListener listener)
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        threadInfo.getListeners().add(listener);
    }

    /**
     * @param listener The listener to remove.
     */
    public static void removeListener(final FewsPiServiceProviderListener listener)
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        threadInfo.getListeners().remove(listener);
    }

    /**
     * @param Calls {@link FewsPiServiceProviderListener#reactToReinitializedPIServiceConnection()} for each listener in
     *            the list.
     */
    public static void fireReactToReinitializedConnection()
    {
        final FewsPiServiceAttributes threadInfo = getStorageAttributes(Thread.currentThread());
        for(int i = 0; i < threadInfo.getListeners().size(); i++)
        {
            threadInfo.getListeners().get(i).reactToReinitializedPIServiceConnection();
        }
    }

    /**
     * Remove any potential inactive Thread from the _threadToServiceInfoMap Map.
     * 
     * @param thread
     */
    private static void removeInactiveThreadsFromServiceInfoMap()
    {
        for(final Thread thread: _threadToServiceInfoMap.keySet())
        {
            if((thread != null) && !thread.isAlive())
            {
                _threadToServiceInfoMap.remove(thread);
            }
        }
    }

}
